"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertDissectProcessorToESQL = convertDissectProcessorToESQL;
var _esqlAst = require("@kbn/esql-ast");
var _dissect_patterns = require("../../../../types/utils/dissect_patterns");
var _condition_to_esql = require("../condition_to_esql");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a Streamlang DissectProcessor into a list of ES|QL AST commands.
 *
 * Conditional execution logic:
 *  - If neither `ignore_missing` nor `where` is provided: emit a single DISSECT command.
 *  - Otherwise, use CASE approach to conditionally execute DISSECT:
 *      * Create temporary field using CASE to conditionally set to source field or empty string (empty string avoids ES|QL NULL errors)
 *      * Apply DISSECT to temporary field
 *      * Drop temporary field
 *    Condition: (exists(from) if ignore_missing) AND (where condition, if provided)
 *
 * Type handling:
 *  - Pre-dissect: cast all prospective DISSECT output fields to avoid ES|QL's type conflict errors.
 *  - DISSECT yields keyword (string) values; further casts are user driven.
 *
 *  @example
 *     ```typescript
 *     const streamlangDSL: StreamlangDSL = {
 *        steps: [
 *          {
 *            action: 'dissect',
 *            from: 'message',
 *            pattern: '[%{log.level}] %{client.ip}',
 *            ignore_missing: true,
 *            where: {
 *              field: 'flags.process',
 *              exists: true,
 *            },
 *          } as DissectProcessor,
 *        ],
 *      };
 *    ```
 *
 *   Generates (conceptually):
 *    ```txt
 *      // | WHERE NOT(message IS NULL)  // Only if ignore_missing = false
 *      | EVAL `log.level` = TO_STRING(`log.level`)
 *      | EVAL `client.ip` = TO_STRING(`client.ip`)
 *      | EVAL __temp_dissect_where_message__ = CASE(NOT(message IS NULL) AND NOT(`flags.process` IS NULL), message, "")
 *      | DISSECT __temp_dissect_where_message__ "[%{log.level}] %{client.ip}"
 *      | DROP __temp_dissect_where_message__
 *    ```
 */
function convertDissectProcessorToESQL(processor) {
  const {
    from,
    pattern,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    append_separator,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    ignore_missing = false,
    // default same as ES Dissect Enrich Processor
    where
  } = processor;
  const fromColumn = _esqlAst.Builder.expression.column(from);
  const dissectCommand = buildDissectCommand(pattern, fromColumn, append_separator);
  const commands = [];

  // Add missing field filter if needed (ignore_missing = false)
  const missingFieldFilter = (0, _common.buildIgnoreMissingFilter)(from, ignore_missing);
  if (missingFieldFilter) {
    commands.push(missingFieldFilter);
  }

  // Check if conditional execution is needed for 'where' clauses, return simple command otherwise
  const needConditional = ignore_missing || Boolean(where);
  if (!needConditional) {
    commands.push(dissectCommand);
    return commands;
  }

  // Pre-cast all dissect output fields to string for consistency
  const {
    allFields
  } = (0, _dissect_patterns.parseMultiDissectPatterns)([pattern]);
  const fieldNames = allFields.map(f => f.name);
  commands.push(...(0, _common.castFieldsToString)(fieldNames));

  // Build condition for when DISSECT should execute
  const dissectCondition = (0, _common.buildWhereCondition)(from, ignore_missing, where, _condition_to_esql.conditionToESQLAst);

  // Create temporary field name for conditional processing
  // Using CASE, set temporary field to source field if condition passes, empty string otherwise
  const tempFieldName = `__temp_dissect_where_${from}__`;
  const tempColumn = _esqlAst.Builder.expression.column(tempFieldName);
  commands.push(_esqlAst.Builder.command({
    name: 'eval',
    args: [_esqlAst.Builder.expression.func.binary('=', [tempColumn, _esqlAst.Builder.expression.func.call('CASE', [dissectCondition, fromColumn, _esqlAst.Builder.expression.literal.string('') // Empty string avoids ES|QL NULL errors that would break the pipeline
    ])])]
  }));

  // Apply DISSECT to the temporary field
  commands.push(buildDissectCommand(pattern, tempColumn, append_separator));

  // Clean up temporary field
  commands.push(_esqlAst.Builder.command({
    name: 'drop',
    args: [tempColumn]
  }));
  return commands;
}

/** Build the base DISSECT command (no conditional logic) */
function buildDissectCommand(pattern, fromColumn, appendSep) {
  const args = [fromColumn, _esqlAst.Builder.expression.literal.string(pattern)];
  const cmd = _esqlAst.Builder.command({
    name: 'dissect',
    args
  });
  if (appendSep) {
    cmd.args.push(_esqlAst.Builder.option({
      name: 'append_separator',
      args: [_esqlAst.Builder.expression.literal.string(appendSep)]
    }));
  }
  return cmd;
}