"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.conditionToESQLAst = conditionToESQLAst;
exports.esqlLiteralFromAny = esqlLiteralFromAny;
var _esqlAst = require("@kbn/esql-ast");
var _conditions = require("../../../types/conditions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function esqlLiteralFromAny(value) {
  if (Array.isArray(value)) {
    // Let the Builder handle nested structures properly
    return _esqlAst.Builder.expression.list.literal({
      values: value.map(item => esqlLiteralFromAny(item))
    });
  }
  if (typeof value === 'string') {
    return _esqlAst.Builder.expression.literal.string(value);
  }
  if (typeof value === 'number') {
    return Number.isInteger(value) ? _esqlAst.Builder.expression.literal.integer(value) : _esqlAst.Builder.expression.literal.decimal(value);
  }
  if (typeof value === 'boolean') {
    return _esqlAst.Builder.expression.literal.boolean(value);
  }
  if (value === null || value === undefined) {
    return _esqlAst.Builder.expression.literal.nil();
  }
  // Fallback to string representation for complex objects
  return _esqlAst.Builder.expression.literal.string(JSON.stringify(value));
}
function conditionToESQLAst(condition) {
  if ((0, _conditions.isFilterCondition)(condition)) {
    const field = _esqlAst.Builder.expression.column(condition.field);
    if ('eq' in condition) {
      return _esqlAst.Builder.expression.func.binary('==', [field, esqlLiteralFromAny(condition.eq)]);
    }
    if ('neq' in condition) {
      return _esqlAst.Builder.expression.func.binary('!=', [field, esqlLiteralFromAny(condition.neq)]);
    }
    if ('gt' in condition) {
      return _esqlAst.Builder.expression.func.binary('>', [field, esqlLiteralFromAny(condition.gt)]);
    }
    if ('gte' in condition) {
      return _esqlAst.Builder.expression.func.binary('>=', [field, esqlLiteralFromAny(condition.gte)]);
    }
    if ('lt' in condition) {
      return _esqlAst.Builder.expression.func.binary('<', [field, esqlLiteralFromAny(condition.lt)]);
    }
    if ('lte' in condition) {
      return _esqlAst.Builder.expression.func.binary('<=', [field, esqlLiteralFromAny(condition.lte)]);
    }
    if ('exists' in condition) {
      if (condition.exists === true) {
        return _esqlAst.Builder.expression.func.call('NOT', [_esqlAst.Builder.expression.func.postfix('IS NULL', field)]);
      } else {
        return _esqlAst.Builder.expression.func.postfix('IS NULL', field);
      }
    }
    if ('range' in condition && condition.range) {
      const parts = [];
      if (condition.range.gt !== undefined) parts.push(_esqlAst.Builder.expression.func.binary('>', [field, esqlLiteralFromAny(condition.range.gt)]));
      if (condition.range.gte !== undefined) parts.push(_esqlAst.Builder.expression.func.binary('>=', [field, esqlLiteralFromAny(condition.range.gte)]));
      if (condition.range.lt !== undefined) parts.push(_esqlAst.Builder.expression.func.binary('<', [field, esqlLiteralFromAny(condition.range.lt)]));
      if (condition.range.lte !== undefined) parts.push(_esqlAst.Builder.expression.func.binary('<=', [field, esqlLiteralFromAny(condition.range.lte)]));
      if (parts.length === 1) return parts[0];
      return parts.reduce((acc, part) => _esqlAst.Builder.expression.func.binary('and', [acc, part]));
    }
    if ('contains' in condition) {
      return _esqlAst.Builder.expression.func.call('LIKE', [field, _esqlAst.Builder.expression.literal.string(`%${condition.contains}%`)]);
    }
    if ('startsWith' in condition) {
      return _esqlAst.Builder.expression.func.call('LIKE', [field, _esqlAst.Builder.expression.literal.string(`${condition.startsWith}%`)]);
    }
    if ('endsWith' in condition) {
      return _esqlAst.Builder.expression.func.call('LIKE', [field, _esqlAst.Builder.expression.literal.string(`%${condition.endsWith}`)]);
    }
  } else if ((0, _conditions.isAndCondition)(condition)) {
    const andConditions = condition.and.map(c => conditionToESQLAst(c));
    return andConditions.reduce((acc, cond) => _esqlAst.Builder.expression.func.binary('and', [acc, cond]));
  } else if ((0, _conditions.isOrCondition)(condition)) {
    const orConditions = condition.or.map(c => conditionToESQLAst(c));
    return orConditions.reduce((acc, cond) => _esqlAst.Builder.expression.func.binary('or', [acc, cond]));
  } else if ((0, _conditions.isNotCondition)(condition)) {
    const notCondition = conditionToESQLAst(condition.not);
    return _esqlAst.Builder.expression.func.unary('NOT', notCondition);
  } else if ((0, _conditions.isAlwaysCondition)(condition)) {
    return _esqlAst.Builder.expression.literal.boolean(true);
  }
  return _esqlAst.Builder.expression.literal.boolean(false);
}