"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateConnectorTypeConfig = validateConnectorTypeConfig;
var _i18n = require("@kbn/i18n");
var _constants = require("../../../common/auth/constants");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateUrl(configuredUrl) {
  try {
    new URL(configuredUrl);
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationErrorNoHostname', {
      defaultMessage: 'error validation webhook action config: unable to parse url: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
}
function ensureUriAllowed(configuredUrl, configurationUtilities) {
  try {
    configurationUtilities.ensureUriAllowed(configuredUrl);
  } catch (allowListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationError', {
      defaultMessage: 'error validation webhook action config: {message}',
      values: {
        message: allowListError.message
      }
    }));
  }
}
function validateAuthType(configObject) {
  if (Boolean(configObject.authType) && !configObject.hasAuth) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.authConfigurationError', {
      defaultMessage: 'error validation webhook action config: authType must be null or undefined if hasAuth is false'
    }));
  }
}
function validateCertType(configObject, configurationUtilities) {
  if (configObject.certType === _constants.SSLCertType.PFX) {
    const webhookSettings = configurationUtilities.getWebhookSettings();
    if (!webhookSettings.ssl.pfx.enabled) {
      throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.pfxConfigurationError', {
        defaultMessage: 'error validation webhook action config: certType "{certType}" is disabled',
        values: {
          certType: _constants.SSLCertType.PFX
        }
      }));
    }
  }
}
function validateAdditionalFields(configObject) {
  if (configObject.additionalFields) {
    try {
      const parsedAdditionalFields = JSON.parse(configObject.additionalFields);
      if (typeof parsedAdditionalFields !== 'object' || Array.isArray(parsedAdditionalFields) || Object.keys(parsedAdditionalFields).length === 0) {
        throw new Error(_translations.ADDITIONAL_FIELD_CONFIG_ERROR);
      }
    } catch (e) {
      throw new Error(_translations.ADDITIONAL_FIELD_CONFIG_ERROR);
    }
  }
}
function validateOAuth2(configObject) {
  if (configObject.authType === _constants.AuthType.OAuth2ClientCredentials && (!configObject.accessTokenUrl || !configObject.clientId)) {
    const missingFields = [];
    if (!configObject.accessTokenUrl) {
      missingFields.push('Access Token URL (accessTokenUrl)');
    }
    if (!configObject.clientId) {
      missingFields.push('Client ID (clientId)');
    }
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.oauth2ConfigurationError', {
      defaultMessage: `error validation webhook action config: missing {missingItems} fields`,
      values: {
        missingItems: missingFields.join(', ')
      }
    }));
  }
}
function validateConnectorTypeConfig(configObject, validatorServices) {
  const {
    configurationUtilities
  } = validatorServices;
  const configuredUrl = configObject.url;
  validateUrl(configuredUrl);
  ensureUriAllowed(configuredUrl, configurationUtilities);
  validateAuthType(configObject);
  validateCertType(configObject, configurationUtilities);
  validateAdditionalFields(configObject);
  validateOAuth2(configObject);
}