"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToEsqlTable = exports.toGroupedEsqlQueryHits = exports.toEsqlQueryHits = exports.rowToDocument = exports.getEsqlQueryHits = exports.getAlertIdFields = exports.ALERT_ID_SUGGESTED_MAX = exports.ALERT_ID_COLUMN = void 0;
var _lodash = require("lodash");
var _esqlAst = require("@kbn/esql-ast");
var _esqlUtils = require("@kbn/esql-utils");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ESQL_DOCUMENT_ID = 'esql_query_document';
const CHUNK_SIZE = 100;
const ALERT_ID_COLUMN = exports.ALERT_ID_COLUMN = 'Alert ID';
const ALERT_ID_SUGGESTED_MAX = exports.ALERT_ID_SUGGESTED_MAX = 10;
const rowToDocument = (columns, row) => {
  const doc = {};
  for (let i = 0; i < columns.length; ++i) {
    doc[columns[i].name] = row[i];
  }
  return doc;
};
exports.rowToDocument = rowToDocument;
const getEsqlQueryHits = async (table, query, isGroupAgg, isPreview = false) => {
  if (isGroupAgg) {
    const alertIdFields = getAlertIdFields(query, table.columns);
    return await toGroupedEsqlQueryHits(table, alertIdFields, isPreview);
  }
  return await toEsqlQueryHits(table, isPreview);
};
exports.getEsqlQueryHits = getEsqlQueryHits;
const toEsqlQueryHits = async (table, isPreview = false, chunkSize = CHUNK_SIZE) => {
  const hits = [];
  const rows = [];
  for (let r = 0; r < table.values.length; r++) {
    const row = table.values[r];
    const document = rowToDocument(table.columns, row);
    hits.push({
      _id: ESQL_DOCUMENT_ID,
      _index: '',
      _source: document
    });
    if (isPreview) {
      rows.push(rows.length > 0 ? document : Object.assign({
        [ALERT_ID_COLUMN]: _constants.ActionGroupId
      }, document));
    }
    if (r !== 0 && r % chunkSize === 0) {
      await new Promise(setImmediate);
    }
  }
  return {
    results: {
      isCountAgg: true,
      isGroupAgg: false,
      esResult: {
        took: 0,
        timed_out: false,
        _shards: {
          failed: 0,
          successful: 0,
          total: 0
        },
        hits: {
          hits,
          total: hits.length
        }
      }
    },
    rows,
    cols: isPreview ? getColumnsForPreview(table.columns) : []
  };
};
exports.toEsqlQueryHits = toEsqlQueryHits;
const toGroupedEsqlQueryHits = async (table, alertIdFields, isPreview = false, chunkSize = CHUNK_SIZE) => {
  const duplicateAlertIds = new Set();
  const longAlertIds = new Set();
  const rows = [];
  const mappedAlertIds = {};
  const groupedHits = {};
  for (let r = 0; r < table.values.length; r++) {
    const row = table.values[r];
    const document = rowToDocument(table.columns, row);
    const mappedAlertId = alertIdFields.filter(a => !(0, _lodash.isNil)(document[a])).map(a => document[a]);
    if (mappedAlertId.length > 0) {
      const alertId = mappedAlertId.join(',');
      const hit = {
        _id: ESQL_DOCUMENT_ID,
        _index: '',
        _source: document
      };
      if (groupedHits[alertId]) {
        duplicateAlertIds.add(alertId);
        groupedHits[alertId].push(hit);
      } else {
        groupedHits[alertId] = [hit];
        mappedAlertIds[alertId] = mappedAlertId;
      }
      if (isPreview) {
        rows.push(Object.assign({
          [ALERT_ID_COLUMN]: alertId
        }, document));
        if (mappedAlertId.length >= ALERT_ID_SUGGESTED_MAX) {
          longAlertIds.add(alertId);
        }
      }
      if (r !== 0 && r % chunkSize === 0) {
        await new Promise(setImmediate);
      }
    }
  }
  const aggregations = {
    groupAgg: {
      buckets: (0, _lodash.entries)(groupedHits).map(([key, value]) => {
        return {
          key: mappedAlertIds[key],
          doc_count: value.length,
          topHitsAgg: {
            hits: {
              hits: value
            }
          }
        };
      })
    }
  };
  return {
    results: {
      isCountAgg: false,
      isGroupAgg: true,
      esResult: {
        took: 0,
        timed_out: false,
        _shards: {
          failed: 0,
          successful: 0,
          total: 0
        },
        hits: {
          hits: []
        },
        aggregations
      },
      termField: alertIdFields
    },
    duplicateAlertIds,
    longAlertIds,
    rows,
    cols: isPreview ? getColumnsForPreview(table.columns) : []
  };
};
exports.toGroupedEsqlQueryHits = toGroupedEsqlQueryHits;
const transformToEsqlTable = datatable => {
  const columns = datatable.columns;
  // Convert each value to string or null to match EsqlResultRow type
  const values = datatable.values.map(row => row.map(v => v === null || typeof v === 'string' ? v : String(v)));
  return {
    columns,
    values
  };
};
exports.transformToEsqlTable = transformToEsqlTable;
const getAlertIdFields = (query, resultColumns) => {
  const {
    root
  } = _esqlAst.Parser.parse(query);
  const commands = root.commands;
  const columns = resultColumns.map(c => c.name);

  // Check for STATS command
  const statsCommandIndex = getLastStatsCommandIndex(commands);
  if (statsCommandIndex > -1) {
    const statsCommand = commands[statsCommandIndex];
    // Check for BY option and get fields
    const byOption = getByOption(statsCommand);
    if (byOption) {
      let fields = getFields(byOption);

      // Check if any STATS fields were renamed
      const renameCommands = getRenameCommands(commands.slice(statsCommandIndex));
      if (renameCommands.length > 0) {
        fields = getFieldsFromRenameCommands(renameCommands, fields);
      }

      // Check if any fields were dropped
      fields = intersection(fields, columns);
      if (fields.length > 0) {
        return fields;
      }
    }
  }

  // Check for METADATA _id
  const metadataOption = getMetadataOption(commands);
  if (metadataOption) {
    const fields = getIdField(metadataOption);

    // Check if _id was dropped
    if (intersection(fields, columns).length > 0) {
      return fields;
    }
  }

  // Return all columns
  return columns;
};
exports.getAlertIdFields = getAlertIdFields;
const getLastStatsCommandIndex = commands => (0, _lodash.findLastIndex)(commands, c => c.name === 'stats');
const getByOption = astCommand => {
  for (const statsArg of astCommand.args) {
    if ((0, _esqlAst.isOptionNode)(statsArg) && statsArg.name === 'by') {
      return statsArg;
    }
  }
};
const getFields = option => {
  const fields = [];
  for (const arg of option.args) {
    if ((0, _esqlAst.isColumn)(arg)) {
      fields.push(arg.name);
    }
  }
  return fields;
};
const getRenameCommands = commands => commands.filter(c => c.name === 'rename');
const getFieldsFromRenameCommands = (astCommands, fields) => {
  for (const command of astCommands) {
    for (const renameArg of command.args) {
      if ((0, _esqlAst.isFunctionExpression)(renameArg)) {
        const {
          original,
          renamed
        } = (0, _esqlUtils.getArgsFromRenameFunction)(renameArg);
        if ((0, _esqlAst.isColumn)(original) && (0, _esqlAst.isColumn)(renamed)) {
          fields = fields.map(field => field === original.name ? renamed.name : field);
        }
      }
    }
  }
  return fields;
};
const getMetadataOption = commands => {
  const fromCommand = commands.find(c => c.name === 'from');
  if (fromCommand) {
    for (const fromArg of fromCommand.args) {
      if ((0, _esqlAst.isOptionNode)(fromArg) && fromArg.name === 'metadata') {
        return fromArg;
      }
    }
  }
  return undefined;
};
const getIdField = option => {
  const fields = [];
  for (const arg of option.args) {
    if ((0, _esqlAst.isColumn)(arg) && arg.name === '_id') {
      fields.push(arg.name);
      return fields;
    }
  }
  return fields;
};
const getColumnsForPreview = columns => {
  const cols = [{
    id: ALERT_ID_COLUMN,
    actions: false
  }];
  for (const c of columns) {
    cols.push({
      id: c.name,
      actions: false
    });
  }
  return cols;
};
const intersection = (fields, columns) => {
  const columnSet = new Set(columns);
  return fields.filter(item => columnSet.has(item));
};