"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HostIsolationExceptionsValidator = void 0;
var _configSchema = require("@kbn/config-schema");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _base_validator = require("./base_validator");
var _errors = require("./errors");
var _is_valid_ip = require("../../../../common/endpoint/utils/is_valid_ip");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateIp(value) {
  if (!(0, _is_valid_ip.isValidIPv4OrCIDR)(value)) {
    return `invalid ip: ${value}`;
  }
}
const EntrySchema = _configSchema.schema.object({
  field: _configSchema.schema.literal('destination.ip'),
  operator: _configSchema.schema.literal('included'),
  type: _configSchema.schema.literal('match'),
  value: _configSchema.schema.string({
    validate: validateIp
  })
});
const HostIsolationDataSchema = _configSchema.schema.object({
  entries: _configSchema.schema.arrayOf(EntrySchema, {
    minSize: 1,
    maxSize: 1
  })
}, {
  unknowns: 'ignore'
});

// use the baseSchema and overwrite the os_type
// to accept all OSs in the list for host isolation exception
const HostIsolationBasicDataSchema = _base_validator.BasicEndpointExceptionDataSchema.extends({
  osTypes: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.WINDOWS), _configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.LINUX), _configSchema.schema.literal(_securitysolutionUtils.OperatingSystem.MAC)]), {
    minSize: 3,
    maxSize: 3
  })
});
class HostIsolationExceptionsValidator extends _base_validator.BaseValidator {
  static isHostIsolationException(item) {
    return item.listId === _securitysolutionListConstants.ENDPOINT_HOST_ISOLATION_EXCEPTIONS_LIST_ID;
  }
  async validateHasWritePrivilege() {
    return this.validateHasPrivilege('canWriteHostIsolationExceptions');
  }
  async validateHasDeletePrivilege() {
    return this.validateHasPrivilege('canDeleteHostIsolationExceptions');
  }
  async validateHasReadPrivilege() {
    return this.validateHasPrivilege('canReadHostIsolationExceptions');
  }
  async validatePreCreateItem(item) {
    await this.validateHasWritePrivilege();
    await this.validateHostIsolationData(item);
    await this.validateByPolicyItem(item);
    await this.validateCanCreateGlobalArtifacts(item);
    await this.validateCreateOwnerSpaceIds(item);
    return item;
  }
  async validatePreUpdateItem(_updatedItem, currentItem) {
    const updatedItem = _updatedItem;
    await this.validateHasWritePrivilege();
    await this.validateHostIsolationData(updatedItem);
    await this.validateByPolicyItem(updatedItem, currentItem);
    await this.validateUpdateOwnerSpaceIds(_updatedItem, currentItem);
    await this.validateCanUpdateItemInActiveSpace(_updatedItem, currentItem);
    return _updatedItem;
  }
  async validatePreGetOneItem(currentItem) {
    await this.validateHasReadPrivilege();
    await this.validateCanReadItemInActiveSpace(currentItem);
  }
  async validatePreSummary() {
    await this.validateHasReadPrivilege();
  }
  async validatePreDeleteItem(currentItem) {
    await this.validateHasDeletePrivilege();
    await this.validateCanDeleteItemInActiveSpace(currentItem);
  }
  async validatePreExport() {
    await this.validateHasReadPrivilege();
  }
  async validatePreSingleListFind() {
    await this.validateHasReadPrivilege();
  }
  async validatePreMultiListFind() {
    await this.validateHasReadPrivilege();
  }
  async validatePreImport() {
    throw new _errors.EndpointArtifactExceptionValidationError('Import is not supported for Endpoint artifact exceptions');
  }
  async validateHostIsolationData(item) {
    try {
      HostIsolationBasicDataSchema.validate(item);
      HostIsolationDataSchema.validate(item);
    } catch (error) {
      throw new _errors.EndpointArtifactExceptionValidationError(error.message);
    }
  }
}
exports.HostIsolationExceptionsValidator = HostIsolationExceptionsValidator;