"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskEngineInitRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../../../common/constants");
var _translations = require("./translations");
var _risk_engine_privileges = require("../risk_engine_privileges");
var _audit = require("../audit");
var _audit2 = require("../../audit");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const riskEngineInitRoute = (router, getStartServices) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.RISK_ENGINE_INIT_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: '1',
    validate: {}
  }, (0, _risk_engine_privileges.withRiskEnginePrivilegeCheck)(getStartServices, async (context, _request, response) => {
    var _securitySolution$get;
    const securitySolution = await context.securitySolution;
    (_securitySolution$get = securitySolution.getAuditLogger()) === null || _securitySolution$get === void 0 ? void 0 : _securitySolution$get.log({
      message: 'User attempted to initialize the risk engine',
      event: {
        action: _audit.RiskEngineAuditActions.RISK_ENGINE_INIT,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.CHANGE,
        outcome: _audit2.AUDIT_OUTCOME.UNKNOWN
      }
    });
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const [_, {
      taskManager
    }] = await getStartServices();
    const riskEngineDataClient = securitySolution.getRiskEngineDataClient();
    const riskScoreDataClient = securitySolution.getRiskScoreDataClient();
    const spaceId = securitySolution.getSpaceId();
    try {
      if (!taskManager) {
        return siemResponse.error({
          statusCode: 400,
          body: _translations.TASK_MANAGER_UNAVAILABLE_ERROR
        });
      }
      const initResult = await riskEngineDataClient.init({
        taskManager,
        namespace: spaceId,
        riskScoreDataClient
      });
      const result = {
        risk_engine_enabled: initResult.riskEngineEnabled,
        risk_engine_resources_installed: initResult.riskEngineResourcesInstalled,
        risk_engine_configuration_created: initResult.riskEngineConfigurationCreated,
        errors: initResult.errors
      };
      if (!initResult.riskEngineEnabled || !initResult.riskEngineResourcesInstalled || !initResult.riskEngineConfigurationCreated) {
        return siemResponse.error({
          statusCode: 400,
          body: {
            message: result.errors.join('\n'),
            full_error: result
          },
          bypassErrorFormat: true
        });
      }
      return response.ok({
        body: {
          result
        }
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: {
          message: error.message,
          full_error: JSON.stringify(e)
        },
        bypassErrorFormat: true
      });
    }
  }));
};
exports.riskEngineInitRoute = riskEngineInitRoute;