"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDataSourcesService = void 0;
var _constants = require("../constants");
var _mappings = require("../engine/elasticsearch/mappings");
var _index_sync = require("./sync/index_sync");
var _integrations_sync = require("./sync/integrations/integrations_sync");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createDataSourcesService = (dataClient, soClient, maxUsersAllowed) => {
  const esClient = dataClient.deps.clusterClient.asCurrentUser;
  const indexSyncService = (0, _index_sync.createIndexSyncService)(dataClient, maxUsersAllowed);
  const integrationsSyncService = (0, _integrations_sync.createIntegrationsSyncService)(dataClient, soClient);

  /**
   * This creates an index for the user to populate privileged users.
   * It already defines the mappings and settings for the index.
   */
  const createImportIndex = (indexName, mode) => {
    dataClient.log('info', `Creating import index: ${indexName} with mode: ${mode}`);
    // Use the current user client to create the index, the internal user does not have permissions to any index
    return esClient.indices.create({
      index: indexName,
      mappings: {
        properties: _mappings.PRIVILEGED_MONITOR_IMPORT_USERS_INDEX_MAPPING
      },
      settings: {
        mode
      }
    });
  };
  const searchPrivilegesIndices = async query => {
    var _fields$userName$key, _fields$userName, _fields$userName$keyw;
    const {
      indices,
      fields
    } = await esClient.fieldCaps({
      index: [query ? `*${query}*` : '*', ..._constants.PRE_EXCLUDE_INDICES],
      types: ['keyword'],
      fields: ['user.name'],
      include_unmapped: true,
      ignore_unavailable: true,
      allow_no_indices: true,
      expand_wildcards: 'open',
      include_empty_fields: true,
      filters: '-parent'
    });
    const indicesWithUserName = (_fields$userName$key = (_fields$userName = fields['user.name']) === null || _fields$userName === void 0 ? void 0 : (_fields$userName$keyw = _fields$userName.keyword) === null || _fields$userName$keyw === void 0 ? void 0 : _fields$userName$keyw.indices) !== null && _fields$userName$key !== void 0 ? _fields$userName$key : indices;
    if (!Array.isArray(indicesWithUserName) || indicesWithUserName.length === 0) {
      return [];
    }
    return indicesWithUserName.filter(name => !_constants.POST_EXCLUDE_INDICES.some(pattern => name.startsWith(pattern)));
  };
  const syncAllSources = async () => {
    const jobs = [indexSyncService.plainIndexSync(soClient)];
    jobs.push(integrationsSyncService.integrationsSync());
    const settled = await Promise.allSettled(jobs);
    settled.filter(r => r.status === 'rejected').forEach(r => dataClient.log('warn', `Data sources sync failed: ${String(r.reason)}`));
  };
  return {
    createImportIndex,
    searchPrivilegesIndices,
    syncAllSources,
    ...(0, _index_sync.createIndexSyncService)(dataClient, maxUsersAllowed),
    ...(0, _integrations_sync.createIntegrationsSyncService)(dataClient, soClient)
  };
};
exports.createDataSourcesService = createDataSourcesService;