"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetCriticalityPublicBulkUploadRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _nodeStream = require("node:stream");
var _zodHelpers = require("@kbn/zod-helpers");
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _constants = require("../../../../../common/constants");
var _check_and_init_asset_criticality_resources = require("../check_and_init_asset_criticality_resources");
var _audit = require("../audit");
var _audit2 = require("../../audit");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const assetCriticalityPublicBulkUploadRoute = (router, logger, config) => {
  router.versioned.post({
    access: 'public',
    path: _constants.ASSET_CRITICALITY_PUBLIC_BULK_UPLOAD_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_entity_analytics.BulkUpsertAssetCriticalityRecordsRequestBody)
      }
    }
  }, async (context, request, response) => {
    var _securitySolution$get;
    const {
      errorRetries,
      maxBulkRequestBodySizeBytes
    } = config.entityAnalytics.assetCriticality.csvUpload;
    const {
      records
    } = request.body;
    const securitySolution = await context.securitySolution;
    (_securitySolution$get = securitySolution.getAuditLogger()) === null || _securitySolution$get === void 0 ? void 0 : _securitySolution$get.log({
      message: 'User attempted to assign many asset criticalities via bulk upload',
      event: {
        action: _audit.AssetCriticalityAuditActions.ASSET_CRITICALITY_BULK_UPDATE,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.CREATION,
        outcome: _audit2.AUDIT_OUTCOME.UNKNOWN
      }
    });
    const start = new Date();
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      await (0, _check_and_init_asset_criticality_resources.checkAndInitAssetCriticalityResources)(context, logger);
      const assetCriticalityClient = securitySolution.getAssetCriticalityDataClient();
      const formattedRecords = records.map(record => ({
        idField: record.id_field,
        idValue: record.id_value,
        criticalityLevel: record.criticality_level
      }));
      const recordsStream = _nodeStream.Readable.from(formattedRecords, {
        objectMode: true
      });
      const {
        errors,
        stats
      } = await assetCriticalityClient.bulkUpsertFromStream({
        recordsStream,
        retries: errorRetries,
        flushBytes: maxBulkRequestBodySizeBytes
      });
      const end = new Date();
      const tookMs = end.getTime() - start.getTime();
      logger.debug(() => `Asset criticality Bulk upload completed in ${tookMs}ms ${JSON.stringify(stats)}`);
      return response.ok({
        body: {
          errors,
          stats
        }
      });
    } catch (e) {
      logger.error(`Error during asset criticality bulk upload: ${e}`);
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.assetCriticalityPublicBulkUploadRoute = assetCriticalityPublicBulkUploadRoute;