"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkCreateWithSuppression = void 0;
var _perf_hooks = require("perf_hooks");
var _lodash = require("lodash");
var _utils = require("../../../../../common/detection_engine/utils");
var _utils2 = require("./utils");
var _get_number_of_suppressed_alerts = require("./get_number_of_suppressed_alerts");
var _enrichments = require("./enrichments");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkCreateWithSuppression = async ({
  sharedParams,
  wrappedDocs,
  services,
  suppressionWindow,
  isSuppressionPerRuleExecution,
  maxAlerts,
  ruleType
}) => {
  const {
    ruleExecutionLogger,
    alertTimestampOverride
  } = sharedParams;
  if (wrappedDocs.length === 0) {
    return {
      errors: [],
      success: true,
      enrichmentDuration: '0',
      bulkCreateDuration: '0',
      createdItemsCount: 0,
      suppressedItemsCount: 0,
      createdItems: [],
      alertsWereTruncated: false
    };
  }
  const start = _perf_hooks.performance.now();
  let enrichmentsTimeStart = 0;
  let enrichmentsTimeFinish = 0;
  const enrichAlertsWrapper = async (alerts, params) => {
    enrichmentsTimeStart = _perf_hooks.performance.now();
    try {
      const enrichedAlerts = await (0, _enrichments.enrichEvents)({
        services,
        logger: sharedParams.ruleExecutionLogger,
        events: alerts,
        spaceId: params.spaceId
      });
      return enrichedAlerts;
    } catch (error) {
      ruleExecutionLogger.error(`Alerts enrichment failed: ${error}`);
      throw error;
    } finally {
      enrichmentsTimeFinish = _perf_hooks.performance.now();
    }
  };
  const alerts = wrappedDocs.map(doc => {
    var _doc$subAlerts;
    return {
      _id: doc._id,
      // `fields` should have already been merged into `doc._source`
      _source: doc._source,
      subAlerts: (doc === null || doc === void 0 ? void 0 : doc.subAlerts) != null ? doc === null || doc === void 0 ? void 0 : (_doc$subAlerts = doc.subAlerts) === null || _doc$subAlerts === void 0 ? void 0 : _doc$subAlerts.map(subAlert => ({
        _id: subAlert._id,
        _source: subAlert._source
      })) : undefined
    };
  });
  const {
    createdAlerts,
    errors,
    suppressedAlerts,
    alertsWereTruncated
  } = await services.alertWithSuppression(alerts, suppressionWindow, enrichAlertsWrapper, alertTimestampOverride, isSuppressionPerRuleExecution, maxAlerts);
  const end = _perf_hooks.performance.now();
  ruleExecutionLogger.debug(`Alerts bulk process took ${(0, _utils2.makeFloatString)(end - start)} ms`);

  // query rule type suppression does not happen in memory, so we can't just count createdAlerts and suppressedAlerts
  // for this rule type we need to look into alerts suppression properties, extract those values and sum up
  const suppressedItemsCount = (0, _utils.isQueryRule)(ruleType) ? (0, _get_number_of_suppressed_alerts.getNumberOfSuppressedAlerts)(createdAlerts, suppressedAlerts.map(({
    _source,
    _id
  }) => ({
    _id,
    ..._source
  }))) : suppressedAlerts.length;
  if (!(0, _lodash.isEmpty)(errors)) {
    ruleExecutionLogger.warn(`Alerts bulk process finished with errors: ${JSON.stringify(errors)}`);
    return {
      errors: Object.keys(errors),
      success: false,
      enrichmentDuration: (0, _utils2.makeFloatString)(enrichmentsTimeFinish - enrichmentsTimeStart),
      bulkCreateDuration: (0, _utils2.makeFloatString)(end - start),
      createdItemsCount: createdAlerts.length,
      createdItems: createdAlerts,
      suppressedItemsCount,
      alertsWereTruncated
    };
  } else {
    return {
      errors: [],
      success: true,
      bulkCreateDuration: (0, _utils2.makeFloatString)(end - start),
      enrichmentDuration: (0, _utils2.makeFloatString)(enrichmentsTimeFinish - enrichmentsTimeStart),
      createdItemsCount: createdAlerts.length,
      createdItems: createdAlerts,
      suppressedItemsCount,
      alertsWereTruncated
    };
  }
};
exports.bulkCreateWithSuppression = bulkCreateWithSuppression;