"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installPromotionRules = installPromotionRules;
var _common = require("@kbn/fleet-plugin/common");
var _create_prebuilt_rules = require("../rule_objects/create_prebuilt_rules");
var _upgrade_prebuilt_rules = require("../rule_objects/upgrade_prebuilt_rules");
var _error_helpers = require("../../../../../utils/error_helpers");
var _constants = require("../../../../../../common/constants");
var _get_fleet_package_installation = require("./get_fleet_package_installation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Install or upgrade promotion rules. These rules are needed for the EASE
 * tier integrations to work (promote alerts from external SIEMs).
 *
 * Note: due to current limitations, the promotion rules cannot be placed in
 * their corresponding packages and are shipped as part of the
 * security_detection_engine package. However, the promotion rules should only
 * be installed if the corresponding integration package is installed. The logic
 * is following: we go through the list of known EASE integrations and check
 * if their package are installed. Then for installed integrations, we need to
 * find corresponding promotion rules, they are tagged as Promotion, and install
 * or upgrade them to latest versions.
 *
 */
async function installPromotionRules({
  rulesClient,
  detectionRulesClient,
  ruleAssetsClient,
  ruleObjectsClient,
  fleetServices,
  logger
}) {
  logger.debug('installPromotionRules: Promotion rules - installing');
  // Get the list of installed integrations
  const installedIntegrations = new Set((await Promise.all(_common.SEARCH_AI_LAKE_PACKAGES.map(async integration => {
    // We don't care about installation status of the integration as all
    // EASE integrations are agentless (don't require setting up an
    // integration policy). So the fact that the corresponding package is
    // installed is enough.
    const installation = await (0, _get_fleet_package_installation.getFleetPackageInstallation)(fleetServices, integration, logger);
    return installation ? integration : [];
  }))).flat());
  const latestRuleAssets = await ruleAssetsClient.fetchLatestAssets();
  const latestPromotionRules = latestRuleAssets.filter(rule => {
    var _rule$related_integra;
    // Rule should be tagged as 'Promotion' and should be related to an enabled integration
    return isPromotionRule(rule) && ((_rule$related_integra = rule.related_integrations) === null || _rule$related_integra === void 0 ? void 0 : _rule$related_integra.some(integration => installedIntegrations.has(integration.package)));
  });
  const installedRuleVersions = await ruleObjectsClient.fetchInstalledRuleVersions();
  const installedRuleVersionsMap = new Map(installedRuleVersions.map(version => [version.rule_id, version]));
  const promotionRulesToInstall = latestPromotionRules.filter(({
    rule_id: ruleId
  }) => {
    return !installedRuleVersionsMap.has(ruleId);
  });
  const {
    results: installationResults,
    errors: installationErrors
  } = await (0, _create_prebuilt_rules.createPrebuiltRules)(detectionRulesClient, promotionRulesToInstall.map(asset => ({
    ...asset,
    enabled: true
  })), logger);
  const promotionRulesToUpgrade = latestPromotionRules.filter(({
    rule_id: ruleId,
    version
  }) => {
    const installedVersion = installedRuleVersionsMap.get(ruleId);
    return installedVersion && installedVersion.version < version;
  });
  const {
    results: upgradeResults,
    errors: upgradeErrors
  } = await (0, _upgrade_prebuilt_rules.upgradePrebuiltRules)(detectionRulesClient, promotionRulesToUpgrade, logger);

  // Cleanup any unknown rules, we don't allow users to install any detection
  // rules that are not part of the enabled integrations, although that is not
  // enforced via the API
  const rulesToDelete = installedRuleVersions.filter(({
    rule_id: ruleId
  }) => {
    return !latestPromotionRules.some(rule => rule.rule_id === ruleId);
  });
  const deletionErrors = [];
  if (rulesToDelete.length > 0) {
    const bulkDeleteResult = await rulesClient.bulkDeleteRules({
      ids: rulesToDelete.map(({
        id
      }) => id)
    });
    deletionErrors.push(...bulkDeleteResult.errors);
  }
  const alreadyUpToDate = latestPromotionRules.filter(({
    rule_id: ruleId,
    version
  }) => {
    const installedVersion = installedRuleVersionsMap.get(ruleId);
    return installedVersion && installedVersion.version === version;
  });
  const allErrors = [...installationErrors, ...upgradeErrors, ...deletionErrors].reduce((errorsMap, currentError) => {
    const errorMessage = 'error' in currentError ? (0, _error_helpers.getErrorMessage)(currentError.error) : currentError.message;
    const ruleId = 'item' in currentError ? currentError.item.rule_id : currentError.rule.id;
    const existingError = errorsMap.get(errorMessage);
    if (existingError) {
      existingError.rules.push({
        rule_id: ruleId
      });
    } else {
      errorsMap.set(errorMessage, {
        rules: [{
          rule_id: ruleId
        }],
        message: errorMessage
      });
    }
    return errorsMap;
  }, new Map());
  const installationResult = {
    total: latestPromotionRules.length,
    installed: installationResults.length,
    updated: upgradeResults.length,
    deleted: rulesToDelete.length,
    skipped: alreadyUpToDate.length,
    errors: allErrors.size > 0 ? Array.from(allErrors.values()) : []
  };
  logger.debug('installPromotionRules: Promotion rules - installation complete:', installationResult);
  return installationResult;
}
function isPromotionRule(rule) {
  var _rule$tags;
  return ((_rule$tags = rule.tags) !== null && _rule$tags !== void 0 ? _rule$tags : []).some(tag => _constants.PROMOTION_RULE_TAGS.includes(tag));
}