"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetInsightsRoute = exports.getInsightsRouteHandler = void 0;
var _workflow_insights = require("../../../../common/api/endpoint/workflow_insights/workflow_insights");
var _services = require("../../services");
var _error_handler = require("../error_handler");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerGetInsightsRoute = (router, endpointContext) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.WORKFLOW_INSIGHTS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _workflow_insights.GetWorkflowInsightsRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canReadWorkflowInsights']
  }, endpointContext.logFactory.get('workflowInsights'), getInsightsRouteHandler(endpointContext)));
};
exports.registerGetInsightsRoute = registerGetInsightsRoute;
const getInsightsRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('workflowInsights');
  return async (context, request, response) => {
    const {
      endpointManagementSpaceAwarenessEnabled,
      defendInsightsPolicyResponseFailure
    } = endpointContext.experimentalFeatures;
    try {
      var _request$query$types;
      // Validate feature flag for policy_response_failure insights
      if ((_request$query$types = request.query.types) !== null && _request$query$types !== void 0 && _request$query$types.includes('policy_response_failure') && !defendInsightsPolicyResponseFailure) {
        return response.badRequest({
          body: 'policy_response_failure insight type requires defendInsightsPolicyResponseFailure feature flag'
        });
      }
      logger.debug('Fetching workflow insights');
      const insightsResponse = await _services.securityWorkflowInsightsService.fetch(request.query);
      const body = insightsResponse.flatMap(insight => insight._source ? {
        ...insight._source,
        id: insight._id
      } : []);

      // Ensure the insights are in the current space, judging by agent IDs
      if (endpointManagementSpaceAwarenessEnabled) {
        const spaceId = (await context.securitySolution).getSpaceId();
        const fleetServices = endpointContext.service.getInternalFleetServices(spaceId);
        const agentIds = Array.from(new Set(body.flatMap(insight => insight.target.ids)));
        await fleetServices.ensureInCurrentSpace({
          agentIds
        });
      }
      return response.ok({
        body
      });
    } catch (e) {
      return (0, _error_handler.errorHandler)(logger, response, e);
    }
  };
};
exports.getInsightsRouteHandler = getInsightsRouteHandler;