"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerResponseActionRoutes = registerResponseActionRoutes;
var _endpoint = require("../../../../common/api/endpoint");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _stringify = require("../../utils/stringify");
var _error_handler = require("../error_handler");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _services = require("../../services");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerResponseActionRoutes(router, endpointContext) {
  const logger = endpointContext.logFactory.get('hostIsolation');
  router.versioned.post({
    access: 'public',
    path: _constants.ISOLATE_HOST_ROUTE_V2,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'isolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.UNISOLATE_HOST_ROUTE_V2,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'unisolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.KILL_PROCESS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.KillProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canKillProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'kill-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.SUSPEND_PROCESS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.SuspendProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canSuspendProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'suspend-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_PROCESSES_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.GetProcessesRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canGetRunningProcesses']
  }, logger, responseActionRequestHandler(endpointContext, 'running-processes')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_FILE_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionGetFileSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'get-file')));
  router.versioned.post({
    access: 'public',
    path: _constants.EXECUTE_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ExecuteActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'execute')));
  router.versioned.post({
    access: 'public',
    path: _constants.UPLOAD_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true,
      body: {
        accepts: ['multipart/form-data'],
        output: 'stream',
        maxBytes: endpointContext.serverConfig.maxUploadResponseActionFileBytes
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UploadActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'upload')));
  router.versioned.post({
    access: 'public',
    path: _constants.SCAN_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ScanActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteScanOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'scan')));
  router.versioned.post({
    access: 'public',
    path: _constants.RUN_SCRIPT_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.RunScriptActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'runscript')));
  router.versioned.post({
    access: 'public',
    path: _constants.CANCEL_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.CancelActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canCancelAction']
  }, logger, responseActionRequestHandler(endpointContext, 'cancel'), (0, _utils.createCancelActionAdditionalChecks)(endpointContext)));
}
function responseActionRequestHandler(endpointContext, command) {
  const logger = endpointContext.logFactory.get('responseActionsHandler');
  return async (context, req, res) => {
    logger.debug(() => `response action [${command}]:\n${(0, _stringify.stringify)(req.body)}`);
    try {
      const experimentalFeatures = endpointContext.experimentalFeatures;

      // Note:  because our API schemas are defined as module static variables (as opposed to a
      //        `getter` function), we need to include this additional validation here, since
      //        `agent_type` is included in the schema independent of the feature flag
      if (isThirdPartyFeatureDisabled(req.body.agent_type, command, experimentalFeatures)) {
        return (0, _error_handler.errorHandler)(logger, res, new _custom_http_request_error.CustomHttpRequestError(`[request body.agent_type]: feature is disabled`, 400));
      }
      const coreContext = await context.core;
      const user = coreContext.security.authc.getCurrentUser();
      const esClient = coreContext.elasticsearch.client.asInternalUser;
      const casesClient = await endpointContext.service.getCasesClient(req);
      const connectorActions = (await context.actions).getActionsClient();
      const spaceId = (await context.securitySolution).getSpaceId();
      const responseActionsClient = (0, _services.getResponseActionsClient)(req.body.agent_type || 'endpoint', {
        esClient,
        casesClient,
        spaceId,
        endpointService: endpointContext.service,
        username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
        connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
      });
      const action = await (0, _utils.executeResponseAction)(command, req.body, responseActionsClient);
      const result = (0, _utils.buildResponseActionResult)(command, action);
      return res.ok(result);
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, res, err);
    }
  };
}
function isThirdPartyFeatureDisabled(agentType, command, experimentalFeatures) {
  if (agentType === 'sentinel_one' && command === 'runscript' && !experimentalFeatures.responseActionsSentinelOneRunScriptEnabled) {
    return true;
  }
  return false;
}