"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tableHasFocus = exports.resetKeyboardFocus = exports.onTimelineTabKeyPressed = exports.isPrimitiveArray = exports.isFullScreen = exports.handleIsOperator = exports.getNonDropAreaFilters = exports.focusUtilityBarAction = exports.buildIsQueryMatch = exports.buildIsOneOfQueryMatch = exports.buildGlobalQuery = exports.buildExistsQueryMatch = exports.TIMELINE_FILTER_DROP_AREA = exports.STATEFUL_EVENT_CSS_CLASS_NAME = exports.EVENTS_COUNT_BUTTON_CLASS_NAME = void 0;
var _fp = require("lodash/fp");
var _public = require("@kbn/timelines-plugin/public");
var _kql = require("../../../../common/utils/kql");
var _utility_types = require("../../../../common/utility_types");
var _kuery = require("../../../common/lib/kuery");
var _data_provider = require("./data_providers/data_provider");
var _timeline = require("../../../../common/api/timeline");
var _styles = require("./styles");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildQueryMatch = (dataProvider, browserFields) => {
  const {
    excluded,
    type,
    queryMatch: {
      field,
      operator,
      value
    }
  } = dataProvider;
  const isFieldTypeNested = (0, _kuery.checkIfFieldTypeIsNested)(field, browserFields);
  const isExcluded = excluded ? 'NOT ' : '';
  switch (operator) {
    case _data_provider.IS_OPERATOR:
      return handleIsOperator({
        browserFields,
        field,
        isExcluded,
        isFieldTypeNested,
        type,
        value
      });
    case _data_provider.EXISTS_OPERATOR:
      return `${isExcluded}${buildExistsQueryMatch({
        browserFields,
        field,
        isFieldTypeNested
      })}`;
    case _data_provider.IS_ONE_OF_OPERATOR:
      return handleIsOneOfOperator({
        field,
        isExcluded,
        value
      });
    default:
      (0, _utility_types.assertUnreachable)(operator);
  }
};
const buildGlobalQuery = (dataProviders, browserFields) => dataProviders.reduce((queries, dataProvider) => {
  const flatDataProviders = [dataProvider, ...dataProvider.and];
  const activeDataProviders = flatDataProviders.filter(flatDataProvider => flatDataProvider.enabled);
  if (!activeDataProviders.length) return queries;
  const activeDataProvidersQueries = activeDataProviders.map(activeDataProvider => buildQueryMatch(activeDataProvider, browserFields));
  const activeDataProvidersQueryMatch = activeDataProvidersQueries.join(' and ');
  return [...queries, activeDataProvidersQueryMatch];
}, []).filter(queriesItem => !(0, _fp.isEmpty)(queriesItem)).reduce((globalQuery, queryMatch, index, queries) => {
  if (queries.length <= 1) return queryMatch;
  return !index ? `(${queryMatch})` : `${globalQuery} or (${queryMatch})`;
}, '');

/**
 * The CSS class name of a "stateful event", which appears in both
 * the `Timeline` and the `Events Viewer` widget
 */
exports.buildGlobalQuery = buildGlobalQuery;
const STATEFUL_EVENT_CSS_CLASS_NAME = exports.STATEFUL_EVENT_CSS_CLASS_NAME = 'event-column-view';
const EVENTS_COUNT_BUTTON_CLASS_NAME = exports.EVENTS_COUNT_BUTTON_CLASS_NAME = 'local-events-count-button';

/** Returns true if the events table has focus */
const tableHasFocus = containerElement => (0, _public.elementOrChildrenHasFocus)(containerElement === null || containerElement === void 0 ? void 0 : containerElement.querySelector(`.${_styles.EVENTS_TABLE_CLASS_NAME}`));

/**
 * This function has a side effect. It will skip focus "after" or "before"
 * Timeline's events table, with exceptions as noted below.
 *
 * If the currently-focused table cell has additional focusable children,
 * i.e. action buttons, draggables, or always-open popover content, the
 * browser's "natural" focus management will determine which element is
 * focused next.
 */
exports.tableHasFocus = tableHasFocus;
const onTimelineTabKeyPressed = ({
  containerElement,
  keyboardEvent,
  onSkipFocusBeforeEventsTable,
  onSkipFocusAfterEventsTable
}) => {
  const {
    shiftKey
  } = keyboardEvent;
  const eventsTableSkipFocus = (0, _public.getTableSkipFocus)({
    containerElement,
    getFocusedCell: _public.getFocusedAriaColindexCell,
    shiftKey,
    tableHasFocus,
    tableClassName: _styles.EVENTS_TABLE_CLASS_NAME
  });
  if (eventsTableSkipFocus !== 'SKIP_FOCUS_NOOP') {
    (0, _public.stopPropagationAndPreventDefault)(keyboardEvent);
    (0, _public.handleSkipFocus)({
      onSkipFocusBackwards: onSkipFocusBeforeEventsTable,
      onSkipFocusForward: onSkipFocusAfterEventsTable,
      skipFocus: eventsTableSkipFocus
    });
  }
};

/**
 * Focuses the utility bar action contained by the provided `containerElement`
 * when a valid container is provided
 */
exports.onTimelineTabKeyPressed = onTimelineTabKeyPressed;
const focusUtilityBarAction = containerElement => {
  var _containerElement$que;
  containerElement === null || containerElement === void 0 ? void 0 : (_containerElement$que = containerElement.querySelector('div.siemUtilityBar__action:last-of-type button')) === null || _containerElement$que === void 0 ? void 0 : _containerElement$que.focus();
};

/**
 * Resets keyboard focus on the page
 */
exports.focusUtilityBarAction = focusUtilityBarAction;
const resetKeyboardFocus = () => {
  document.body.focus();
};
exports.resetKeyboardFocus = resetKeyboardFocus;
const handleIsOperator = ({
  browserFields,
  field,
  isExcluded,
  isFieldTypeNested,
  type,
  value
}) => {
  if (!isPrimitiveArray(value)) {
    return `${isExcluded}${type !== _timeline.DataProviderTypeEnum.template ? buildIsQueryMatch({
      browserFields,
      field,
      isFieldTypeNested,
      value
    }) : buildExistsQueryMatch({
      browserFields,
      field,
      isFieldTypeNested
    })}`;
  } else {
    return `${isExcluded}${field} : ${JSON.stringify(value)}`;
  }
};
exports.handleIsOperator = handleIsOperator;
const handleIsOneOfOperator = ({
  field,
  isExcluded,
  value
}) => {
  if (isPrimitiveArray(value)) {
    return `${isExcluded}${buildIsOneOfQueryMatch({
      field,
      value
    })}`;
  } else {
    return `${isExcluded}${field} : ${JSON.stringify(value)}`;
  }
};
const buildIsQueryMatch = ({
  browserFields,
  field,
  isFieldTypeNested,
  value
}) => {
  if (isFieldTypeNested) {
    return (0, _kuery.convertNestedFieldToQuery)(field, value, browserFields);
  } else if ((0, _kuery.checkIfFieldTypeIsDate)(field, browserFields)) {
    return (0, _kuery.convertDateFieldToQuery)(field, value);
  } else {
    return `${field} : ${(0, _kql.prepareKQLParam)(value)}`;
  }
};
exports.buildIsQueryMatch = buildIsQueryMatch;
const buildExistsQueryMatch = ({
  browserFields,
  field,
  isFieldTypeNested
}) => {
  return isFieldTypeNested ? (0, _kuery.convertNestedFieldToExistQuery)(field, browserFields).trim() : `${field} ${_data_provider.EXISTS_OPERATOR}`.trim();
};
exports.buildExistsQueryMatch = buildExistsQueryMatch;
const buildIsOneOfQueryMatch = ({
  field,
  value
}) => {
  const trimmedField = field.trim();
  if (value.length) {
    return `${trimmedField} : (${value.map(item => (0, _fp.isNumber)(item) ? item : (0, _kql.prepareKQLStringParam)(String(item).trim())).join(' OR ')})`;
  }
  return `${trimmedField} : ''`;
};
exports.buildIsOneOfQueryMatch = buildIsOneOfQueryMatch;
const isPrimitiveArray = value => Array.isArray(value) && (value.every(x => typeof x === 'string') || value.every(x => typeof x === 'number'));
exports.isPrimitiveArray = isPrimitiveArray;
const TIMELINE_FILTER_DROP_AREA = exports.TIMELINE_FILTER_DROP_AREA = 'timeline-filter-drop-area';
const getNonDropAreaFilters = (filters = []) => filters.filter(f => f.meta.controlledBy !== TIMELINE_FILTER_DROP_AREA);
exports.getNonDropAreaFilters = getNonDropAreaFilters;
const isFullScreen = ({
  globalFullScreen,
  isActiveTimelines,
  timelineFullScreen
}) => isActiveTimelines && timelineFullScreen || isActiveTimelines === false && globalFullScreen;
exports.isFullScreen = isFullScreen;