"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isUninitialisedResourceState = exports.isStaleResourceState = exports.isOutdatedResourceState = exports.isLoadingResourceState = exports.isLoadedResourceState = exports.isFailedResourceState = exports.getLastLoadedResourceState = exports.getCurrentResourceError = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * this file contains set of types to represent state of asynchronous resource.
 * Resource is defined as a reference to potential data that is loaded/updated
 * using asynchronous communication with data source (for example through REST API call).
 * Asynchronous update implies that next to just having data:
 *  - there is moment in time when data is not loaded/initialised and not in process of loading/updating
 *  - process performing data update can take considerable time which needs to be communicated to user
 *  - update can fail due to multiple reasons and also needs to be communicated to the user
 */

/**
 * Data type to represent uninitialised state of asynchronous resource.
 * This state indicates that no actions to load the data has be taken.
 */

/**
 * Data type to represent loading state of asynchronous resource. Loading state
 * should be used to indicate that data is in the process of loading/updating.
 * It contains reference to previous stale state that can be used to present
 * previous state of resource to the user (like show previous already loaded
 * data or show previous failure).
 *
 * @param Data - type of the data that is referenced by resource state
 * @param Error - type of the error that can happen during attempt to update data
 */

/**
 * Data type to represent loaded state of asynchronous resource. Loaded state
 * is characterised with reference to the loaded data.
 *
 * @param Data - type of the data that is referenced by resource state
 */

/**
 * Data type to represent failed state of asynchronous resource. Failed state
 * is characterised with error and can reference last loaded state. Reference
 * to last loaded state can be used to present previous successfully loaded data.
 *
 * @param Data - type of the data that is referenced by resource state
 * @param Error - type of the error that can happen during attempt to update data
 */

/**
 * Data type to represent stale (not loading) state of asynchronous resource.
 *
 * @param Data - type of the data that is referenced by resource state
 * @param Error - type of the error that can happen during attempt to update data
 */

/**
 * Data type to represent any state of asynchronous resource.
 *
 * @param Data - type of the data that is referenced by resource state
 * @param Error - type of the error that can happen during attempt to update data
 */

// Set of guards to narrow the type of AsyncResourceState that make further refactoring easier

const isUninitialisedResourceState = state => state.type === 'UninitialisedResourceState';
exports.isUninitialisedResourceState = isUninitialisedResourceState;
const isLoadingResourceState = state => state.type === 'LoadingResourceState';
exports.isLoadingResourceState = isLoadingResourceState;
const isLoadedResourceState = state => state.type === 'LoadedResourceState';
exports.isLoadedResourceState = isLoadedResourceState;
const isFailedResourceState = state => state.type === 'FailedResourceState';
exports.isFailedResourceState = isFailedResourceState;
const isStaleResourceState = state => isUninitialisedResourceState(state) || isLoadedResourceState(state) || isFailedResourceState(state);

// Set of functions to work with AsyncResourceState
exports.isStaleResourceState = isStaleResourceState;
const getLastLoadedResourceState = state => {
  if (isLoadedResourceState(state)) {
    return state;
  } else if (isLoadingResourceState(state) && state.previousState !== undefined) {
    return getLastLoadedResourceState(state.previousState);
  } else if (isFailedResourceState(state)) {
    return state.lastLoadedState;
  } else {
    return undefined;
  }
};
exports.getLastLoadedResourceState = getLastLoadedResourceState;
const getCurrentResourceError = state => {
  return isFailedResourceState(state) ? state.error : undefined;
};
exports.getCurrentResourceError = getCurrentResourceError;
const isOutdatedResourceState = (state, isFresh) => isUninitialisedResourceState(state) || isLoadedResourceState(state) && !isFresh(state.data) || isFailedResourceState(state) && (!state.lastLoadedState || !isFresh(state.lastLoadedState.data));
exports.isOutdatedResourceState = isOutdatedResourceState;