"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SimpleRuleSchedule = exports.RuleSchedule = void 0;
var _zod = require("@kbn/zod");
var _common_attributes = require("./common_attributes.gen");
var _time_duration = require("./time_duration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RuleSchedule = exports.RuleSchedule = _zod.z.object({
  interval: (0, _time_duration.TimeDuration)({
    allowedUnits: ['s', 'm', 'h']
  }),
  from: _common_attributes.RuleIntervalFrom,
  to: _common_attributes.RuleIntervalTo
});

/**
 * Simpler version of RuleSchedule. It's only feasible when
 * - `to` equals `now` (current moment in time)
 * - `from` is less than `now` - `interval`
 *
 * Examples:
 *
 * - rule schedule: interval = 10m, from = now-15m, to = now
 *   simpler rule schedule: interval = 10m, lookback = 5m
 *
 * - rule schedule: interval = 1h, from = now-120m, to = now
 *   simpler rule schedule: interval = 10m, lookback = 5m
 */

const SimpleRuleSchedule = exports.SimpleRuleSchedule = _zod.z.object({
  /**
   * Rule running interval in time duration format, e.g. `2m`, `3h`
   */
  interval: (0, _time_duration.TimeDuration)({
    allowedUnits: ['s', 'm', 'h']
  }),
  /**
   * Non-negative additional source events look-back to compensate rule execution delays
   * in time duration format, e.g. `2m`, `3h`.
   *
   * Having `interval`, `from` and `to` and can be calculated as
   *
   * lookback = now - `interval` - `from`, where `now` is the current moment in time
   *
   * In the other words rules use time range [now - interval - lookback, now]
   * to select source events for analysis.
   */
  lookback: (0, _time_duration.TimeDuration)({
    allowedUnits: ['s', 'm', 'h']
  })
});