"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.register = void 0;
var _lodash = require("lodash");
var _lib = require("../../../common/lib");
var _constants = require("../../../common/constants");
var _license_pre_routing_factory = require("../../lib/license_pre_routing_factory");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CLUSTER_STATUS_CHUNK_SIZE = 10;
const register = deps => {
  const {
    router,
    lib: {
      handleEsError
    }
  } = deps;
  const allHandler = async (ctx, request, response) => {
    try {
      const {
        client: clusterClient
      } = (await ctx.core).elasticsearch;
      const clusterSettings = await clusterClient.asCurrentUser.cluster.getSettings();
      const transientClusters = (0, _lodash.get)(clusterSettings, 'transient.cluster.remote', {});
      const persistentClusters = (0, _lodash.get)(clusterSettings, 'persistent.cluster.remote', {});
      const clustersSettingsByName = {
        ...transientClusters,
        ...persistentClusters
      };
      const transientClusterNames = Object.keys(transientClusters);
      const persistentClusterNames = Object.keys(persistentClusters);
      const clustersByName = await clusterClient.asCurrentUser.cluster.remoteInfo();
      const clusterNames = clustersByName && Object.keys(clustersByName) || [];
      const clusterNamesChunks = (0, _lodash.chunk)(clusterNames, CLUSTER_STATUS_CHUNK_SIZE);
      const promises = clusterNamesChunks.map(async clustersChunk => {
        try {
          return await clusterClient.asCurrentUser.indices.resolveCluster({
            name: clustersChunk.map(cluster => `${cluster}:*`),
            filter_path: '*.connected'
          }, {
            // Set a longer timeout given that sometimes unresponsive clusters
            // can take a while to respond.
            // We should be able to be more aggresive with this timeout once
            // https://github.com/elastic/elasticsearch/issues/114020 is resolved.
            requestTimeout: '60s'
          });
        } catch (error) {
          return Promise.resolve(null);
        }
      });
      const resolvedClusterStatus = await Promise.all(promises);
      // Flatten the resolved cluster status and filter out any null values
      const flattenedClusterStatus = resolvedClusterStatus.flat().filter(Boolean);
      // Combine the resolved cluster status into a single object
      const clusterStatus = (0, _lodash.assign)({}, ...flattenedClusterStatus);
      const body = clusterNames.map(clusterName => {
        var _clustersSettingsByNa, _clustersSettingsByNa2, _clusterStatus$cluste;
        const cluster = {
          ...clustersByName[clusterName],
          ...clustersSettingsByName[clusterName]
        };

        // Node connections and proxy socket connections by the user are not available in the remote info API, so we need to assign
        // them from the cluster settings.
        const nodeConnections = (_clustersSettingsByNa = clustersSettingsByName[clusterName]) === null || _clustersSettingsByNa === void 0 ? void 0 : _clustersSettingsByNa.node_connections;
        const proxySocketConnections = (_clustersSettingsByNa2 = clustersSettingsByName[clusterName]) === null || _clustersSettingsByNa2 === void 0 ? void 0 : _clustersSettingsByNa2.proxy_socket_connections;
        const isTransient = transientClusterNames.includes(clusterName);
        const isPersistent = persistentClusterNames.includes(clusterName);
        const {
          config
        } = deps;

        // If the cluster hasn't been stored in the cluster state, then it's defined by the
        // node's config file.
        const isConfiguredByNode = !isTransient && !isPersistent;

        // Pre-7.6, ES supported an undocumented "proxy" field
        // ES does not handle migrating this to the new implementation, so we need to surface it in the UI
        // This value is not available via the GET /_remote/info API, so we get it from the cluster settings
        const deprecatedProxyAddress = isPersistent ? (0, _lodash.get)(clusterSettings, `persistent.cluster.remote[${clusterName}].proxy`, undefined) : undefined;
        return {
          ...(0, _lib.deserializeCluster)(clusterName, cluster, deprecatedProxyAddress, config.isCloudEnabled, nodeConnections, proxySocketConnections),
          isConfiguredByNode,
          // We prioritize the cluster status from the resolve cluster API, and fallback to
          // the cluster connected status in case it's not present.
          isConnected: ((_clusterStatus$cluste = clusterStatus[clusterName]) === null || _clusterStatus$cluste === void 0 ? void 0 : _clusterStatus$cluste.connected) || cluster.connected
        };
      });
      return response.ok({
        body
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  };
  router.get({
    path: _constants.API_BASE_PATH,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: false
  }, (0, _license_pre_routing_factory.licensePreRoutingFactory)(deps, allHandler));
};
exports.register = register;