"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActionResultsRoute = void 0;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _server = require("@kbn/data-plugin/server");
var _route_validation = require("../../utils/build_validation/route_validation");
var _get_action_results_route = require("../../../common/api/action_results/get_action_results_route");
var _constants = require("../../../common/constants");
var _common = require("../../../common");
var _search_strategy = require("../../../common/search_strategy");
var _errors = require("../../../common/translations/errors");
var _build_query = require("../../../common/utils/build_query");
var _get_internal_saved_object_client = require("../../utils/get_internal_saved_object_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getActionResultsRoute = (router, osqueryContext) => {
  router.versioned.get({
    access: 'public',
    path: '/api/osquery/action_results/{actionId}',
    security: {
      authz: {
        requiredPrivileges: [`${_common.PLUGIN_ID}-read`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _route_validation.buildRouteValidation)(_get_action_results_route.getActionResultsRequestQuerySchema),
        params: (0, _route_validation.buildRouteValidation)(_get_action_results_route.getActionResultsRequestParamsSchema)
      }
    }
  }, async (context, request, response) => {
    const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
    try {
      var _osqueryContext$servi, _request$query$page, _request$query$pageSi, _request$query$sortOr, _request$query$sort, _integrationNamespace, _res$rawResponse$aggr, _res$rawResponse, _res$rawResponse$aggr2, _res$rawResponse$aggr3, _res$rawResponse$aggr4, _res$rawResponse2, _res$rawResponse2$agg, _res$rawResponse2$agg2, _res$rawResponse2$agg3, _res$rawResponse3, _res$rawResponse3$agg, _res$rawResponse3$agg2, _aggsBuckets$find$doc, _aggsBuckets$find, _aggsBuckets$find$doc2, _aggsBuckets$find2;
      let integrationNamespaces = {};
      const logger = osqueryContext.logFactory.get('get_action_results');
      if (osqueryContext !== null && osqueryContext !== void 0 && (_osqueryContext$servi = osqueryContext.service) !== null && _osqueryContext$servi !== void 0 && _osqueryContext$servi.getIntegrationNamespaces) {
        const spaceScopedClient = await (0, _get_internal_saved_object_client.createInternalSavedObjectsClientForSpaceId)(osqueryContext, request);
        integrationNamespaces = await osqueryContext.service.getIntegrationNamespaces([_common.OSQUERY_INTEGRATION_NAME], spaceScopedClient, logger);
        logger.debug(`Retrieved integration namespaces: ${JSON.stringify(integrationNamespaces)}`);
      }
      const search = await context.search;

      // Parse agentIds from query parameter if provided (for external API consumers)
      const requestedAgentIds = request.query.agentIds ? request.query.agentIds.split(',').map(id => id.trim()) : undefined;
      if (requestedAgentIds && requestedAgentIds.length > 100) {
        return response.badRequest({
          body: _errors.TOO_MANY_AGENT_IDS
        });
      }
      let agentIds;
      let agentIdsKuery;
      if (requestedAgentIds) {
        // Use provided agentIds for filtering (external API consumers)
        agentIds = requestedAgentIds;

        // Build kuery to filter results to requested agents
        agentIdsKuery = requestedAgentIds.map(id => `agent.id: "${id}"`).join(' OR ');
      } else {
        // Fetch action details to get agent IDs (internal UI usage)
        const {
          actionDetails
        } = await (0, _rxjs.lastValueFrom)(search.search({
          actionId: request.params.actionId,
          factoryQueryType: _search_strategy.OsqueryQueries.actionDetails,
          spaceId: (await context.core).savedObjects.client.getCurrentNamespace() || 'default'
        }, {
          abortSignal,
          strategy: 'osquerySearchStrategy'
        }));

        // Extract agent IDs from action document
        if (actionDetails !== null && actionDetails !== void 0 && actionDetails._source) {
          // Check if actionId is a child query action_id
          const queries = actionDetails._source.queries || [];
          const matchingQuery = queries.find(q => q.action_id === request.params.actionId);

          // Use query-specific agents if found, otherwise use parent action's agents
          agentIds = (matchingQuery === null || matchingQuery === void 0 ? void 0 : matchingQuery.agents) || actionDetails._source.agents || [];
        } else {
          agentIds = [];
        }
      }

      // Combine agentIds filtering with user-provided kuery
      let combinedKuery = request.query.kuery;
      if (agentIdsKuery) {
        combinedKuery = combinedKuery ? `(${agentIdsKuery}) AND (${combinedKuery})` : agentIdsKuery;
      }
      const res = await (0, _rxjs.lastValueFrom)(search.search({
        actionId: request.params.actionId,
        factoryQueryType: _search_strategy.OsqueryQueries.actionResults,
        kuery: combinedKuery,
        startDate: request.query.startDate,
        pagination: (0, _build_query.generateTablePaginationOptions)((_request$query$page = request.query.page) !== null && _request$query$page !== void 0 ? _request$query$page : 0, (_request$query$pageSi = request.query.pageSize) !== null && _request$query$pageSi !== void 0 ? _request$query$pageSi : 100),
        sort: {
          direction: (_request$query$sortOr = request.query.sortOrder) !== null && _request$query$sortOr !== void 0 ? _request$query$sortOr : _search_strategy.Direction.desc,
          field: (_request$query$sort = request.query.sort) !== null && _request$query$sort !== void 0 ? _request$query$sort : '@timestamp'
        },
        integrationNamespaces: (_integrationNamespace = integrationNamespaces[_common.OSQUERY_INTEGRATION_NAME]) !== null && _integrationNamespace !== void 0 && _integrationNamespace.length ? integrationNamespaces[_common.OSQUERY_INTEGRATION_NAME] : undefined
      }, {
        abortSignal,
        strategy: 'osquerySearchStrategy'
      }));
      const totalResponded = (_res$rawResponse$aggr = (_res$rawResponse = res.rawResponse) === null || _res$rawResponse === void 0 ? void 0 : (_res$rawResponse$aggr2 = _res$rawResponse.aggregations) === null || _res$rawResponse$aggr2 === void 0 ? void 0 : (_res$rawResponse$aggr3 = _res$rawResponse$aggr2.aggs.responses_by_action_id) === null || _res$rawResponse$aggr3 === void 0 ? void 0 : _res$rawResponse$aggr3.doc_count) !== null && _res$rawResponse$aggr !== void 0 ? _res$rawResponse$aggr : 0;
      const totalRowCount = (_res$rawResponse$aggr4 = (_res$rawResponse2 = res.rawResponse) === null || _res$rawResponse2 === void 0 ? void 0 : (_res$rawResponse2$agg = _res$rawResponse2.aggregations) === null || _res$rawResponse2$agg === void 0 ? void 0 : (_res$rawResponse2$agg2 = _res$rawResponse2$agg.aggs.responses_by_action_id) === null || _res$rawResponse2$agg2 === void 0 ? void 0 : (_res$rawResponse2$agg3 = _res$rawResponse2$agg2.rows_count) === null || _res$rawResponse2$agg3 === void 0 ? void 0 : _res$rawResponse2$agg3.value) !== null && _res$rawResponse$aggr4 !== void 0 ? _res$rawResponse$aggr4 : 0;
      const aggsBuckets = (_res$rawResponse3 = res.rawResponse) === null || _res$rawResponse3 === void 0 ? void 0 : (_res$rawResponse3$agg = _res$rawResponse3.aggregations) === null || _res$rawResponse3$agg === void 0 ? void 0 : (_res$rawResponse3$agg2 = _res$rawResponse3$agg.aggs.responses_by_action_id) === null || _res$rawResponse3$agg2 === void 0 ? void 0 : _res$rawResponse3$agg2.responses.buckets;
      const aggregations = {
        totalRowCount,
        totalResponded,
        successful: (_aggsBuckets$find$doc = aggsBuckets === null || aggsBuckets === void 0 ? void 0 : (_aggsBuckets$find = aggsBuckets.find(bucket => bucket.key === 'success')) === null || _aggsBuckets$find === void 0 ? void 0 : _aggsBuckets$find.doc_count) !== null && _aggsBuckets$find$doc !== void 0 ? _aggsBuckets$find$doc : 0,
        failed: (_aggsBuckets$find$doc2 = aggsBuckets === null || aggsBuckets === void 0 ? void 0 : (_aggsBuckets$find2 = aggsBuckets.find(bucket => bucket.key === 'error')) === null || _aggsBuckets$find2 === void 0 ? void 0 : _aggsBuckets$find2.doc_count) !== null && _aggsBuckets$find$doc2 !== void 0 ? _aggsBuckets$find$doc2 : 0,
        pending: Math.max(0, agentIds.length - totalResponded)
      };
      let processedEdges = res.edges;

      // Only process edges for internal UI (when agentIds NOT provided in request)
      if (!requestedAgentIds && agentIds.length > 0) {
        // Create placeholder edges for ALL agents
        const placeholderEdges = agentIds.map(agentId => ({
          _index: '.logs-osquery_manager.action.responses-default',
          _id: `placeholder-${agentId}`,
          _source: {},
          fields: {
            agent_id: [agentId]
          }
        }));

        // Merge real results with placeholders, keeping real results when duplicates exist
        // reverse() ensures proper ordering, uniqBy keeps first occurrence (real data)
        processedEdges = (0, _lodash.reverse)((0, _lodash.uniqBy)((0, _lodash.flatten)([res.edges, placeholderEdges]), 'fields.agent_id[0]'));
      }
      return response.ok({
        body: {
          edges: processedEdges,
          total: processedEdges.length,
          aggregations,
          inspect: res.inspect
        }
      });
    } catch (err) {
      const error = err;
      return response.customError({
        statusCode: 500,
        body: {
          message: error.message
        }
      });
    }
  });
};
exports.getActionResultsRoute = getActionResultsRoute;