"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _onechatCommon = require("@kbn/onechat-common");
var _spaces = require("../../../../utils/spaces");
var _storage = require("./storage");
var _converters = require("./converters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Client for persisted tool definitions.
 */

const createClient = ({
  space,
  logger,
  esClient
}) => {
  const storage = (0, _storage.createStorage)({
    logger,
    esClient
  });
  return new ToolClientImpl({
    space,
    storage
  });
};
exports.createClient = createClient;
class ToolClientImpl {
  constructor({
    space,
    storage
  }) {
    (0, _defineProperty2.default)(this, "space", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    this.space = space;
    this.storage = storage;
  }
  async get(id) {
    const document = await this._get(id);
    if (!document) {
      throw (0, _onechatCommon.createToolNotFoundError)({
        toolId: id
      });
    }
    return (0, _converters.fromEs)(document);
  }
  async list() {
    const document = await this.storage.getClient().search({
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space)]
        }
      },
      size: 1000,
      track_total_hits: false
    });
    return document.hits.hits.map(hit => (0, _converters.fromEs)(hit));
  }
  async create(createRequest) {
    const {
      id
    } = createRequest;
    const document = await this._get(id);
    if (document) {
      throw (0, _onechatCommon.createBadRequestError)(`Tool with id '${id}' already exists.`);
    }
    const attributes = (0, _converters.createAttributes)({
      createRequest,
      space: this.space
    });
    await this.storage.getClient().index({
      document: attributes
    });
    return this.get(id);
  }
  async update(id, update) {
    const document = await this._get(id);
    if (!document) {
      throw (0, _onechatCommon.createToolNotFoundError)({
        toolId: id
      });
    }
    const updatedAttributes = (0, _converters.updateDocument)({
      current: document._source,
      update
    });
    await this.storage.getClient().index({
      id: document._id,
      document: updatedAttributes
    });
    return (0, _converters.fromEs)({
      _id: document._id,
      _source: updatedAttributes
    });
  }
  async delete(id) {
    const document = await this._get(id);
    if (!document) {
      throw (0, _onechatCommon.createToolNotFoundError)({
        toolId: id
      });
    }
    const result = await this.storage.getClient().delete({
      id: document._id
    });
    if (result.result === 'not_found') {
      throw (0, _onechatCommon.createToolNotFoundError)({
        toolId: id
      });
    }
    return true;
  }
  async _get(id) {
    const response = await this.storage.getClient().search({
      track_total_hits: false,
      size: 1,
      terminate_after: 1,
      query: {
        bool: {
          filter: [(0, _spaces.createSpaceDslFilter)(this.space), {
            term: {
              id
            }
          }]
        }
      }
    });
    if (response.hits.hits.length === 0) {
      return undefined;
    } else {
      return response.hits.hits[0];
    }
  }
}