"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _prompts = require("./prompts");
var _i18n = require("./i18n");
var _constants = require("./constants");
var _state = require("./state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAgentGraph = ({
  chatModel,
  tools,
  configuration,
  capabilities,
  logger,
  events
}) => {
  const toolNode = new _prebuilt.ToolNode(tools);
  const researcherModel = chatModel.bindTools(tools).withConfig({
    tags: [_constants.tags.agent, _constants.tags.researchAgent]
  });
  const researchAgent = async state => {
    if (state.addedMessages.length === 0) {
      events.emit((0, _langchain.createReasoningEvent)((0, _i18n.getRandomThinkingMessage)(), {
        transient: true
      }));
    }
    const response = await researcherModel.invoke((0, _prompts.getActPrompt)({
      customInstructions: configuration.research.instructions,
      capabilities,
      messages: [...state.initialMessages, ...state.addedMessages]
    }));
    return {
      currentCycle: state.currentCycle + 1,
      nextMessage: response
    };
  };
  const shouldContinue = async state => {
    var _state$nextMessage$to;
    const hasToolCalls = state.nextMessage && ((_state$nextMessage$to = state.nextMessage.tool_calls) !== null && _state$nextMessage$to !== void 0 ? _state$nextMessage$to : []).length > 0;
    const maxCycleReached = state.currentCycle > state.cycleLimit;
    if (hasToolCalls && !maxCycleReached) {
      return _constants.steps.executeTool;
    } else {
      return _constants.steps.prepareToAnswer;
    }
  };
  const executeTool = async state => {
    const toolNodeResult = await toolNode.invoke([state.nextMessage], {});
    return {
      addedMessages: [state.nextMessage, ...toolNodeResult]
    };
  };
  const prepareToAnswer = async state => {
    const maxCycleReached = state.currentCycle > state.cycleLimit;
    let handoverNote;
    if (!maxCycleReached) {
      const handoverMessage = state.nextMessage;
      handoverNote = (0, _langchain.extractTextContent)(handoverMessage);
    }
    return {
      handoverNote,
      maxCycleReached
    };
  };
  const answeringModel = chatModel.withConfig({
    tags: [_constants.tags.agent, _constants.tags.answerAgent]
  });
  const answerAgent = async state => {
    events.emit((0, _langchain.createReasoningEvent)((0, _i18n.getRandomAnsweringMessage)(), {
      transient: true
    }));
    const response = await answeringModel.invoke((0, _prompts.getAnswerPrompt)({
      customInstructions: configuration.answer.instructions,
      capabilities,
      handoverNote: state.handoverNote,
      searchInterrupted: state.maxCycleReached,
      discussion: [...state.initialMessages, ...state.addedMessages]
    }));
    return {
      addedMessages: [response]
    };
  };

  // note: the node names are used in the event convertion logic, they should *not* be changed
  const graph = new _langgraph.StateGraph(_state.StateAnnotation)
  // nodes
  .addNode(_constants.steps.researchAgent, researchAgent).addNode(_constants.steps.executeTool, executeTool).addNode(_constants.steps.prepareToAnswer, prepareToAnswer).addNode(_constants.steps.answerAgent, answerAgent)
  // edges
  .addEdge(_langgraph.START, _constants.steps.researchAgent).addEdge(_constants.steps.executeTool, _constants.steps.researchAgent).addConditionalEdges(_constants.steps.researchAgent, shouldContinue, {
    [_constants.steps.executeTool]: _constants.steps.executeTool,
    [_constants.steps.prepareToAnswer]: _constants.steps.prepareToAnswer
  }).addEdge(_constants.steps.prepareToAnswer, _constants.steps.answerAgent).addEdge(_constants.steps.answerAgent, _langgraph.END).compile();
  return graph;
};
exports.createAgentGraph = createAgentGraph;