"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.painlessLanguageAttributes = exports.buildPainlessStartRule = exports.buildPainlessRules = void 0;
var _lexer_rules = require("../../painless/lexer_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/*
 * This rule is used inside json root to start a painless highlighting sequence
 */
const buildPainlessStartRule = (painlessRoot = 'painless_root') => {
  return [/("(?:[^"]*_)?script"|"inline"|"source")(\s*?)(:)(\s*?)(""")/, ['variable', 'whitespace', 'punctuation.colon', 'whitespace', {
    token: 'punctuation',
    next: `@${painlessRoot}`
  }]];
};

/*
 * This function creates a group of rules needed for painless highlighting in console.
 * It reuses the lexer rules from the "painless" language, but since not all rules are referenced in the root
 * tokenizer and to avoid conflicts with existing console rules, only selected rules are used.
 */
exports.buildPainlessStartRule = buildPainlessStartRule;
const buildPainlessRules = (painlessRoot = 'painless_root') => {
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const {
    root,
    comment,
    string_dq,
    string_sq
  } = _lexer_rules.lexerRules.tokenizer;
  return {
    [painlessRoot]: [
    // the rule to end painless highlighting and get back to the previous tokenizer state
    [/"""/, {
      token: 'punctuation',
      next: '@pop'
    }], ...root],
    comment,
    string_dq,
    string_sq
  };
};

/*
 * These language attributes need to be added to the console language definition for painless tokenizer
 * to work correctly.
 */
exports.buildPainlessRules = buildPainlessRules;
const painlessLanguageAttributes = exports.painlessLanguageAttributes = {
  keywords: _lexer_rules.lexerRules.keywords,
  primitives: _lexer_rules.lexerRules.primitives,
  constants: _lexer_rules.lexerRules.constants,
  operators: _lexer_rules.lexerRules.operators,
  symbols: _lexer_rules.lexerRules.symbols,
  digits: _lexer_rules.lexerRules.digits,
  octaldigits: _lexer_rules.lexerRules.octaldigits,
  binarydigits: _lexer_rules.lexerRules.binarydigits,
  hexdigits: _lexer_rules.lexerRules.hexdigits
};