"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDstEvents = createDstEvents;
exports.filterCalendarsForDst = filterCalendarsForDst;
exports.generateTimeZones = generateTimeZones;
exports.getDSTChangeDates = getDSTChangeDates;
exports.isDstCalendar = isDstCalendar;
exports.separateCalendarsByType = separateCalendarsByType;
var _i18n = require("@kbn/i18n");
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _utils = require("./edit/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const YEARS_OF_DST_EVENTS = 20;
function addZeroPadding(num) {
  return num < 10 ? `0${num}` : num;
}
const DST_CHANGE_DESCRIPTIONS = {
  WINTER: _i18n.i18n.translate('xpack.ml.calendarsEdit.dstChangeDescriptionWinter', {
    defaultMessage: 'Winter'
  }),
  SUMMER: _i18n.i18n.translate('xpack.ml.calendarsEdit.dstChangeDescriptionSummer', {
    defaultMessage: 'Summer'
  })
};
function createDstEvent(time, year, shiftSecs) {
  return {
    event_id: (0, _utils.generateTempId)(),
    description: `${shiftSecs > 0 ? DST_CHANGE_DESCRIPTIONS.SUMMER : DST_CHANGE_DESCRIPTIONS.WINTER} ${year}`,
    start_time: time.valueOf(),
    end_time: time.add(2, 'days').valueOf(),
    skip_result: false,
    skip_model_update: false,
    force_time_shift: shiftSecs
  };
}
function getDSTChangeDates(timezone, year) {
  let start = null;
  let end = null;
  for (let month = 1; month < 13; month++) {
    for (let day = 1; day <= 31; day++) {
      const date = _momentTimezone.default.tz(`${year}-${addZeroPadding(month)}-${addZeroPadding(day)} 09:00:00`, timezone);
      if (date.isValid() === false) {
        continue;
      }
      if (!start && date.isDST()) {
        // loop over hours
        for (let hour = 0; hour < 24; hour++) {
          const date2 = _momentTimezone.default.tz(`${year}-${addZeroPadding(month)}-${addZeroPadding(day)} ${addZeroPadding(hour)}:00:00`, timezone);
          if (date2.isDST() === true) {
            start = date2;
            break;
          }
        }
      }
      if (start && !end && date.isDST() === false) {
        // loop over hours
        for (let hour = 0; hour < 24; hour++) {
          const date2 = _momentTimezone.default.tz(`${year}-${addZeroPadding(month)}-${addZeroPadding(day)} ${addZeroPadding(hour)}:00:00`, timezone);
          if (date2.isDST() === false) {
            end = date2;
            break;
          }
        }
      }
    }
  }
  return {
    start,
    end,
    year
  };
}
function generateDSTChangeDates(timezone, years) {
  const thisYear = new Date().getFullYear();
  const endYear = thisYear + years;
  const dates = [];
  for (let year = thisYear; year < endYear; year++) {
    const dstChanges = getDSTChangeDates(timezone, year);
    dates.push(dstChanges);
  }
  const janDate = _momentTimezone.default.tz(`${thisYear}-01-10 09:00:00`, timezone);
  const juneDate = _momentTimezone.default.tz(`${thisYear}-06-10 09:00:00`, timezone);
  const diffMins = juneDate.utcOffset() - janDate.utcOffset();
  const shiftSecs = diffMins * 60;
  return {
    dates,
    shiftSecs
  };
}
function createDstEvents(timezone) {
  const {
    dates,
    shiftSecs
  } = generateDSTChangeDates(timezone, YEARS_OF_DST_EVENTS);
  return dates.reduce((acc, date) => {
    if (!date.start || !date.end) {
      return acc;
    }
    acc.push(createDstEvent(date.start, date.year, shiftSecs));
    acc.push(createDstEvent(date.end, date.year, -shiftSecs));
    return acc;
  }, []);
}
function isDstCalendar(calendar) {
  return calendar.events.some(event => {
    return event.force_time_shift !== undefined;
  });
}
function filterCalendarsForDst(calendars, isDst) {
  return calendars.filter(calendar => {
    return isDstCalendar(calendar) === isDst;
  });
}
function separateCalendarsByType(allCalendars) {
  const calendarsDst = [];
  const calendars = [];
  allCalendars.forEach(calendar => {
    if (isDstCalendar(calendar)) {
      calendarsDst.push(calendar);
    } else {
      calendars.push(calendar);
    }
  });
  return {
    calendarsDst,
    calendars
  };
}
function generateTimeZones() {
  const zones = _momentTimezone.default.tz.names();
  return zones;
}