"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogsSharedLogEntriesDomain = exports.LOG_ENTRIES_PAGE_SIZE = void 0;
var _log_views = require("../../../../common/log_views");
var _runtime_types = require("../../../../common/runtime_types");
var _builtin_rules = require("../../../services/log_entries/message/builtin_rules");
var _message = require("../../../services/log_entries/message/message");
var _log_entry_datasets = require("./queries/log_entry_datasets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LOG_ENTRIES_PAGE_SIZE = exports.LOG_ENTRIES_PAGE_SIZE = 200;
const FIELDS_FROM_CONTEXT = ['log.file.path', 'host.name', 'container.id'];
const COMPOSITE_AGGREGATION_BATCH_SIZE = 1000;
class LogsSharedLogEntriesDomain {
  constructor(adapter, libs) {
    this.adapter = adapter;
    this.libs = libs;
  }
  async getLogEntries(requestContext, logView, params, columnOverrides) {
    const [, {
      logsDataAccess
    }, {
      logViews
    }] = await this.libs.getStartServices();
    const {
      savedObjects,
      elasticsearch
    } = await requestContext.core;
    const logSourcesService = logsDataAccess.services.logSourcesServiceFactory.getLogSourcesService(savedObjects.client);
    const resolvedLogView = await logViews.getClient(savedObjects.client, elasticsearch.client.asCurrentUser, logSourcesService).getResolvedLogView(logView);
    const columnDefinitions = columnOverrides !== null && columnOverrides !== void 0 ? columnOverrides : resolvedLogView.columns;
    const messageFormattingRules = (0, _message.compileFormattingRules)((0, _builtin_rules.getBuiltinRules)(resolvedLogView.messageField));
    const requiredFields = getRequiredFields(resolvedLogView, messageFormattingRules);
    const {
      documents,
      hasMoreBefore,
      hasMoreAfter
    } = await this.adapter.getLogEntries(requestContext, resolvedLogView, requiredFields, params);
    const entries = documents.map(doc => {
      return {
        id: doc.id,
        index: doc.index,
        cursor: doc.cursor,
        columns: columnDefinitions.map(column => {
          if ('timestampColumn' in column) {
            return {
              columnId: column.timestampColumn.id,
              time: doc.cursor.time
            };
          } else if ('messageColumn' in column) {
            return {
              columnId: column.messageColumn.id,
              message: messageFormattingRules.format(doc.fields, doc.highlights)
            };
          } else {
            var _doc$fields$column$fi, _doc$highlights$colum;
            return {
              columnId: column.fieldColumn.id,
              field: column.fieldColumn.field,
              value: (_doc$fields$column$fi = doc.fields[column.fieldColumn.field]) !== null && _doc$fields$column$fi !== void 0 ? _doc$fields$column$fi : [],
              highlights: (_doc$highlights$colum = doc.highlights[column.fieldColumn.field]) !== null && _doc$highlights$colum !== void 0 ? _doc$highlights$colum : []
            };
          }
        }),
        context: getContextFromDoc(doc)
      };
    });
    return {
      entries,
      hasMoreBefore,
      hasMoreAfter
    };
  }
  async getLogEntryDatasets(requestContext, timestampField, indexName, startTime, endTime, runtimeMappings) {
    let datasetBuckets = [];
    let afterLatestBatchKey;
    while (true) {
      const datasetsReponse = await this.libs.framework.callWithRequest(requestContext, 'search', (0, _log_entry_datasets.createLogEntryDatasetsQuery)(indexName, timestampField, startTime, endTime, runtimeMappings, COMPOSITE_AGGREGATION_BATCH_SIZE, afterLatestBatchKey));
      const {
        after_key: afterKey,
        buckets: latestBatchBuckets
      } = (0, _runtime_types.decodeOrThrow)(_log_entry_datasets.logEntryDatasetsResponseRT)(datasetsReponse).aggregations.dataset_buckets;
      datasetBuckets = [...datasetBuckets, ...latestBatchBuckets];
      afterLatestBatchKey = afterKey;
      if (latestBatchBuckets.length < COMPOSITE_AGGREGATION_BATCH_SIZE) {
        break;
      }
    }
    return datasetBuckets.map(({
      key: {
        dataset
      }
    }) => dataset);
  }
}
exports.LogsSharedLogEntriesDomain = LogsSharedLogEntriesDomain;
const getRequiredFields = (configuration, messageFormattingRules) => {
  const fieldsFromCustomColumns = configuration.columns.reduce((accumulatedFields, logColumn) => {
    if (_log_views.logViewFieldColumnConfigurationRT.is(logColumn)) {
      return [...accumulatedFields, logColumn.fieldColumn.field];
    }
    return accumulatedFields;
  }, []);
  const fieldsFromFormattingRules = messageFormattingRules.requiredFields;
  return Array.from(new Set([...fieldsFromCustomColumns, ...fieldsFromFormattingRules, ...FIELDS_FROM_CONTEXT]));
};
const getContextFromDoc = doc => {
  var _doc$fields$container, _doc$fields$hostName, _doc$fields$logFile;
  // Get all context fields, then test for the presence and type of the ones that go together
  const containerId = (_doc$fields$container = doc.fields['container.id']) === null || _doc$fields$container === void 0 ? void 0 : _doc$fields$container[0];
  const hostName = (_doc$fields$hostName = doc.fields['host.name']) === null || _doc$fields$hostName === void 0 ? void 0 : _doc$fields$hostName[0];
  const logFilePath = (_doc$fields$logFile = doc.fields['log.file.path']) === null || _doc$fields$logFile === void 0 ? void 0 : _doc$fields$logFile[0];
  if (typeof containerId === 'string') {
    return {
      'container.id': containerId
    };
  }
  if (typeof hostName === 'string' && typeof logFilePath === 'string') {
    return {
      'host.name': hostName,
      'log.file.path': logFilePath
    };
  }
  return {};
};