"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.canHaveParams = canHaveParams;
exports.getInvalidParams = getInvalidParams;
exports.getMissingParams = getMissingParams;
exports.getRawQueryValidationError = exports.getQueryValidationError = void 0;
exports.getWrongTypeParams = getWrongTypeParams;
exports.hasFiltersConflicts = hasFiltersConflicts;
exports.hasFunctionFieldArgument = hasFunctionFieldArgument;
exports.hasInvalidOperations = hasInvalidOperations;
exports.isArgumentValidType = isArgumentValidType;
exports.isParsingError = isParsingError;
exports.runASTValidation = runASTValidation;
exports.shouldHaveFieldArgument = shouldHaveFieldArgument;
exports.tryToParse = tryToParse;
exports.validateMathNodes = validateMathNodes;
exports.validateParams = validateParams;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _tinymath = require("@kbn/tinymath");
var _esQuery = require("@kbn/es-query");
var _lensFormulaDocs = require("@kbn/lens-formula-docs");
var _common = require("@kbn/data-plugin/common");
var _utils = require("../../../../../utils");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RETURN_TYPE = (0, _lensFormulaDocs.getTypeI18n)('number');
function getNodeLocation(node) {
  return [node.location].filter(_utils.nonNullable);
}
function getArgumentType(arg, operations) {
  if (!(0, _lodash.isObject)(arg)) {
    return (0, _lensFormulaDocs.getTypeI18n)(typeof arg);
  }
  if (arg.type === 'function') {
    if (_lensFormulaDocs.tinymathFunctions[arg.name]) {
      var _tinymathFunctions$ar;
      return (_tinymathFunctions$ar = _lensFormulaDocs.tinymathFunctions[arg.name].outputType) !== null && _tinymathFunctions$ar !== void 0 ? _tinymathFunctions$ar : DEFAULT_RETURN_TYPE;
    }
    // Assume it's a number for now
    if (operations[arg.name]) {
      return DEFAULT_RETURN_TYPE;
    }
  }
  // leave for now other argument types
}
function isParsingError(message) {
  return message.includes('Failed to parse expression');
}
function findFunctionNodes(root) {
  function flattenFunctionNodes(node) {
    if (!(0, _lodash.isObject)(node) || node.type !== 'function') {
      return [];
    }
    return [node, ...node.args.flatMap(flattenFunctionNodes)].filter(_utils.nonNullable);
  }
  return flattenFunctionNodes(root);
}
function hasInvalidOperations(node, operations) {
  const nodes = findFunctionNodes(node).filter(v => !(0, _util.isMathNode)(v) && !operations[v.name]);
  return {
    // avoid duplicates
    names: Array.from(new Set(nodes.map(({
      name
    }) => name))),
    locations: nodes.map(({
      location
    }) => location).filter(_utils.nonNullable)
  };
}
const getRawQueryValidationError = (text, operations) => {
  // try to extract the query context here
  const singleLine = text.split('\n').join('');
  const languagesRegexp = /(kql|lucene)/;
  const allArgs = singleLine.split(',').filter(args => languagesRegexp.test(args));
  // check for the presence of a valid ES operation
  const containsOneValidOperation = Object.keys(operations).some(operation => singleLine.includes(operation));
  // no args or no valid operation, no more work to do here
  if (allArgs.length === 0 || !containsOneValidOperation) {
    return undefined;
  }
  // at this point each entry in allArgs may contain one or more
  // in the worst case it would be a math chain of count operation
  // For instance: count(kql=...) + count(lucene=...) - count(kql=...)
  // therefore before partition them, split them by "count" keywork and filter only string with a length
  const flattenArgs = allArgs.flatMap(arg => arg.split('count').filter(subArg => languagesRegexp.test(subArg)));
  const [kqlQueries, luceneQueries] = (0, _lodash.partition)(flattenArgs, arg => /kql/.test(arg));
  for (const kqlQuery of kqlQueries) {
    const message = validateQueryQuotes(kqlQuery, 'kql');
    if (message) {
      return {
        id: 'invalidQuery',
        meta: {
          language: 'kql'
        },
        message
      };
    }
  }
  for (const luceneQuery of luceneQueries) {
    const message = validateQueryQuotes(luceneQuery, 'lucene');
    if (message) {
      return {
        id: 'invalidQuery',
        meta: {
          language: 'lucene'
        },
        message
      };
    }
  }
};
exports.getRawQueryValidationError = getRawQueryValidationError;
const validateQueryQuotes = (rawQuery, language) => {
  // check if the raw argument has the minimal requirements
  // use the rest operator here to handle cases where comparison operations are used in the query
  const [, ...rawValue] = rawQuery.split('=');
  const fullRawValue = (rawValue || ['']).join('');
  const cleanedRawValue = fullRawValue.trim();
  // it must start with a single quote, and quotes must have a closure
  if (cleanedRawValue.length && (cleanedRawValue[0] !== "'" || !/'\s*([^']+?)\s*'/.test(fullRawValue)) &&
  // there's a special case when it's valid as two single quote strings
  cleanedRawValue !== "''") {
    return _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationQueryError', {
      defaultMessage: `Single quotes are required for {language}='' at {rawQuery}`,
      values: {
        language,
        rawQuery
      }
    });
  }
};
const getQueryValidationError = ({
  value: query,
  name: language,
  text
}, indexPattern) => {
  if (language !== 'kql' && language !== 'lucene') {
    return;
  }
  // check if the raw argument has the minimal requirements
  const result = validateQueryQuotes(text, language);
  // forward the error here is ok?
  if (result) {
    return result;
  }
  try {
    if (language === 'kql') {
      (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(query), indexPattern);
    } else {
      (0, _esQuery.luceneStringToDsl)(query);
    }
    return;
  } catch (e) {
    return e.message;
  }
};
exports.getQueryValidationError = getQueryValidationError;
function getMessageFromId({
  id,
  meta
}, locations) {
  switch (id) {
    case 'invalidQuery':
      return {
        id,
        meta,
        locations,
        // this is just a placeholder because the actual message comes from the query validator
        message: _i18n.i18n.translate('xpack.lens.indexPattern.invalidQuery', {
          defaultMessage: 'Invalid {language} query, please check the syntax and retry',
          values: {
            language: meta.language
          }
        })
      };
    case 'wrongFirstArgument':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationWrongFirstArgument', {
          defaultMessage: 'The first argument for {operation} should be a {type} name. Found {argument}',
          values: {
            operation: meta.operation,
            type: meta.type,
            argument: meta.argument
          }
        })
      };
    case 'shouldNotHaveField':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldNotRequired', {
          defaultMessage: 'The operation {operation} does not accept any field as argument',
          values: {
            operation: meta.operation
          }
        })
      };
    case 'cannotAcceptParameter':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaParameterNotRequired', {
          defaultMessage: 'The operation {operation} does not accept any parameter',
          values: {
            operation: meta.operation
          }
        })
      };
    case 'missingParameter':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaExpressionNotHandled', {
          defaultMessage: 'The operation {operation} in the Formula is missing the following parameters: {params}',
          values: {
            operation: meta.operation,
            params: meta.params
          }
        })
      };
    case 'wrongTypeParameter':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaExpressionWrongType', {
          defaultMessage: 'The parameters for the operation {operation} in the Formula are of the wrong type: {params}',
          values: {
            operation: meta.operation,
            params: meta.params
          }
        })
      };
    case 'wrongTypeArgument':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaExpressionWrongTypeArgument', {
          defaultMessage: 'The {name} argument for the operation {operation} in the Formula is of the wrong type: {type} instead of {expectedType}',
          values: {
            operation: meta.operation,
            name: meta.name,
            type: meta.type,
            expectedType: meta.expectedType
          }
        })
      };
    case 'duplicateArgument':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationDuplicateParams', {
          defaultMessage: 'The parameters for the operation {operation} have been declared multiple times: {params}',
          values: {
            operation: meta.operation,
            params: meta.params
          }
        })
      };
    case 'missingField':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldNotFound', {
          defaultMessage: '{missingFieldCount, plural, one {Field} other {Fields}} {missingFieldList} not found',
          values: {
            missingFieldCount: meta.fieldList.length,
            missingFieldList: meta.fieldList.join(', ')
          }
        })
      };
    case 'missingOperation':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.operationsNotFound', {
          defaultMessage: '{operationLength, plural, one {Operation} other {Operations}} {operationsList} not found',
          values: {
            operationLength: meta.operationLength,
            operationsList: meta.operationsList
          }
        })
      };
    case 'fieldWithNoOperation':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.fieldNoOperation', {
          defaultMessage: 'The field {field} cannot be used without operation',
          values: {
            field: meta.field
          }
        })
      };
    case 'failedParsing':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaExpressionParseError', {
          defaultMessage: 'The Formula {expression} cannot be parsed',
          values: {
            expression: meta.expression
          }
        })
      };
    case 'tooManyArguments':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaWithTooManyArguments', {
          defaultMessage: 'The operation {operation} has too many arguments',
          values: {
            operation: meta.operation
          }
        })
      };
    case 'missingMathArgument':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaMathMissingArgument', {
          defaultMessage: 'The operation {operation} in the Formula is missing {count} arguments: {params}',
          values: {
            operation: meta.operation,
            count: meta.count,
            params: meta.params
          }
        })
      };
    case 'tooManyQueries':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationDoubleQueryError', {
          defaultMessage: 'Use only one of kql= or lucene=, not both'
        })
      };
    case 'tooManyFirstArguments':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationTooManyFirstArguments', {
          defaultMessage: 'The operation {operation} in the Formula requires a {supported, plural, one {single} other {supported}} {type}, found: {text}',
          values: {
            operation: meta.operation,
            text: meta.text,
            type: meta.type,
            supported: meta.supported || 1
          }
        })
      };
    case 'wrongArgument':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationwrongArgument', {
          defaultMessage: 'The operation {operation} in the Formula does not support {type} parameters, found: {text}',
          values: {
            operation: meta.operation,
            text: meta.text,
            type: meta.type
          }
        })
      };
    case 'wrongReturnedType':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationWrongReturnedType', {
          defaultMessage: 'The return value type of the operation {text} is not supported in Formula',
          values: {
            text: meta.text
          }
        })
      };
    case 'filtersTypeConflict':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationFiltersTypeConflicts', {
          defaultMessage: 'The Formula filter of type "{outerType}" is not compatible with the inner filter of type "{innerType}" from the {operation} operation',
          values: {
            operation: meta.operation,
            outerType: meta.outerType,
            innerType: meta.innerType
          }
        })
      };
    case 'useAlternativeFunction':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.formulaUseAlternative', {
          defaultMessage: `The operation {operation} in the Formula is missing the {params} argument: use the {alternativeFn} operation instead`,
          values: {
            operation: meta.operation,
            params: meta.params,
            alternativeFn: meta.alternativeFn
          }
        })
      };
    case _common.REASON_IDS.missingTimerange:
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.absoluteMissingTimeRange', {
          defaultMessage: 'Invalid time shift. No time range found as reference'
        })
      };
    case _common.REASON_IDS.invalidDate:
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.absoluteInvalidDate', {
          defaultMessage: 'Invalid time shift. The date is not of the correct format'
        })
      };
    case _common.REASON_IDS.shiftAfterTimeRange:
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.absoluteAfterTimeRange', {
          defaultMessage: 'Invalid time shift. The provided date is after the current time range'
        })
      };
    case _common.REASON_IDS.notAbsoluteTimeShift:
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.notAbsoluteTimeShift', {
          defaultMessage: 'Invalid time shift.'
        })
      };
    case 'invalidTimeShift':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.invalidTimeShift', {
          defaultMessage: 'Invalid time shift. Enter positive integer amount followed by one of the units s, m, h, d, w, M, y. For example 3h for 3 hours'
        })
      };
    case 'invalidReducedTimeRange':
      return {
        id,
        meta,
        locations,
        message: _i18n.i18n.translate('xpack.lens.indexPattern.invalidReducedTimeRange', {
          defaultMessage: 'Invalid reduced time range. Enter positive integer amount followed by one of the units s, m, h, d, w, M, y. For example 3h for 3 hours'
        })
      };
  }
}
function tryToParse(formula, operations) {
  let root;
  try {
    root = (0, _tinymath.parse)(formula);
  } catch (e) {
    // A tradeoff is required here, unless we want to reimplement a full parser
    // Internally the function has the following logic:
    // * if the formula contains no existing ES operation, assume it's a plain parse failure
    // * if the formula contains at least one existing operation, check for query problems
    const maybeQueryProblems = getRawQueryValidationError(formula, operations);
    if (maybeQueryProblems) {
      return {
        error: {
          ...maybeQueryProblems,
          locations: []
        }
      };
    }
    return {
      error: getMessageFromId({
        id: 'failedParsing',
        meta: {
          expression: formula
        }
      }, [])
    };
  }
  return {
    root
  };
}
function runASTValidation(ast, layer, indexPattern, operations, currentColumn, dateRange) {
  return [...checkMissingVariableOrFunctions(ast, layer, indexPattern, operations), ...checkTopNodeReturnType(ast), ...runFullASTValidation(ast, layer, indexPattern, operations, dateRange, currentColumn)];
}
function checkVariableEdgeCases(ast, missingVariables) {
  const invalidVariableErrors = [];
  if ((0, _lodash.isObject)(ast) && ast.type === 'variable' && !missingVariables.has(ast.value)) {
    invalidVariableErrors.push(getMessageFromId({
      id: 'fieldWithNoOperation',
      meta: {
        field: ast.value
      }
    }, [ast.location]));
  }
  return invalidVariableErrors;
}
function checkMissingVariableOrFunctions(ast, layer, indexPattern, operations) {
  const missingErrors = [];
  const missingOperations = hasInvalidOperations(ast, operations);
  if (missingOperations.names.length) {
    missingErrors.push(getMessageFromId({
      id: 'missingOperation',
      meta: {
        operationLength: missingOperations.names.length,
        operationsList: missingOperations.names.join(', ')
      }
    }, missingOperations.locations));
  }
  const missingVariables = (0, _util.findVariables)(ast).filter(
  // filter empty string as well?
  ({
    value
  }) => !indexPattern.getFieldByName(value) && !layer.columns[value]);

  // need to check the arguments here: check only strings for now
  if (missingVariables.length) {
    missingErrors.push({
      ...getMessageFromId({
        id: 'missingField',
        meta: {
          fieldList: [...new Set(missingVariables.map(({
            value
          }) => value))]
        }
      }, missingVariables.map(({
        location
      }) => location))
    });
  }
  const invalidVariableErrors = checkVariableEdgeCases(ast, new Set(missingVariables.map(({
    value
  }) => value)));
  return [...missingErrors, ...invalidVariableErrors];
}
function getQueryValidationErrors(namedArguments, indexPattern, dateRange) {
  const errors = [];
  (namedArguments !== null && namedArguments !== void 0 ? namedArguments : []).forEach(arg => {
    if (arg.name === 'kql' || arg.name === 'lucene') {
      const message = getQueryValidationError(arg, indexPattern);
      if (message) {
        errors.push({
          id: 'invalidQuery',
          message,
          meta: {
            language: arg.name
          },
          locations: [arg.location]
        });
      }
    }
    if (arg.name === 'shift') {
      const parsedShift = (0, _common.parseTimeShift)(arg.value);
      if (parsedShift === 'invalid') {
        if ((0, _common.isAbsoluteTimeShift)(arg.value)) {
          // try to parse as absolute time shift
          const errorId = (0, _common.validateAbsoluteTimeShift)(arg.value, dateRange ? {
            from: dateRange.fromDate,
            to: dateRange.toDate
          } : undefined);
          if (errorId) {
            errors.push(getMessageFromId({
              id: errorId
            }, [arg.location]));
          }
        } else {
          errors.push(getMessageFromId({
            id: 'invalidTimeShift'
          }, [arg.location]));
        }
      }
    }
    if (arg.name === 'reducedTimeRange') {
      const parsedReducedTimeRange = (0, _common.parseTimeShift)(arg.value || '');
      if (parsedReducedTimeRange === 'invalid' || parsedReducedTimeRange === 'previous') {
        errors.push(getMessageFromId({
          id: 'invalidReducedTimeRange'
        }, [arg.location]));
      }
    }
  });
  return errors;
}
function checkSingleQuery(namedArguments) {
  return namedArguments ? namedArguments.filter(arg => arg.name === 'kql' || arg.name === 'lucene').length > 1 : undefined;
}
function validateFiltersArguments(node, nodeOperation, namedArguments, globalFilters) {
  const errors = [];
  const {
    conflicts,
    innerType,
    outerType
  } = hasFiltersConflicts(nodeOperation, namedArguments, globalFilters);
  if (conflicts) {
    if (innerType && outerType) {
      errors.push(getMessageFromId({
        id: 'filtersTypeConflict',
        meta: {
          operation: node.name,
          innerType,
          outerType
        }
      }, getNodeLocation(node)));
    }
  }
  return errors;
}
function validateNameArguments(node, nodeOperation, namedArguments, indexPattern, dateRange) {
  const errors = [];
  const missingParams = getMissingParams(nodeOperation, namedArguments);
  if (missingParams.length) {
    errors.push(getMessageFromId({
      id: 'missingParameter',
      meta: {
        operation: node.name,
        params: missingParams.map(({
          name
        }) => name).join(', ')
      }
    }, getNodeLocation(node)));
  }
  const wrongTypeParams = getWrongTypeParams(nodeOperation, namedArguments);
  if (wrongTypeParams.length) {
    errors.push(getMessageFromId({
      id: 'wrongTypeParameter',
      meta: {
        operation: node.name,
        params: wrongTypeParams.map(({
          name
        }) => name).join(', ')
      }
    }, getNodeLocation(node)));
  }
  const duplicateParams = getDuplicateParams(namedArguments);
  if (duplicateParams.length) {
    errors.push(getMessageFromId({
      id: 'duplicateArgument',
      meta: {
        operation: node.name,
        params: duplicateParams.join(', ')
      }
    }, getNodeLocation(node)));
  }
  const queryValidationErrors = getQueryValidationErrors(namedArguments, indexPattern, dateRange);
  errors.push(...queryValidationErrors);
  const hasTooManyQueries = checkSingleQuery(namedArguments);
  if (hasTooManyQueries) {
    errors.push(getMessageFromId({
      id: 'tooManyQueries'
    }, getNodeLocation(node)));
  }
  return errors;
}
function checkTopNodeReturnType(ast) {
  var _tinymathFunctions$as;
  if ((0, _lodash.isObject)(ast) && ast.type === 'function' && ast.text && (((_tinymathFunctions$as = _lensFormulaDocs.tinymathFunctions[ast.name]) === null || _tinymathFunctions$as === void 0 ? void 0 : _tinymathFunctions$as.outputType) || DEFAULT_RETURN_TYPE) !== DEFAULT_RETURN_TYPE) {
    return [getMessageFromId({
      id: 'wrongReturnedType',
      meta: {
        text: ast.text
      }
    }, getNodeLocation(ast))];
  }
  return [];
}
function runFullASTValidation(ast, layer, indexPattern, operations, dateRange, currentColumn) {
  const missingVariables = (0, _util.findVariables)(ast).filter(
  // filter empty string as well?
  ({
    value
  }) => !indexPattern.getFieldByName(value) && !layer.columns[value]);
  const missingVariablesSet = new Set(missingVariables.map(({
    value
  }) => value));
  const globalFilter = currentColumn === null || currentColumn === void 0 ? void 0 : currentColumn.filter;
  function validateNode(node) {
    if (!(0, _lodash.isObject)(node) || node.type !== 'function') {
      return [];
    }
    const nodeOperation = operations[node.name];
    const errors = [];
    const {
      namedArguments,
      functions,
      variables
    } = (0, _util.groupArgsByType)(node.args);
    const [firstArg] = (node === null || node === void 0 ? void 0 : node.args) || [];
    if (!nodeOperation) {
      errors.push(...validateMathNodes(node, missingVariablesSet, operations));
      // carry on with the validation for all the functions within the math operation
      if (functions !== null && functions !== void 0 && functions.length) {
        return errors.concat(functions.flatMap(fn => validateNode(fn)));
      }
    } else {
      if (nodeOperation.input === 'field') {
        if (!isArgumentValidType(firstArg, 'variable')) {
          if ((0, _util.isMathNode)(firstArg)) {
            errors.push(getMessageFromId({
              id: 'wrongFirstArgument',
              meta: {
                operation: node.name,
                type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldValue', {
                  defaultMessage: 'field'
                }),
                argument: `math operation`
              }
            }, getNodeLocation(node)));
          } else {
            if (shouldHaveFieldArgument(node)) {
              errors.push(getMessageFromId({
                id: 'wrongFirstArgument',
                meta: {
                  operation: node.name,
                  type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldValue', {
                    defaultMessage: 'field'
                  }),
                  argument: (0, _util.getValueOrName)(firstArg) || _i18n.i18n.translate('xpack.lens.indexPattern.formulaNoFieldForOperation', {
                    defaultMessage: 'no field'
                  })
                }
              }, getNodeLocation(node)));
            }
          }
        } else {
          // If the first argument is valid proceed with the other arguments validation
          const fieldErrors = validateFieldArguments(node, variables, {
            isFieldOperation: true,
            firstArg,
            returnedType: getReturnedType(nodeOperation, indexPattern, firstArg)
          });
          if (fieldErrors.length) {
            errors.push(...fieldErrors);
          }
        }
        const functionErrors = validateFunctionArguments(node, functions, 0, {
          isFieldOperation: true,
          type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldValue', {
            defaultMessage: 'field'
          }),
          firstArgValidation: false
        });
        if (functionErrors.length) {
          errors.push(...functionErrors);
        }
        if (!canHaveParams(nodeOperation) && namedArguments.length) {
          errors.push(getMessageFromId({
            id: 'cannotAcceptParameter',
            meta: {
              operation: node.name
            }
          }, getNodeLocation(node)));
        } else {
          const argumentsErrors = validateNameArguments(node, nodeOperation, namedArguments, indexPattern, dateRange);
          const filtersErrors = validateFiltersArguments(node, nodeOperation, namedArguments, globalFilter);
          errors.push(...argumentsErrors, ...filtersErrors);
        }
        return errors;
      }
      if (nodeOperation.input === 'fullReference') {
        // What about fn(7 + 1)? We may want to allow that
        // In general this should be handled down the Esaggs route rather than here
        const isFirstArgumentNotValid = Boolean(!isArgumentValidType(firstArg, 'function') || (0, _util.isMathNode)(firstArg) && validateMathNodes(firstArg, missingVariablesSet, operations).length);
        // First field has a special handling
        if (isFirstArgumentNotValid) {
          errors.push(getMessageFromId({
            id: 'wrongFirstArgument',
            meta: {
              operation: node.name,
              type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaOperationValue', {
                defaultMessage: 'operation'
              }),
              argument: (0, _util.getValueOrName)(firstArg) || _i18n.i18n.translate('xpack.lens.indexPattern.formulaNoOperation', {
                defaultMessage: 'no operation'
              })
            }
          }, getNodeLocation(node)));
        }
        // Check for multiple function passed
        const requiredFunctions = nodeOperation.requiredReferences ? nodeOperation.requiredReferences.length : 1;
        const functionErrors = validateFunctionArguments(node, functions, requiredFunctions, {
          isFieldOperation: false,
          firstArgValidation: isFirstArgumentNotValid,
          type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaMetricValue', {
            defaultMessage: 'metric'
          })
        });
        if (functionErrors.length) {
          errors.push(...functionErrors);
        }
        if (!canHaveParams(nodeOperation) && namedArguments.length) {
          errors.push(getMessageFromId({
            id: 'cannotAcceptParameter',
            meta: {
              operation: node.name
            }
          }, getNodeLocation(node)));
        } else {
          // check for fields passed at any position
          const fieldErrors = validateFieldArguments(node, variables, {
            isFieldOperation: false,
            firstArg,
            returnedType: undefined
          });
          const argumentsErrors = validateNameArguments(node, nodeOperation, namedArguments, indexPattern, dateRange);
          const filtersErrors = validateFiltersArguments(node, nodeOperation, namedArguments, globalFilter);
          errors.push(...fieldErrors, ...argumentsErrors, ...filtersErrors);
        }
      }
      return errors.concat(validateNode(functions[0]));
    }
    return errors;
  }
  return validateNode(ast);
}
function canHaveParams(operation) {
  return Boolean((operation.operationParams || []).length) || operation.filterable;
}
function getInvalidParams(operation, params = []) {
  return validateParams(operation, params).filter(({
    isMissing,
    isCorrectType,
    isRequired
  }) => isMissing && isRequired || !isCorrectType);
}
function getMissingParams(operation, params = []) {
  return validateParams(operation, params).filter(({
    isMissing,
    isRequired
  }) => isMissing && isRequired);
}
function getWrongTypeParams(operation, params = []) {
  return validateParams(operation, params).filter(({
    isCorrectType,
    isMissing
  }) => !isCorrectType && !isMissing);
}
function getReturnedType(operation, indexPattern, firstArg) {
  var _operation$getPossibl;
  const variables = (0, _util.findVariables)(firstArg);
  if (variables.length !== 1) {
    return;
  }
  const field = indexPattern.getFieldByName((0, _util.getValueOrName)(variables[0]));
  // while usually this is used where it is safe, as generic function it should check anyway
  if (!field) {
    return;
  }
  // here we're validating the support of the returned type for Formula, not for the operation itself
  // that is already handled indipendently by the operation. So return the scale type
  return (_operation$getPossibl = operation.getPossibleOperationForField(field)) === null || _operation$getPossibl === void 0 ? void 0 : _operation$getPossibl.scale;
}
function getDuplicateParams(params = []) {
  const uniqueArgs = Object.create(null);
  for (const {
    name
  } of params) {
    const counter = uniqueArgs[name] || 0;
    uniqueArgs[name] = counter + 1;
  }
  const uniqueNames = Object.keys(uniqueArgs);
  if (params.length > uniqueNames.length) {
    return uniqueNames.filter(name => uniqueArgs[name] > 1);
  }
  return [];
}
function hasFiltersConflicts(operation, params = [], globalFilter) {
  const paramsObj = (0, _util.getOperationParams)(operation, params);
  if (!operation.filterable || !globalFilter || !(paramsObj.kql || paramsObj.lucene)) {
    return {
      conflicts: false
    };
  }
  const language = globalFilter.language === 'kuery' ? 'kql' : globalFilter.language;
  const conflicts = !(language in paramsObj);
  return {
    conflicts,
    innerType: paramsObj.lucene ? 'lucene' : 'kql',
    outerType: language
  };
}
function validateParams(operation, params = []) {
  var _operation$operationP;
  const paramsObj = (0, _util.getOperationParams)(operation, params);
  const formalArgs = [...((_operation$operationP = operation.operationParams) !== null && _operation$operationP !== void 0 ? _operation$operationP : [])];
  if (operation.filterable) {
    formalArgs.push({
      name: 'kql',
      type: 'string',
      required: false
    }, {
      name: 'lucene',
      type: 'string',
      required: false
    });
  }
  return formalArgs.map(({
    name,
    type,
    required
  }) => ({
    name,
    isMissing: !(name in paramsObj),
    isCorrectType: typeof paramsObj[name] === type,
    isRequired: required
  }));
}
function shouldHaveFieldArgument(node) {
  return hasFunctionFieldArgument(node.name);
}
function hasFunctionFieldArgument(type) {
  return !['count'].includes(type);
}
function isArgumentValidType(arg, type) {
  return (0, _lodash.isObject)(arg) && arg.type === type;
}
function validateMathNodes(root, missingVariableSet, operations) {
  const mathNodes = (0, _util.findMathNodes)(root);
  const errors = [];
  mathNodes.forEach(node => {
    const {
      positionalArguments
    } = _lensFormulaDocs.tinymathFunctions[node.name];
    const mandatoryArguments = positionalArguments.filter(({
      optional
    }) => !optional);
    if (!node.args.length) {
      // we can stop here
      return errors.push(getMessageFromId({
        id: 'missingMathArgument',
        meta: {
          operation: node.name,
          count: mandatoryArguments.length,
          params: mandatoryArguments.map(({
            name
          }) => name).join(', ')
        }
      }, getNodeLocation(node)));
    }
    if (node.args.length > positionalArguments.length) {
      errors.push(getMessageFromId({
        id: 'tooManyArguments',
        meta: {
          operation: node.name
        }
      }, getNodeLocation(node)));
    }

    // no need to iterate all the arguments, one field is enough to trigger the error
    const hasFieldAsArgument = positionalArguments.some((requirements, index) => {
      const arg = node.args[index];
      if (arg != null && typeof arg !== 'number') {
        return arg.type === 'variable' && !missingVariableSet.has(arg.value);
      }
    });
    if (hasFieldAsArgument) {
      errors.push(getMessageFromId({
        id: 'shouldNotHaveField',
        meta: {
          operation: node.name
        }
      }, getNodeLocation(node)));
    }

    // if there is only 1 mandatory arg, this is already handled by the wrongFirstArgument check
    if (mandatoryArguments.length > 1 && node.args.length < mandatoryArguments.length) {
      const missingArgs = mandatoryArguments.filter((_, i) => node.args[i] == null);
      const [missingArgsWithAlternatives, missingArgsWithoutAlternative] = (0, _lodash.partition)(missingArgs, v => v.alternativeWhenMissing != null);
      if (missingArgsWithoutAlternative.length) {
        errors.push(getMessageFromId({
          id: 'missingMathArgument',
          meta: {
            operation: node.name,
            count: mandatoryArguments.length - node.args.length,
            params: missingArgsWithoutAlternative.map(({
              name
            }) => name).join(', ')
          }
        }, getNodeLocation(node)));
      }
      if (missingArgsWithAlternatives.length) {
        // pick only the first missing argument alternative
        const [firstArg] = missingArgsWithAlternatives;
        errors.push(getMessageFromId({
          id: 'useAlternativeFunction',
          meta: {
            operation: node.name,
            params: firstArg.name,
            alternativeFn: firstArg.alternativeWhenMissing
          }
        }, getNodeLocation(node)));
      }
    }
    const wrongTypeArgumentIndexes = positionalArguments.map(({
      type
    }, index) => {
      const arg = node.args[index];
      if (arg != null) {
        const argType = getArgumentType(arg, operations);
        if (argType && argType !== type) {
          return index;
        }
      }
    }).filter(_utils.nonNullable);
    for (const wrongTypeArgumentIndex of wrongTypeArgumentIndexes) {
      const arg = node.args[wrongTypeArgumentIndex];
      errors.push(getMessageFromId({
        id: 'wrongTypeArgument',
        meta: {
          operation: node.name,
          name: positionalArguments[wrongTypeArgumentIndex].name,
          type: getArgumentType(arg, operations) || DEFAULT_RETURN_TYPE,
          expectedType: positionalArguments[wrongTypeArgumentIndex].type || ''
        }
      }, getNodeLocation(node)));
    }
  });
  return errors;
}
function validateFieldArguments(node, variables, {
  isFieldOperation,
  firstArg,
  returnedType
}) {
  const fields = variables.filter(arg => isArgumentValidType(arg, 'variable') && !(0, _util.isMathNode)(arg));
  const errors = [];
  if (isFieldOperation && (fields.length > 1 || fields.length === 1 && fields[0] !== firstArg)) {
    errors.push(getMessageFromId({
      id: 'tooManyFirstArguments',
      meta: {
        operation: node.name,
        type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldValue', {
          defaultMessage: 'field'
        }),
        supported: 1,
        text: fields.map(({
          text
        }) => text).join(', ')
      }
    }, getNodeLocation(node)));
  }
  if (isFieldOperation && fields.length === 1 && fields[0] === firstArg) {
    if (returnedType === 'ordinal') {
      var _node$text;
      errors.push(getMessageFromId({
        id: 'wrongReturnedType',
        meta: {
          text: (_node$text = node.text) !== null && _node$text !== void 0 ? _node$text : `${node.name}(${(0, _util.getValueOrName)(firstArg)})`
        }
      }, getNodeLocation(node)));
    }
  }
  if (!isFieldOperation && fields.length) {
    errors.push(getMessageFromId({
      id: 'wrongArgument',
      meta: {
        operation: node.name,
        text: fields.map(({
          text
        }) => text).join(', '),
        type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFieldValue', {
          defaultMessage: 'field'
        })
      }
    }, getNodeLocation(node)));
  }
  return errors;
}
function validateFunctionArguments(node, functions, requiredFunctions = 0, {
  isFieldOperation,
  firstArgValidation,
  type
}) {
  const errors = [];
  // For math operation let the native operation run its own validation
  const [esOperations, mathOperations] = (0, _lodash.partition)(functions, arg => !(0, _util.isMathNode)(arg));
  if (esOperations.length > requiredFunctions) {
    if (isFieldOperation) {
      errors.push(getMessageFromId({
        id: 'wrongArgument',
        meta: {
          operation: node.name,
          text: esOperations.map(({
            text
          }) => text).join(', '),
          type: _i18n.i18n.translate('xpack.lens.indexPattern.formulaMetricValue', {
            defaultMessage: 'metric'
          })
        }
      }, getNodeLocation(node)));
    } else {
      errors.push(getMessageFromId({
        id: 'tooManyFirstArguments',
        meta: {
          operation: node.name,
          type,
          supported: requiredFunctions,
          text: esOperations.map(({
            text
          }) => text).join(', ')
        }
      }, getNodeLocation(node)));
    }
  }
  // full reference operation have another way to handle math operations
  if (isFieldOperation && (!firstArgValidation && mathOperations.length || mathOperations.length > 1)) {
    errors.push(getMessageFromId({
      id: 'wrongArgument',
      meta: {
        operation: node.name,
        type,
        text: mathOperations.map(({
          text
        }) => text).join(', ')
      }
    }, getNodeLocation(node)));
  }
  return errors;
}