"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCurrentUrl = exports.createKbnUrlControls = void 0;
exports.getRelativeToHistoryPath = getRelativeToHistoryPath;
exports.getStateFromKbnUrl = getStateFromKbnUrl;
exports.getStatesFromKbnUrl = getStatesFromKbnUrl;
exports.setStateToKbnUrl = setStateToKbnUrl;
var _url = require("url");
var _queryString = require("query-string");
var _history = require("history");
var _parse = require("../../../common/state_management/parse");
var _state_encoder = require("../state_encoder");
var _common = require("../../../common");
var _set_state_to_kbn_url = require("../../../common/state_management/set_state_to_kbn_url");
var _state_hash = require("../state_hash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getCurrentUrl = history => history.createHref(history.location);

/**
 * Parses a kibana url and retrieves all the states encoded into the URL,
 * Handles both expanded rison state and hashed state (where the actual state stored in sessionStorage)
 * e.g.:
 *
 * given an url:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a=(tab:indexedFields)&_b=(f:test,i:'',l:'')
 * will return object:
 * {_a: {tab: 'indexedFields'}, _b: {f: 'test', i: '', l: ''}};
 *
 *
 * By default due to Kibana legacy reasons assumed that state is stored in a query inside a hash part of the URL:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a={STATE}
 *
 * { getFromHashQuery: false } option should be used in case state is stored in a main query (not in a hash):
 * http://localhost:5601/oxf/app/kibana?_a={STATE}#/yourApp
 *
 */
exports.getCurrentUrl = getCurrentUrl;
function getStatesFromKbnUrl(url = window.location.href, keys, {
  getFromHashQuery = true
} = {
  getFromHashQuery: true
}) {
  var _parseUrlHash;
  const query = getFromHashQuery ? (_parseUrlHash = (0, _parse.parseUrlHash)(url)) === null || _parseUrlHash === void 0 ? void 0 : _parseUrlHash.query : (0, _parse.parseUrl)(url).query;
  if (!query) return {};
  const decoded = {};
  Object.entries(query).filter(([key]) => keys ? keys.includes(key) : true).forEach(([q, value]) => {
    decoded[q] = (0, _state_encoder.decodeState)(value);
  });
  return decoded;
}

/**
 * Retrieves specific state from url by key
 * e.g.:
 *
 * given an url:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a=(tab:indexedFields)&_b=(f:test,i:'',l:'')
 * and key '_a'
 * will return object:
 * {tab: 'indexedFields'}
 *
 *
 * By default due to Kibana legacy reasons assumed that state is stored in a query inside a hash part of the URL:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a={STATE}
 *
 * { getFromHashQuery: false } option should be used in case state is stored in a main query (not in a hash):
 * http://localhost:5601/oxf/app/kibana?_a={STATE}#/yourApp
 */
function getStateFromKbnUrl(key, url = window.location.href, {
  getFromHashQuery = true
} = {
  getFromHashQuery: true
}) {
  return getStatesFromKbnUrl(url, [key], {
    getFromHashQuery
  })[key] || null;
}

/**
 * Sets state to the url by key and returns a new url string.
 * Doesn't actually updates history
 *
 * e.g.:
 * given a url: http://localhost:5601/oxf/app/kibana#/yourApp?_a=(tab:indexedFields)&_b=(f:test,i:'',l:'')
 * key: '_a'
 * and state: {tab: 'other'}
 *
 * will return url:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a=(tab:other)&_b=(f:test,i:'',l:'')
 *
 * By default due to Kibana legacy reasons assumed that state is stored in a query inside a hash part of the URL:
 * http://localhost:5601/oxf/app/kibana#/yourApp?_a={STATE}
 *
 * { storeInHashQuery: false } option should be used in you want to store your state in a main query (not in a hash):
 * http://localhost:5601/oxf/app/kibana?_a={STATE}#/yourApp
 */
function setStateToKbnUrl(key, state, {
  useHash = false,
  storeInHashQuery = true
} = {
  useHash: false,
  storeInHashQuery: true
}, rawUrl = window.location.href) {
  return internalSetStateToKbnUrl(key, state, {
    useHash,
    storeInHashQuery
  }, rawUrl);
}
const internalSetStateToKbnUrl = (0, _set_state_to_kbn_url.createSetStateToKbnUrl)(_state_hash.persistState);

/**
 * A tiny wrapper around history library to listen for url changes and update url
 * History library handles a bunch of cross browser edge cases
 */

const createKbnUrlControls = (history = (0, _history.createBrowserHistory)()) => {
  const updateQueue = [];

  // if we should replace or push with next async update,
  // if any call in a queue asked to push, then we should push
  let shouldReplace = true;
  function updateUrl(newUrl, replace = false) {
    const currentUrl = getCurrentUrl(history);
    if (newUrl === currentUrl) return undefined; // skip update

    const historyPath = getRelativeToHistoryPath(newUrl, history);
    if (replace) {
      history.replace(historyPath);
    } else {
      history.push(historyPath);
    }
    return getCurrentUrl(history);
  }

  // queue clean up
  function cleanUp() {
    updateQueue.splice(0, updateQueue.length);
    shouldReplace = true;
  }

  // runs scheduled url updates
  function flush(replace = shouldReplace) {
    if (updateQueue.length === 0) {
      return;
    }
    const nextUrl = getPendingUrl();
    cleanUp();
    const newUrl = updateUrl(nextUrl, replace);
    return newUrl;
  }
  function getPendingUrl() {
    const resultUrl = updateQueue.reduce((url, nextUpdate) => {
      var _nextUpdate;
      return (_nextUpdate = nextUpdate(url)) !== null && _nextUpdate !== void 0 ? _nextUpdate : url;
    }, getCurrentUrl(history));
    return resultUrl;
  }
  return {
    listen: cb => history.listen(() => {
      cb();
    }),
    update: (newUrl, replace = false) => updateUrl(newUrl, replace),
    updateAsync: (updater, replace = false) => {
      updateQueue.push(updater);
      if (shouldReplace) {
        shouldReplace = replace;
      }

      // Schedule url update to the next microtask
      // this allows to batch synchronous url changes
      return Promise.resolve().then(() => {
        return flush();
      });
    },
    flush: replace => {
      return flush(replace);
    },
    cancel: () => {
      cleanUp();
    },
    getPendingUrl: () => {
      return getPendingUrl();
    }
  };
};

/**
 * Depending on history configuration extracts relative path for history updates
 * 4 possible cases (see tests):
 * 1. Browser history with empty base path
 * 2. Browser history with base path
 * 3. Hash history with empty base path
 * 4. Hash history with base path
 */
exports.createKbnUrlControls = createKbnUrlControls;
function getRelativeToHistoryPath(absoluteUrl, history) {
  var _parsedUrl$pathname;
  function stripBasename(path) {
    if (path === null) path = '';
    const stripLeadingHash = _ => _.charAt(0) === '#' ? _.substr(1) : _;
    const stripTrailingSlash = _ => _.charAt(_.length - 1) === '/' ? _.substr(0, _.length - 1) : _;
    const baseName = stripLeadingHash(stripTrailingSlash(history.createHref({})));
    return path.startsWith(baseName) ? path.substr(baseName.length) : path;
  }
  const isHashHistory = history.createHref({}).includes('#');
  const parsedUrl = isHashHistory ? (0, _parse.parseUrlHash)(absoluteUrl) : (0, _parse.parseUrl)(absoluteUrl);
  const parsedHash = isHashHistory ? null : (0, _parse.parseUrlHash)(absoluteUrl);
  return (0, _url.format)({
    pathname: stripBasename((_parsedUrl$pathname = parsedUrl.pathname) !== null && _parsedUrl$pathname !== void 0 ? _parsedUrl$pathname : null),
    // @ts-expect-error `urlUtils.encodeQuery` expects key/value pairs with values of type `string | string[] | null`,
    // however `@types/node` says that `url.query` has values of type `string | string[] | undefined`.
    // After investigating this, it seems that no matter what the values will be of type `string | string[]`
    search: (0, _queryString.stringify)(_common.url.encodeQuery(parsedUrl.query), {
      sort: false,
      encode: false
    }),
    hash: parsedHash ? (0, _url.format)({
      pathname: parsedHash.pathname,
      // @ts-expect-error `urlUtils.encodeQuery` expects key/value pairs with values of type `string | string[] | null`,
      // however `@types/node` says that `url.query` has values of type `string | string[] | undefined`.
      // After investigating this, it seems that no matter what the values will be of type `string | string[]`
      search: (0, _queryString.stringify)(_common.url.encodeQuery(parsedHash.query), {
        sort: false,
        encode: false
      })
    }) : parsedUrl.hash
  });
}