"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ChangePrivilegeActionRunner = void 0;
exports.bulkChangePrivilegeAgentsBatch = bulkChangePrivilegeAgentsBatch;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _package_policy = require("../package_policy");
var _errors = require("../../errors");
var _constants = require("../../constants");
var _services = require("../../../common/services");
var _action_runner = require("./action_runner");
var _bulk_action_types = require("./bulk_action_types");
var _actions = require("./actions");
var _change_privilege_level = require("./change_privilege_level");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ChangePrivilegeActionRunner extends _action_runner.ActionRunner {
  async processAgents(agents) {
    return await bulkChangePrivilegeAgentsBatch(this.esClient, this.soClient, agents, this.actionParams);
  }
  getTaskType() {
    return _bulk_action_types.BulkActionTaskType.PRIVILEGE_LEVEL_CHANGE_RETRY;
  }
  getActionType() {
    return 'PRIVILEGE_LEVEL_CHANGE';
  }
}
exports.ChangePrivilegeActionRunner = ChangePrivilegeActionRunner;
async function bulkChangePrivilegeAgentsBatch(esClient, soClient, agents, options) {
  var _options$actionId, _options$total, _options$user_info;
  const errors = {};
  const now = new Date().toISOString();
  const agentsToAction = [];
  const actionId = (_options$actionId = options.actionId) !== null && _options$actionId !== void 0 ? _options$actionId : (0, _uuid.v4)();
  const total = (_options$total = options.total) !== null && _options$total !== void 0 ? _options$total : agents.length;
  const spaceId = options.spaceId;
  const namespaces = spaceId ? [spaceId] : [];
  await (0, _pMap.default)(agents, async agent => {
    // Create error if agent is on an unsupported version.
    if (!(0, _services.isAgentPrivilegeLevelChangeSupported)(agent)) {
      errors[agent.id] = new _errors.FleetUnauthorizedError(`Cannot remove root privilege. Privilege level change is supported from version ${_services.MINIMUM_PRIVILEGE_LEVEL_CHANGE_AGENT_VERSION}.`);
      return;
    }
    if (agent !== null && agent !== void 0 && agent.policy_id) {
      const allPackagePolicies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, agent.policy_id)) || [];
      const packagesWithRootPrivilege = (0, _change_privilege_level.getPackagesWithRootPrivilege)(allPackagePolicies);
      // Create error if agent contains an integration that requires root privilege.
      if (packagesWithRootPrivilege.length > 0) {
        errors[agent.id] = new _errors.FleetUnauthorizedError(`Agent ${agent.id} is on policy ${agent.policy_id}, which contains integrations that require root privilege: ${packagesWithRootPrivilege.map(pkg => pkg === null || pkg === void 0 ? void 0 : pkg.name).join(', ')}`);
      } else {
        agentsToAction.push(agent);
      }
    } else {
      agentsToAction.push(agent);
    }
  }, {
    concurrency: _constants.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
  });
  const agentIds = agentsToAction.map(agent => agent.id);

  // Create action to change the privilege level of eligible agents.
  // Extract password from options if provided and pass it as a secret.
  await (0, _actions.createAgentAction)(esClient, soClient, {
    id: actionId,
    agents: agentIds,
    created_at: now,
    type: 'PRIVILEGE_LEVEL_CHANGE',
    total,
    data: {
      unprivileged: true,
      ...((options === null || options === void 0 ? void 0 : options.user_info) && {
        user_info: (0, _lodash.omit)(options === null || options === void 0 ? void 0 : options.user_info, ['password'])
      })
    },
    ...((options === null || options === void 0 ? void 0 : (_options$user_info = options.user_info) === null || _options$user_info === void 0 ? void 0 : _options$user_info.password) && {
      secrets: {
        user_info: {
          password: options.user_info.password
        }
      },
      namespaces
    })
  });
  await (0, _actions.createErrorActionResults)(esClient, actionId, errors, 'agent does not support privilege change action');
  return {
    actionId
  };
}