"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkChangeAgentsPrivilegeLevel = bulkChangeAgentsPrivilegeLevel;
exports.changeAgentPrivilegeLevel = changeAgentPrivilegeLevel;
exports.getPackagesWithRootPrivilege = getPackagesWithRootPrivilege;
var _lodash = require("lodash");
var _errors = require("../../errors");
var _package_policy = require("../package_policy");
var _services = require("../../../common/services");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _constants = require("../../constants");
var _actions = require("./actions");
var _crud = require("./crud");
var _change_privilege_runner = require("./change_privilege_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function changeAgentPrivilegeLevel(esClient, soClient, agentId, options) {
  var _agent$local_metadata, _agent$local_metadata2, _agent$local_metadata3, _options$user_info;
  const agent = await (0, _crud.getAgentById)(esClient, soClient, agentId);

  // Return fast if agent is already unprivileged.
  if (((_agent$local_metadata = agent.local_metadata) === null || _agent$local_metadata === void 0 ? void 0 : (_agent$local_metadata2 = _agent$local_metadata.elastic) === null || _agent$local_metadata2 === void 0 ? void 0 : (_agent$local_metadata3 = _agent$local_metadata2.agent) === null || _agent$local_metadata3 === void 0 ? void 0 : _agent$local_metadata3.unprivileged) === true) {
    return {
      message: `Agent ${agentId} is already unprivileged`
    };
  }

  // Fail fast if agent is on an unsupported version.
  if (!(0, _services.isAgentPrivilegeLevelChangeSupported)(agent)) {
    throw new _errors.FleetError(`Cannot remove root privilege. Privilege level change is supported from version ${_services.MINIMUM_PRIVILEGE_LEVEL_CHANGE_AGENT_VERSION}.`);
  }

  // Fail fast if agent contains an integration that requires root privilege.
  const packagePolicies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, agent.policy_id || '')) || [];
  const packagesWithRootPrivilege = getPackagesWithRootPrivilege(packagePolicies);
  if (packagesWithRootPrivilege.length > 0) {
    throw new _errors.FleetUnauthorizedError(`Agent ${agent.id} is on policy ${agent.policy_id}, which contains integrations that require root privilege: ${packagesWithRootPrivilege.map(pkg => pkg === null || pkg === void 0 ? void 0 : pkg.name).join(', ')}`);
  }

  // Create action to change the agent's privilege level.
  // Extract password from options if provided and pass it as a secret.
  const res = await (0, _actions.createAgentAction)(esClient, soClient, {
    agents: [agentId],
    created_at: new Date().toISOString(),
    type: 'PRIVILEGE_LEVEL_CHANGE',
    data: {
      unprivileged: true,
      ...((options === null || options === void 0 ? void 0 : options.user_info) && {
        user_info: (0, _lodash.omit)(options === null || options === void 0 ? void 0 : options.user_info, ['password'])
      })
    },
    ...((options === null || options === void 0 ? void 0 : (_options$user_info = options.user_info) === null || _options$user_info === void 0 ? void 0 : _options$user_info.password) && {
      secrets: {
        user_info: {
          password: options.user_info.password
        }
      }
    })
  });
  return {
    actionId: res.id
  };
}
async function bulkChangeAgentsPrivilegeLevel(esClient, soClient, options) {
  var _options$batchSize;
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  if ('agentIds' in options) {
    const givenAgents = await (0, _crud.getAgents)(esClient, soClient, options);
    return await (0, _change_privilege_runner.bulkChangePrivilegeAgentsBatch)(esClient, soClient, givenAgents, {
      ...options,
      spaceId: currentSpaceId
    });
  }
  const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
  const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
    kuery: options.kuery,
    spaceId: currentSpaceId,
    showInactive: false,
    page: 1,
    perPage: batchSize
  });
  if (res.total <= batchSize) {
    return await (0, _change_privilege_runner.bulkChangePrivilegeAgentsBatch)(esClient, soClient, res.agents, {
      ...options,
      spaceId: currentSpaceId
    });
  } else {
    return await new _change_privilege_runner.ChangePrivilegeActionRunner(esClient, soClient, {
      ...options,
      batchSize,
      total: res.total,
      spaceId: currentSpaceId
    }, {
      pitId: await (0, _crud.openPointInTime)(esClient)
    }).runActionAsyncTask();
  }
}
function getPackagesWithRootPrivilege(packagePolicies) {
  return packagePolicies.map(policy => policy.package).filter(pkg => pkg && Boolean(pkg === null || pkg === void 0 ? void 0 : pkg.requires_root));
}