"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deletePackageDatastreamAssetsHandler = void 0;
var _valid = _interopRequireDefault(require("semver/functions/valid"));
var _errors = require("../../errors");
var _services = require("../../services");
var _get = require("../../services/epm/packages/get");
var _input_type_packages = require("../../services/epm/packages/input_type_packages");
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deletePackageDatastreamAssetsHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const logger = _services.appContextService.getLogger();
  const {
    pkgName,
    pkgVersion
  } = request.params;
  const {
    packagePolicyId
  } = request.query;
  try {
    const packageInfo = await (0, _get.getPackageInfo)({
      savedObjectsClient,
      pkgName,
      pkgVersion
    });
    if (pkgVersion && !(0, _valid.default)(pkgVersion)) {
      throw new _errors.PackagePolicyRequestError('Package version is not a valid semver');
    }
    if (!packageInfo || packageInfo.version !== pkgVersion) {
      throw new _errors.FleetNotFoundError('Version is not installed');
    }
    if ((packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.type) !== 'input') {
      throw new _errors.PackagePolicyRequestError(`Requested package ${pkgName}-${pkgVersion} is not an input package`);
    }
    const allSpacesSoClient = _services.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const {
      items: allPackagePolicies
    } = await _services.packagePolicyService.list(allSpacesSoClient, {
      kuery: `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`,
      spaceId: '*'
    });
    const packagePolicy = allPackagePolicies.find(policy => policy.id === packagePolicyId);
    if (!packagePolicy) {
      throw new _errors.FleetNotFoundError(`Package policy with id ${packagePolicyId} not found`);
    }
    const datasetName = (0, _input_type_packages.getDatasetName)(packagePolicy === null || packagePolicy === void 0 ? void 0 : packagePolicy.inputs);
    const datasetNameUsedByMultiplePolicies = (0, _input_type_packages.isInputPackageDatasetUsedByMultiplePolicies)(allPackagePolicies, datasetName, pkgName);
    if (datasetNameUsedByMultiplePolicies) {
      throw new _errors.FleetError(`Datastreams matching ${datasetName} are in use by other package policies and cannot be removed`);
    }
    const {
      existingDataStreams
    } = await (0, _input_type_packages.findDataStreamsFromDifferentPackages)(datasetName, packageInfo, esClient);
    const existingDataStreamsAreFromDifferentPackage = (0, _input_type_packages.checkExistingDataStreamsAreFromDifferentPackage)(packageInfo, existingDataStreams);
    if (existingDataStreamsAreFromDifferentPackage) {
      throw new _errors.FleetError(`Datastreams matching ${datasetName} exist on other packages and cannot be removed`);
    }
    logger.info(`Removing datastreams matching ${datasetName}`);
    await (0, _input_type_packages.removeAssetsForInputPackagePolicy)({
      packageInfo,
      logger,
      datasetName,
      esClient,
      savedObjectsClient
    });
    return response.ok({
      body: {
        success: true
      }
    });
  } catch (error) {
    logger.error(`error ${error.message}`);
    throw error;
  }
};
exports.deletePackageDatastreamAssetsHandler = deletePackageDatastreamAssetsHandler;