"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AnalyticsClient = void 0;
var tslib_1 = require("tslib");
var rxjs_1 = require("rxjs");
var rxjs_2 = require("rxjs");
var shippers_registry_1 = require("./shippers_registry");
var opt_in_config_1 = require("./opt_in_config");
var context_service_1 = require("./context_service");
var validation_1 = require("../schema/validation");
var AnalyticsClient = /** @class */ (function () {
    function AnalyticsClient(initContext) {
        var _this = this;
        this.initContext = initContext;
        this.internalTelemetryCounter$ = new rxjs_1.Subject();
        this.telemetryCounter$ = this.internalTelemetryCounter$.pipe((0, rxjs_2.share)()); // Using `share` so we can have multiple subscribers
        /**
         * This queue holds all the events until both conditions occur:
         * 1. We know the user's optIn decision.
         * 2. We have, at least, one registered shipper.
         * @private
         */
        this.internalEventQueue$ = new rxjs_1.Subject();
        this.shippersRegistry = new shippers_registry_1.ShippersRegistry();
        /**
         * Observable used to report when a shipper is registered.
         * @private
         */
        this.shipperRegistered$ = new rxjs_1.Subject();
        this.eventTypeRegistry = new Map();
        this.context$ = new rxjs_1.BehaviorSubject({});
        this.optInConfig$ = new rxjs_1.BehaviorSubject(undefined);
        this.optInConfigWithReplay$ = this.optInConfig$.pipe((0, rxjs_2.filter)(function (optInConfig) { return typeof optInConfig !== 'undefined'; }), (0, rxjs_2.shareReplay)(1));
        this.contextWithReplay$ = this.context$.pipe((0, rxjs_2.skipWhile)(function () { return !_this.optInConfig$.value; }), // Do not forward the context events until we have an optInConfig value
        (0, rxjs_2.shareReplay)(1));
        this.reportEvent = function (eventType, eventData) {
            var _a, _b;
            // Fetch the timestamp as soon as we receive the event.
            var timestamp = new Date().toISOString();
            _this.internalTelemetryCounter$.next({
                type: 'enqueued',
                source: 'client',
                event_type: eventType,
                code: 'enqueued',
                count: 1,
            });
            var eventTypeOpts = _this.eventTypeRegistry.get(eventType);
            if (!eventTypeOpts) {
                _this.internalTelemetryCounter$.next({
                    type: 'dropped',
                    source: 'client',
                    event_type: eventType,
                    code: 'UnregisteredType',
                    count: 1,
                });
                throw new Error("Attempted to report event type \"".concat(eventType, "\", before registering it. Use the \"registerEventType\" API to register it."));
            }
            // If the validator is registered (dev-mode only), perform the validation.
            if (eventTypeOpts.validator) {
                (0, validation_1.validateSchema)("Event Type '".concat(eventType, "'"), eventTypeOpts.validator, eventData);
            }
            var event = {
                timestamp: timestamp,
                event_type: eventType,
                context: _this.context$.value,
                properties: eventData,
                trace: (_b = (_a = _this.initContext) === null || _a === void 0 ? void 0 : _a.getTraceContext) === null || _b === void 0 ? void 0 : _b.call(_a),
            };
            _this.initContext.logger.debug("Report event \"".concat(eventType, "\""), {
                ebt_event: event,
            });
            var optInConfig = _this.optInConfig$.value;
            if ((optInConfig === null || optInConfig === void 0 ? void 0 : optInConfig.isEventTypeOptedIn(eventType)) === false) {
                // If opted out, skip early
                return;
            }
            if (typeof optInConfig === 'undefined') {
                // If the opt-in config is not provided yet, we need to enqueue the event to an internal queue
                _this.internalEventQueue$.next(event);
            }
            else {
                _this.sendToShipper(eventType, [event]);
            }
        };
        this.registerEventType = function (eventTypeOps) {
            if (_this.eventTypeRegistry.get(eventTypeOps.eventType)) {
                throw new Error("Event Type \"".concat(eventTypeOps.eventType, "\" is already registered."));
            }
            _this.eventTypeRegistry.set(eventTypeOps.eventType, tslib_1.__assign(tslib_1.__assign({}, eventTypeOps), { validator: _this.initContext.isDev ? (0, validation_1.schemaToIoTs)(eventTypeOps.schema) : undefined }));
        };
        this.optIn = function (optInConfig) {
            var optInConfigInstance = new opt_in_config_1.OptInConfigService(optInConfig);
            _this.optInConfig$.next(optInConfigInstance);
        };
        this.registerContextProvider = function (contextProviderOpts) {
            _this.contextService.registerContextProvider(contextProviderOpts);
        };
        this.removeContextProvider = function (name) {
            _this.contextService.removeContextProvider(name);
        };
        this.registerShipper = function (ShipperClass, shipperConfig, _a) {
            var _b;
            var _c = _a === void 0 ? {} : _a, _d = _c.exclusiveEventTypes, exclusiveEventTypes = _d === void 0 ? [] : _d;
            var shipperName = ShipperClass.shipperName;
            var shipper = new ShipperClass(shipperConfig, tslib_1.__assign(tslib_1.__assign({}, _this.initContext), { logger: _this.initContext.logger.get('shipper', shipperName) }));
            if (exclusiveEventTypes.length) {
                // This feature is not intended to be supported in the MVP.
                // I can remove it if we think it causes more bad than good.
                exclusiveEventTypes.forEach(function (eventType) {
                    _this.shippersRegistry.addEventExclusiveShipper(eventType, shipperName, shipper);
                });
            }
            else {
                _this.shippersRegistry.addGlobalShipper(shipperName, shipper);
            }
            // Subscribe to the shipper's telemetryCounter$ and pass it over to the client's-level observable
            (_b = shipper.telemetryCounter$) === null || _b === void 0 ? void 0 : _b.subscribe(function (counter) {
                return _this.internalTelemetryCounter$.next(tslib_1.__assign(tslib_1.__assign({}, counter), { source: shipperName }));
            });
            // Spread the optIn configuration updates
            _this.optInConfigWithReplay$.subscribe(function (optInConfig) {
                var isOptedIn = optInConfig.isShipperOptedIn(shipperName);
                try {
                    shipper.optIn(isOptedIn);
                }
                catch (err) {
                    _this.initContext.logger.warn("Failed to set isOptedIn:".concat(isOptedIn, " in shipper ").concat(shipperName), err);
                }
            });
            // Spread the global context if it has custom extendContext method
            if (shipper.extendContext) {
                _this.contextWithReplay$.subscribe(function (context) {
                    try {
                        shipper.extendContext(context);
                    }
                    catch (err) {
                        _this.initContext.logger.warn("Shipper \"".concat(shipperName, "\" failed to extend the context"), err);
                    }
                });
            }
            // Notify that a shipper is registered
            _this.shipperRegistered$.next();
        };
        this.flush = function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, Promise.all(tslib_1.__spreadArray([], tslib_1.__read(this.shippersRegistry.allShippers.entries()), false).map(function (_a) { return tslib_1.__awaiter(_this, [_a], void 0, function (_b) {
                            var err_1;
                            var _c = tslib_1.__read(_b, 2), shipperName = _c[0], shipper = _c[1];
                            return tslib_1.__generator(this, function (_d) {
                                switch (_d.label) {
                                    case 0:
                                        _d.trys.push([0, 2, , 3]);
                                        return [4 /*yield*/, shipper.flush()];
                                    case 1:
                                        _d.sent();
                                        return [3 /*break*/, 3];
                                    case 2:
                                        err_1 = _d.sent();
                                        this.initContext.logger.warn("Failed to flush shipper \"".concat(shipperName, "\""), err_1);
                                        return [3 /*break*/, 3];
                                    case 3: return [2 /*return*/];
                                }
                            });
                        }); }))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        }); };
        this.shutdown = function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.flush()];
                    case 1:
                        _a.sent();
                        this.shippersRegistry.allShippers.forEach(function (shipper, shipperName) {
                            try {
                                shipper.shutdown();
                            }
                            catch (err) {
                                _this.initContext.logger.warn("Failed to shutdown shipper \"".concat(shipperName, "\""), err);
                            }
                        });
                        this.internalEventQueue$.complete();
                        this.internalTelemetryCounter$.complete();
                        this.shipperRegistered$.complete();
                        this.optInConfig$.complete();
                        this.context$.complete();
                        return [2 /*return*/];
                }
            });
        }); };
        this.contextService = new context_service_1.ContextService(this.context$, this.initContext.isDev, this.initContext.logger.get('context-service'));
        this.reportEnqueuedEventsWhenClientIsReady();
    }
    /**
     * Forwards the `events` to the registered shippers, bearing in mind if the shipper is opted-in for that eventType.
     * @param eventType The event type's name
     * @param events A bulk array of events matching the eventType.
     * @private
     */
    AnalyticsClient.prototype.sendToShipper = function (eventType, events) {
        var _this = this;
        var sentToShipper = false;
        this.shippersRegistry.getShippersForEventType(eventType).forEach(function (shipper, shipperName) {
            var _a;
            var isShipperOptedIn = (_a = _this.optInConfig$.value) === null || _a === void 0 ? void 0 : _a.isShipperOptedIn(shipperName, eventType);
            // Only send it to the non-explicitly opted-out shippers
            if (isShipperOptedIn) {
                sentToShipper = true;
                try {
                    shipper.reportEvents(events);
                }
                catch (err) {
                    _this.initContext.logger.warn("Failed to report event \"".concat(eventType, "\" via shipper \"").concat(shipperName, "\""), err);
                }
            }
        });
        if (sentToShipper) {
            this.internalTelemetryCounter$.next({
                type: 'sent_to_shipper',
                source: 'client',
                event_type: eventType,
                code: 'OK',
                count: events.length,
            });
        }
    };
    /**
     * Once the client is ready (it has a valid optInConfig and at least one shipper),
     * flush any early events and ship them or discard them based on the optInConfig.
     * @private
     */
    AnalyticsClient.prototype.reportEnqueuedEventsWhenClientIsReady = function () {
        var _this = this;
        // Observer that will emit when both events occur: the OptInConfig is set + a shipper has been registered
        var configReceivedAndShipperReceivedObserver$ = (0, rxjs_1.combineLatest)([
            this.optInConfigWithReplay$,
            (0, rxjs_1.merge)([
                this.shipperRegistered$,
                // Merging shipperRegistered$ with the optInConfigWithReplay$ when optedIn is false, so that we don't need to wait for the shipper if opted-in === false
                this.optInConfigWithReplay$.pipe((0, rxjs_2.filter)(function (cfg) { return (cfg === null || cfg === void 0 ? void 0 : cfg.isOptedIn()) === false; })),
            ]),
        ]);
        // Flush the internal queue when we get any optInConfig and, at least, 1 shipper
        this.internalEventQueue$
            .pipe(
        // Take until will close the observer once we reach the condition below
        (0, rxjs_2.takeUntil)(configReceivedAndShipperReceivedObserver$), 
        // Accumulate the events until we can send them
        (0, rxjs_2.buffer)(configReceivedAndShipperReceivedObserver$), 
        // Minimal delay only to make this chain async and let the optIn operation to complete first.
        (0, rxjs_2.delay)(0), 
        // Re-emit the context to make sure all the shippers got it (only if opted-in)
        (0, rxjs_2.tap)(function () {
            var _a;
            if ((_a = _this.optInConfig$.value) === null || _a === void 0 ? void 0 : _a.isOptedIn()) {
                _this.context$.next(_this.context$.value);
            }
        }), 
        // Minimal delay only to make this chain async and let
        // the context update operation to complete first.
        (0, rxjs_2.delay)(0), 
        // Flatten the array of events
        (0, rxjs_2.concatMap)(function (events) { return (0, rxjs_1.from)(events); }), 
        // Discard opted-out events
        (0, rxjs_2.filter)(function (event) { var _a; return ((_a = _this.optInConfig$.value) === null || _a === void 0 ? void 0 : _a.isEventTypeOptedIn(event.event_type)) === true; }), 
        // Let's group the requests per eventType for easier batching
        (0, rxjs_2.groupBy)(function (event) { return event.event_type; }), (0, rxjs_2.mergeMap)(function (groupedObservable) {
            return groupedObservable.pipe((0, rxjs_2.bufferCount)(1000), // Batching up-to 1000 events per event type for backpressure reasons
            (0, rxjs_2.map)(function (events) { return ({ eventType: groupedObservable.key, events: events }); }));
        }))
            .subscribe(function (_a) {
            var eventType = _a.eventType, events = _a.events;
            _this.sendToShipper(eventType, events);
        });
    };
    return AnalyticsClient;
}());
exports.AnalyticsClient = AnalyticsClient;
