"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEndpointConsoleCommands = exports.HELP_GROUPS = void 0;
var _i18n = require("@kbn/i18n");
var _cancel_action = require("../command_render_components/cancel_action");
var _is_response_action_supported = require("../../../../../common/endpoint/service/response_actions/is_response_action_supported");
var _cancel_authz_utils = require("../../../../../common/endpoint/service/authz/cancel_authz_utils");
var _custom_script_selector = require("../../console_argument_selectors/custom_scripts_selector/custom_script_selector");
var _pending_actions_selector = require("../../console_argument_selectors/pending_actions_selector/pending_actions_selector");
var _run_script_action = require("../command_render_components/run_script_action");
var _endpoint = require("../../../../common/lib/endpoint");
var _utils = require("../../../../../common/endpoint/service/response_actions/utils");
var _upload_action = require("../command_render_components/upload_action");
var _console_argument_selectors = require("../../console_argument_selectors");
var _experimental_features_service = require("../../../../common/experimental_features_service");
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _get_file_action = require("../command_render_components/get_file_action");
var _isolate_action = require("../command_render_components/isolate_action");
var _release_action = require("../command_render_components/release_action");
var _kill_process_action = require("../command_render_components/kill_process_action");
var _suspend_process_action = require("../command_render_components/suspend_process_action");
var _status_action = require("../command_render_components/status_action");
var _get_processes_action = require("../command_render_components/get_processes_action");
var _execute_action = require("../command_render_components/execute_action");
var _translations = require("../../../../common/translations");
var _get_command_about_info = require("./get_command_about_info");
var _utils2 = require("./utils");
var _translations2 = require("../../../common/translations");
var _scan_action = require("../command_render_components/scan_action");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const emptyArgumentValidator = argData => {
  var _argData$;
  if ((argData === null || argData === void 0 ? void 0 : argData.length) > 0 && typeof argData[0] === 'string' && ((_argData$ = argData[0]) === null || _argData$ === void 0 ? void 0 : _argData$.trim().length) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.emptyArgumentMessage', {
      defaultMessage: 'Argument cannot be empty'
    });
  }
};
const pidValidator = argData => {
  const emptyResult = emptyArgumentValidator(argData);
  if (emptyResult !== true) {
    return emptyResult;
  } else if (Number.isSafeInteger(Number(argData)) && Number(argData) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidPidMessage', {
      defaultMessage: 'Argument must be a positive number representing the PID of a process'
    });
  }
};
const executeTimeoutValidator = argData => {
  if (String(argData).trim().length && (0, _utils2.validateUnitOfTime)(String(argData).trim())) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidExecuteTimeout', {
      defaultMessage: 'Argument must be a string with a positive integer value followed by a unit of time (h for hours, m for minutes, s for seconds). Example: 37m.'
    });
  }
};
const capabilitiesAndPrivilegesValidator = agentType => {
  return command => {
    const privileges = command.commandDefinition.meta.privileges;
    const agentCapabilities = command.commandDefinition.meta.capabilities;
    const commandName = command.commandDefinition.name;
    const responderCapability = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_ENDPOINT_CAPABILITY[commandName];
    let errorMessage = '';

    // We only validate Agent capabilities for the command for Endpoint agents
    if (agentType === 'endpoint') {
      if (!responderCapability) {
        errorMessage = errorMessage.concat((0, _translations.UPGRADE_AGENT_FOR_RESPONDER)(agentType, commandName));
      }
      if (responderCapability && !agentCapabilities.includes(responderCapability)) {
        errorMessage = errorMessage.concat((0, _translations.UPGRADE_AGENT_FOR_RESPONDER)(agentType, commandName));
      }
    }
    if (!(0, _utils.getRbacControl)({
      commandName,
      privileges
    })) {
      errorMessage = errorMessage.concat(_translations.INSUFFICIENT_PRIVILEGES_FOR_COMMAND);
    }
    if (errorMessage.length) {
      return errorMessage;
    }
    return true;
  };
};
const HELP_GROUPS = exports.HELP_GROUPS = Object.freeze({
  responseActions: {
    position: 0,
    label: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.groups.responseActions', {
      defaultMessage: 'Response actions'
    })
  }
});
const ENTER_PID_OR_ENTITY_ID_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterPidOrEntityId', {
  defaultMessage: 'Enter a pid or an entity id to execute'
});
const ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterOrAddOptionalComment', {
  defaultMessage: 'Hit enter to execute or add an optional comment'
});
const COMMENT_ARG_ABOUT = _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.commandArgAbout', {
  defaultMessage: 'A comment to go along with the action'
});
const commandCommentArgument = () => {
  return {
    comment: {
      required: false,
      allowMultiples: false,
      about: COMMENT_ARG_ABOUT
    }
  };
};
const getEndpointConsoleCommands = ({
  endpointAgentId,
  agentType,
  endpointCapabilities,
  endpointPrivileges,
  platform
}) => {
  const featureFlags = _experimental_features_service.ExperimentalFeaturesService.get();
  const {
    crowdstrikeRunScriptEnabled,
    microsoftDefenderEndpointRunScriptEnabled,
    microsoftDefenderEndpointCancelEnabled
  } = featureFlags;
  const commandMeta = {
    agentType,
    platform,
    endpointId: endpointAgentId,
    capabilities: endpointCapabilities,
    privileges: endpointPrivileges
  };
  const doesEndpointSupportCommand = commandName => {
    // Agent capabilities are only validated for Endpoint agent types
    if (agentType !== 'endpoint') {
      return true;
    }
    const responderCapability = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_ENDPOINT_CAPABILITY[commandName];
    if (responderCapability) {
      return endpointCapabilities.includes(responderCapability);
    }
    return false;
  };
  const canCancelForCurrentContext = () => {
    return (0, _cancel_authz_utils.isCancelFeatureAvailable)(endpointPrivileges, featureFlags, agentType);
  };
  const consoleCommands = [{
    name: 'isolate',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.isolate.about,
      isSupported: doesEndpointSupportCommand('isolate')
    }),
    RenderComponent: _isolate_action.IsolateActionResult,
    meta: commandMeta,
    exampleUsage: 'isolate --comment "isolate this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 0,
    helpDisabled: doesEndpointSupportCommand('isolate') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'isolate',
      privileges: endpointPrivileges
    })
  }, {
    name: 'release',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.release.about,
      isSupported: doesEndpointSupportCommand('release')
    }),
    RenderComponent: _release_action.ReleaseActionResult,
    meta: commandMeta,
    exampleUsage: 'release --comment "release this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 1,
    helpDisabled: doesEndpointSupportCommand('release') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'release',
      privileges: endpointPrivileges
    })
  }, {
    name: 'kill-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.killProcess.about,
      isSupported: doesEndpointSupportCommand('kill-process')
    }),
    RenderComponent: _kill_process_action.KillProcessActionResult,
    meta: commandMeta,
    exampleUsage: 'kill-process --pid 123 --comment "kill this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      ...commandCommentArgument(),
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.killProcess.args.pid.about,
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.killProcess.args.entityId.about,
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 4,
    helpDisabled: doesEndpointSupportCommand('kill-process') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'kill-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'suspend-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.suspendProcess.about,
      isSupported: doesEndpointSupportCommand('suspend-process')
    }),
    RenderComponent: _suspend_process_action.SuspendProcessActionResult,
    meta: commandMeta,
    exampleUsage: 'suspend-process --pid 123 --comment "suspend this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      ...commandCommentArgument(),
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.suspendProcess.args.pid.about,
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.suspendProcess.args.entityId.about,
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 5,
    helpDisabled: doesEndpointSupportCommand('suspend-process') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'suspend-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'status',
    about: _translations2.CONSOLE_COMMANDS.status.about,
    RenderComponent: _status_action.EndpointStatusActionResult,
    meta: commandMeta,
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 2
  }, {
    name: 'processes',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.processes.about,
      isSupported: doesEndpointSupportCommand('processes')
    }),
    RenderComponent: _get_processes_action.GetProcessesActionResult,
    meta: commandMeta,
    exampleUsage: 'processes --comment "get the processes"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 3,
    helpDisabled: doesEndpointSupportCommand('processes') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'processes',
      privileges: endpointPrivileges
    })
  }, {
    name: 'get-file',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.getFile.about,
      isSupported: doesEndpointSupportCommand('get-file')
    }),
    RenderComponent: _get_file_action.GetFileActionResult,
    meta: commandMeta,
    exampleUsage: 'get-file --path "/full/path/to/file.txt" --comment "Possible malware"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      path: {
        required: true,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.getFile.args.path.about,
        validate: argData => {
          return emptyArgumentValidator(argData);
        }
      },
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 6,
    helpDisabled: !doesEndpointSupportCommand('get-file'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'get-file',
      privileges: endpointPrivileges
    })
  }, {
    name: 'execute',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.execute.about,
      isSupported: doesEndpointSupportCommand('execute')
    }),
    RenderComponent: _execute_action.ExecuteActionResult,
    meta: commandMeta,
    exampleUsage: 'execute --command "ls -al" --timeout 2s --comment "Get list of all files"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      command: {
        required: true,
        allowMultiples: false,
        about: (0, _execute_action.getExecuteCommandArgAboutInfo)(),
        mustHaveValue: 'non-empty-string'
      },
      timeout: {
        required: false,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.execute.args.timeout.about,
        mustHaveValue: 'non-empty-string',
        validate: executeTimeoutValidator
      },
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 6,
    helpDisabled: !doesEndpointSupportCommand('execute'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'execute',
      privileges: endpointPrivileges
    })
  }, {
    name: 'scan',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.scan.about,
      isSupported: doesEndpointSupportCommand('scan')
    }),
    RenderComponent: _scan_action.ScanActionResult,
    meta: commandMeta,
    exampleUsage: 'scan --path "/full/path/to/folder" --comment "Scan folder for malware"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      path: {
        required: true,
        allowMultiples: false,
        mustHaveValue: 'non-empty-string',
        about: _translations2.CONSOLE_COMMANDS.scan.args.path.about
      },
      ...commandCommentArgument()
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 8,
    helpDisabled: !doesEndpointSupportCommand('scan'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'scan',
      privileges: endpointPrivileges
    })
  }, {
    name: 'runscript',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.runscript.about,
      isSupported: doesEndpointSupportCommand('runscript')
    }),
    RenderComponent: _run_script_action.RunScriptActionResult,
    meta: commandMeta,
    exampleInstruction: _translations2.CONSOLE_COMMANDS.runscript.about,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: false,
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 9,
    helpDisabled: !(0, _is_response_action_supported.isActionSupportedByAgentType)(agentType, 'runscript', 'manual'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'runscript',
      privileges: endpointPrivileges
    }) || agentType === 'endpoint' && !doesEndpointSupportCommand('runscript')
  }];

  // `upload` command
  consoleCommands.push({
    name: 'upload',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.upload.about,
      isSupported: doesEndpointSupportCommand('upload')
    }),
    RenderComponent: _upload_action.UploadActionResult,
    meta: commandMeta,
    exampleUsage: 'upload --file --overwrite --comment "script to fix registry"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      file: {
        required: true,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.upload.args.file.about,
        mustHaveValue: 'truthy',
        SelectorComponent: _console_argument_selectors.ArgumentFileSelector
      },
      overwrite: {
        required: false,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.upload.args.overwrite.about,
        mustHaveValue: false
      },
      comment: {
        required: false,
        allowMultiples: false,
        mustHaveValue: 'non-empty-string',
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 7,
    helpDisabled: !doesEndpointSupportCommand('upload'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'upload',
      privileges: endpointPrivileges
    })
  });
  if (microsoftDefenderEndpointCancelEnabled) {
    const isSupported = canCancelForCurrentContext();
    consoleCommands.push({
      name: 'cancel',
      about: (0, _get_command_about_info.getCommandAboutInfo)({
        aboutInfo: _translations2.CONSOLE_COMMANDS.cancel.about,
        isSupported
      }),
      RenderComponent: _cancel_action.CancelActionResult,
      meta: commandMeta,
      exampleUsage: 'cancel --action="action-123-456-789"',
      exampleInstruction: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.cancel.exampleInstruction', {
        defaultMessage: 'Select a pending action to cancel'
      }),
      mustHaveArgs: true,
      args: {
        ...(isSupported ? {
          action: {
            required: true,
            allowMultiples: false,
            about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.cancel.action.about', {
              defaultMessage: 'The response action to cancel'
            }),
            mustHaveValue: 'truthy',
            selectorShowTextValue: true,
            SelectorComponent: _pending_actions_selector.PendingActionsSelector
          }
        } : {}),
        comment: {
          required: false,
          allowMultiples: false,
          mustHaveValue: 'non-empty-string',
          about: COMMENT_ARG_ABOUT
        }
      },
      helpGroupLabel: HELP_GROUPS.responseActions.label,
      helpGroupPosition: HELP_GROUPS.responseActions.position,
      helpCommandPosition: 10,
      helpDisabled: !isSupported,
      helpHidden: !isSupported,
      validate: capabilitiesAndPrivilegesValidator(agentType)
    });
  }
  switch (agentType) {
    case 'sentinel_one':
      return adjustCommandsForSentinelOne({
        commandList: consoleCommands,
        platform
      });
    case 'crowdstrike':
      return adjustCommandsForCrowdstrike({
        commandList: consoleCommands,
        crowdstrikeRunScriptEnabled
      });
    case 'microsoft_defender_endpoint':
      return adjustCommandsForMicrosoftDefenderEndpoint({
        commandList: consoleCommands,
        microsoftDefenderEndpointRunScriptEnabled
      });
    default:
      // agentType === endpoint: just returns the defined command list
      return consoleCommands;
  }
};

/** @internal */
exports.getEndpointConsoleCommands = getEndpointConsoleCommands;
const disableCommand = (command, agentType) => {
  command.helpDisabled = true;
  command.helpHidden = true;
  command.validate = () => (0, _translations.UPGRADE_AGENT_FOR_RESPONDER)(agentType, command.name);
};

/** @internal */
const adjustCommandsForSentinelOne = ({
  commandList,
  platform
}) => {
  const featureFlags = _experimental_features_service.ExperimentalFeaturesService.get();
  const isRunscriptEnabled = featureFlags.responseActionsSentinelOneRunScriptEnabled;
  return commandList.map(command => {
    // Kill-Process: adjust command to accept only `processName`
    if (command.name === 'kill-process') {
      command.args = {
        ...commandCommentArgument(),
        processName: {
          required: true,
          allowMultiples: false,
          about: _translations2.CONSOLE_COMMANDS.killProcess.args.processName.about,
          mustHaveValue: 'non-empty-string'
        }
      };
      command.exampleUsage = 'kill-process --processName="notepad" --comment="kill malware"';
      command.exampleInstruction = _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.killProcess.sentinelOne.instructions', {
        defaultMessage: 'Enter a process name to execute'
      });
    }
    if (command.name === 'status' || !(0, _endpoint.isAgentTypeAndActionSupported)('sentinel_one', _constants.RESPONSE_CONSOLE_COMMAND_TO_API_COMMAND_MAP[command.name], 'manual')) {
      disableCommand(command, 'sentinel_one');
    } else {
      // processes is not currently supported for Windows hosts
      if (command.name === 'processes' && platform.toLowerCase() === 'windows') {
        const message = _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.sentineloneProcessesWindowRestriction', {
          defaultMessage: 'Processes command is not currently supported for SentinelOne hosts running on Windows'
        });
        command.helpDisabled = true;
        command.about = (0, _get_command_about_info.getCommandAboutInfo)({
          aboutInfo: command.about,
          isSupported: false,
          dataTestSubj: 'sentineloneProcessesWindowsWarningTooltip',
          tooltipContent: message
        });
        command.validate = () => {
          return message;
        };
      } else if (command.name === 'runscript' && isRunscriptEnabled) {
        command.helpDisabled = false;
        command.mustHaveArgs = true;
        command.exampleUsage = enteredCommand => {
          let exampleUsageText = `runscript --script="copy.sh" --inputParams="~/logs/log.txt /tmp/log.backup.txt"`;
          if (enteredCommand) {
            var _enteredCommand$argSt, _enteredCommand$argSt2, _scriptArgState$store, _selectedScript$meta;
            const scriptArgState = enteredCommand === null || enteredCommand === void 0 ? void 0 : (_enteredCommand$argSt = enteredCommand.argState) === null || _enteredCommand$argSt === void 0 ? void 0 : (_enteredCommand$argSt2 = _enteredCommand$argSt.script) === null || _enteredCommand$argSt2 === void 0 ? void 0 : _enteredCommand$argSt2.at(0);
            const selectedScript = scriptArgState === null || scriptArgState === void 0 ? void 0 : (_scriptArgState$store = scriptArgState.store) === null || _scriptArgState$store === void 0 ? void 0 : _scriptArgState$store.selectedOption;
            if (selectedScript !== null && selectedScript !== void 0 && (_selectedScript$meta = selectedScript.meta) !== null && _selectedScript$meta !== void 0 && _selectedScript$meta.inputExample) {
              var _selectedScript$meta2;
              exampleUsageText = _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.runscript.sentinelOne.scriptInputExample', {
                defaultMessage: '{scriptName} script input: {example}',
                values: {
                  scriptName: scriptArgState === null || scriptArgState === void 0 ? void 0 : scriptArgState.valueText,
                  example: selectedScript === null || selectedScript === void 0 ? void 0 : (_selectedScript$meta2 = selectedScript.meta) === null || _selectedScript$meta2 === void 0 ? void 0 : _selectedScript$meta2.inputExample
                }
              });
            }
          }
          return exampleUsageText;
        };
        command.args = {
          script: {
            required: true,
            allowMultiples: false,
            about: _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.runscript.sentinelOne.scriptArg', {
              defaultMessage: 'The script to run (selected from popup list)'
            }),
            mustHaveValue: 'non-empty-string',
            SelectorComponent: _custom_script_selector.CustomScriptSelector,
            selectorShowTextValue: true
          },
          inputParams: {
            required: false,
            allowMultiples: false,
            about: _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.runscript.sentinelOne.inputParamsArg', {
              defaultMessage: 'Input arguments for the selected script'
            }),
            mustHaveValue: 'non-empty-string'
          },
          ...commandCommentArgument()
        };
        const priorValidateFn = command.validate;
        command.validate = enteredCommand => {
          var _argState$script, _argState$script$, _argState$script$$sto, _args$args, _args$args$inputParam, _scriptInfo$meta;
          // First do the base validation - like authz checks
          const baseValidation = priorValidateFn ? priorValidateFn(enteredCommand) : true;
          if (baseValidation !== true) {
            return baseValidation;
          }
          const {
            argState,
            args
          } = enteredCommand;

          // No need to validate display of command help `help`
          if (args.hasArg('help')) {
            return true;
          }

          // Validate the script that was selected
          const scriptInfo = argState === null || argState === void 0 ? void 0 : (_argState$script = argState.script) === null || _argState$script === void 0 ? void 0 : (_argState$script$ = _argState$script[0]) === null || _argState$script$ === void 0 ? void 0 : (_argState$script$$sto = _argState$script$.store) === null || _argState$script$$sto === void 0 ? void 0 : _argState$script$$sto.selectedOption;
          const script = args.args.script[0];
          const inputParams = (_args$args = args.args) === null || _args$args === void 0 ? void 0 : (_args$args$inputParam = _args$args.inputParams) === null || _args$args$inputParam === void 0 ? void 0 : _args$args$inputParam[0];
          if (!script) {
            return _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.runscript.sentinelOne.scriptArgValueMissing', {
              defaultMessage: 'A script selection is required'
            });
          }
          if (scriptInfo !== null && scriptInfo !== void 0 && (_scriptInfo$meta = scriptInfo.meta) !== null && _scriptInfo$meta !== void 0 && _scriptInfo$meta.inputRequired && !inputParams) {
            return _i18n.i18n.translate('xpack.securitySolution.consoleCommandsDefinition.runscript.sentinelOne.scriptInputParamsMissing', {
              defaultMessage: 'Script "{name}" requires input parameters to be entered{instructions, select, false {.} other {: {instructions}}}',
              values: {
                name: scriptInfo.name,
                instructions: (scriptInfo.meta.inputInstructions || scriptInfo.meta.inputExample || '').trim() || false
              }
            });
          }
          return true;
        };
      }
    }
    return command;
  });
};

/** @internal */
const adjustCommandsForCrowdstrike = ({
  commandList,
  crowdstrikeRunScriptEnabled
}) => {
  return commandList.map(command => {
    if (command.name === 'status' || !(0, _endpoint.isAgentTypeAndActionSupported)('crowdstrike', _constants.RESPONSE_CONSOLE_COMMAND_TO_API_COMMAND_MAP[command.name], 'manual')) {
      disableCommand(command, 'crowdstrike');
    }
    if (command.name === 'runscript') {
      if (!crowdstrikeRunScriptEnabled) {
        disableCommand(command, 'crowdstrike');
      } else {
        return {
          ...command,
          exampleUsage: `runscript --Raw=\`\`\`Get-ChildItem .\`\`\` --CommandLine=""`,
          helpUsage: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.helpUsage,
          mustHaveArgs: true,
          args: {
            Raw: {
              required: false,
              allowMultiples: false,
              about: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.args.raw.about,
              mustHaveValue: 'non-empty-string',
              exclusiveOr: true
            },
            CloudFile: {
              required: false,
              allowMultiples: false,
              about: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.args.cloudFile.about,
              mustHaveValue: 'truthy',
              exclusiveOr: true,
              selectorShowTextValue: true,
              SelectorComponent: _custom_script_selector.CustomScriptSelector
            },
            CommandLine: {
              required: false,
              allowMultiples: false,
              about: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.args.commandLine.about,
              mustHaveValue: 'non-empty-string'
            },
            HostPath: {
              required: false,
              allowMultiples: false,
              about: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.args.hostPath.about,
              mustHaveValue: 'non-empty-string',
              exclusiveOr: true
            },
            Timeout: {
              required: false,
              allowMultiples: false,
              about: _translations2.CROWDSTRIKE_CONSOLE_COMMANDS.runscript.args.timeout.about,
              mustHaveValue: 'number-greater-than-zero'
            },
            ...commandCommentArgument()
          }
        };
      }
    }
    return command;
  });
};
const adjustCommandsForMicrosoftDefenderEndpoint = ({
  commandList,
  microsoftDefenderEndpointRunScriptEnabled
}) => {
  const featureFlags = _experimental_features_service.ExperimentalFeaturesService.get();
  const microsoftDefenderEndpointCancelEnabled = featureFlags.microsoftDefenderEndpointCancelEnabled;
  return commandList.map(command => {
    if (command.name === 'status' || !(0, _endpoint.isAgentTypeAndActionSupported)('microsoft_defender_endpoint', _constants.RESPONSE_CONSOLE_COMMAND_TO_API_COMMAND_MAP[command.name], 'manual')) {
      disableCommand(command, 'microsoft_defender_endpoint');
    }
    if (command.name === 'cancel' && !microsoftDefenderEndpointCancelEnabled) {
      disableCommand(command, 'microsoft_defender_endpoint');
    }
    if (command.name === 'runscript') {
      if (!microsoftDefenderEndpointRunScriptEnabled) {
        disableCommand(command, 'microsoft_defender_endpoint');
      } else {
        return {
          ...command,
          exampleUsage: `runscript --ScriptName='test.ps1'`,
          mustHaveArgs: true,
          args: {
            ScriptName: {
              required: true,
              allowMultiples: false,
              about: _translations2.MS_DEFENDER_ENDPOINT_CONSOLE_COMMANDS.runscript.args.scriptName.about,
              mustHaveValue: 'truthy',
              selectorShowTextValue: true,
              SelectorComponent: _custom_script_selector.CustomScriptSelector
            },
            Args: {
              required: false,
              allowMultiples: false,
              about: _translations2.MS_DEFENDER_ENDPOINT_CONSOLE_COMMANDS.runscript.args.args.about,
              mustHaveValue: 'non-empty-string'
            },
            ...commandCommentArgument()
          }
        };
      }
    }
    return command;
  });
};