"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateEql = exports.EQL_ERROR_CODES = void 0;
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _eql = require("../../../../common/search_strategy/eql");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let EQL_ERROR_CODES = exports.EQL_ERROR_CODES = /*#__PURE__*/function (EQL_ERROR_CODES) {
  EQL_ERROR_CODES["FAILED_REQUEST"] = "EQL_ERR_FAILED_REQUEST";
  EQL_ERROR_CODES["INVALID_EQL"] = "EQL_ERR_INVALID_EQL";
  EQL_ERROR_CODES["INVALID_SYNTAX"] = "EQL_ERR_INVALID_SYNTAX";
  EQL_ERROR_CODES["MISSING_DATA_SOURCE"] = "EQL_ERR_MISSING_DATA_SOURCE";
  return EQL_ERROR_CODES;
}({});
const validateEql = async ({
  data,
  dataViewTitle,
  query,
  runtimeMappings,
  eqlOptions,
  signal
}) => {
  try {
    const {
      rawResponse: response
    } = await (0, _rxjs.firstValueFrom)(data.search.search({
      params: {
        index: dataViewTitle,
        query,
        runtime_mappings: runtimeMappings,
        size: 0,
        // Prevent passing empty string values
        timestamp_field: eqlOptions !== null && eqlOptions !== void 0 && eqlOptions.timestampField ? eqlOptions.timestampField : undefined,
        tiebreaker_field: eqlOptions !== null && eqlOptions !== void 0 && eqlOptions.tiebreakerField ? eqlOptions.tiebreakerField : undefined,
        event_category_field: eqlOptions !== null && eqlOptions !== void 0 && eqlOptions.eventCategoryField ? eqlOptions.eventCategoryField : undefined
      },
      options: {
        ignore: [400]
      }
    }, {
      strategy: _common.EQL_SEARCH_STRATEGY,
      abortSignal: signal
    }));
    if ((0, _eql.isParsingErrorResponse)(response)) {
      return {
        valid: false,
        error: {
          code: EQL_ERROR_CODES.INVALID_SYNTAX,
          messages: (0, _eql.getValidationErrors)(response)
        }
      };
    }
    if ((0, _eql.isVerificationErrorResponse)(response) || (0, _eql.isMappingErrorResponse)(response)) {
      return {
        valid: false,
        error: {
          code: EQL_ERROR_CODES.INVALID_EQL,
          messages: (0, _eql.getValidationErrors)(response)
        }
      };
    }
    if ((0, _eql.isErrorResponse)(response)) {
      return {
        valid: false,
        error: {
          code: EQL_ERROR_CODES.FAILED_REQUEST,
          error: new Error(JSON.stringify(response))
        }
      };
    }
    return {
      valid: true
    };
  } catch (error) {
    if (error instanceof Error && error.message.startsWith('index_not_found_exception')) {
      return {
        valid: false,
        error: {
          code: EQL_ERROR_CODES.MISSING_DATA_SOURCE,
          messages: [error.message]
        }
      };
    }
    return {
      valid: false,
      error: {
        code: EQL_ERROR_CODES.FAILED_REQUEST,
        error
      }
    };
  }
};
exports.validateEql = validateEql;