"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWithResponderActionDataFromAlert = exports.useResponderActionData = void 0;
var _react = require("react");
var _use_alert_response_actions_support = require("../../../../hooks/endpoint/use_alert_response_actions_support");
var _hooks = require("../../../../../management/hooks");
var _types = require("../../../../../../common/endpoint/types");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This hook is used to get the data needed to show the context menu items for the responder
 * actions using Alert data.
 *
 * NOTE:  If wanting to get teh same type of response but don't have Alert
 *        data, use `useResponderActionData()` instead
 *
 * @param onClick the callback to handle the click event
 * @param eventData the event data, exists only when agentType !== 'endpoint'
 * @returns an object with the data needed to show the context menu item
 */
const useWithResponderActionDataFromAlert = ({
  eventData = [],
  onClick
}) => {
  const {
    isSupported: hostSupportsResponseActions,
    unsupportedReason,
    details: {
      agentType,
      agentId,
      platform,
      hostName
    }
  } = (0, _use_alert_response_actions_support.useAlertResponseActionsSupport)(eventData);
  const isEndpointHost = agentType === 'endpoint';
  const endpointHostData = useResponderDataForEndpointHost(agentId, hostSupportsResponseActions && isEndpointHost);
  const showResponseActionsConsole = (0, _hooks.useWithShowResponder)();
  const [isDisabled, tooltip] = (0, _react.useMemo)(() => {
    if (!hostSupportsResponseActions) {
      return [true, agentType && unsupportedReason ? unsupportedReason : _translations.NOT_FROM_ENDPOINT_HOST_TOOLTIP];
    }
    if (isEndpointHost) {
      return [endpointHostData.isDisabled, endpointHostData.tooltip];
    }
    return [false, undefined];
  }, [hostSupportsResponseActions, isEndpointHost, agentType, unsupportedReason, endpointHostData.isDisabled, endpointHostData.tooltip]);
  const handleResponseActionsClick = (0, _react.useCallback)(() => {
    if (!isDisabled) {
      showResponseActionsConsole({
        agentId,
        agentType,
        hostName,
        platform,
        capabilities: isEndpointHost ? endpointHostData.capabilities : []
      });
      if (onClick) {
        onClick();
      }
    }
  }, [isDisabled, showResponseActionsConsole, agentId, agentType, hostName, platform, isEndpointHost, endpointHostData.capabilities, onClick]);
  return {
    handleResponseActionsClick,
    isDisabled,
    tooltip
  };
};
exports.useWithResponderActionDataFromAlert = useWithResponderActionDataFromAlert;
/**
 * Hook to specifically for the responder data for Elastic Defend endpoints
 * @param endpointAgentId
 * @param enabled
 *
 * @internal
 */
const useResponderDataForEndpointHost = (endpointAgentId, enabled = true) => {
  const {
    data: endpointHostInfo,
    isFetching,
    error
  } = (0, _hooks.useGetEndpointDetails)(endpointAgentId, {
    enabled
  });
  return (0, _react.useMemo)(() => {
    var _error$body, _error$body2, _error$body3, _endpointHostInfo$met, _endpointHostInfo$met2, _endpointHostInfo$met3;
    const response = {
      isDisabled: false,
      tooltip: undefined,
      capabilities: [],
      hostName: '',
      platform: ''
    };
    if (!enabled) {
      response.isDisabled = true;
      return response;
    }
    if (isFetching) {
      response.isDisabled = true;
      response.tooltip = _translations.LOADING_ENDPOINT_DATA_TOOLTIP;
      return response;
    }

    // if we got an error, and it's a 404, it means the endpoint is not from the endpoint host
    if (error && ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.statusCode) === 404) {
      response.isDisabled = true;
      response.tooltip = _translations.NOT_FROM_ENDPOINT_HOST_TOOLTIP;
      return response;
    }

    // if we got an error and,
    // it's a 400 with unenrolled in the error message (alerts can exist for endpoint that are no longer around)
    // or,
    // the Host status is `unenrolled`
    if (error && ((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.statusCode) === 400 && (_error$body3 = error.body) !== null && _error$body3 !== void 0 && _error$body3.message.includes('unenrolled') || (endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.host_status) === _types.HostStatus.UNENROLLED) {
      response.isDisabled = true;
      response.tooltip = _translations.HOST_ENDPOINT_UNENROLLED_TOOLTIP;
      return response;
    }

    // return general error tooltip
    if (error) {
      response.isDisabled = true;
      response.tooltip = _translations.METADATA_API_ERROR_TOOLTIP;
    }
    response.capabilities = (_endpointHostInfo$met = endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.Endpoint.capabilities) !== null && _endpointHostInfo$met !== void 0 ? _endpointHostInfo$met : [];
    response.hostName = (_endpointHostInfo$met2 = endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.host.name) !== null && _endpointHostInfo$met2 !== void 0 ? _endpointHostInfo$met2 : '';
    response.platform = (_endpointHostInfo$met3 = endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.host.os.name.toLowerCase()) !== null && _endpointHostInfo$met3 !== void 0 ? _endpointHostInfo$met3 : '';
    return response;
  }, [enabled, isFetching, error, endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.host_status, endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.Endpoint.capabilities, endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.host.name, endpointHostInfo === null || endpointHostInfo === void 0 ? void 0 : endpointHostInfo.metadata.host.os.name]);
};
/**
 * Returns the data necessary to render a Responder action item (ex. menu item) when only the
 * `agentId` and `agentType` is available (ex. when showing the `Respond` button on the Host
 * details page of SIEM
 * @param onClick
 * @param agentId
 * @param agentType
 */
const useResponderActionData = ({
  onClick,
  agentId,
  agentType
}) => {
  const isEndpointHost = agentType === 'endpoint';
  const showResponseActionsConsole = (0, _hooks.useWithShowResponder)();
  const {
    tooltip,
    isDisabled,
    capabilities,
    hostName,
    platform
  } = useResponderDataForEndpointHost(agentId, isEndpointHost);

  // TODO:PT add support for other agent types once we add the `Respond` button to the Host details page in SIEM

  const handleResponseActionsClick = (0, _react.useCallback)(() => {
    if (!isDisabled) {
      showResponseActionsConsole({
        agentId,
        agentType,
        hostName,
        platform,
        capabilities: isEndpointHost ? capabilities : []
      });
      if (onClick) {
        onClick();
      }
    }
  }, [isDisabled, showResponseActionsConsole, agentId, agentType, hostName, platform, isEndpointHost, capabilities, onClick]);
  return (0, _react.useMemo)(() => {
    return {
      isDisabled: isEndpointHost ? isDisabled : true,
      tooltip: isEndpointHost ? tooltip : _translations.NOT_FROM_ENDPOINT_HOST_TOOLTIP,
      handleResponseActionsClick
    };
  }, [handleResponseActionsClick, isDisabled, isEndpointHost, tooltip]);
};
exports.useResponderActionData = useResponderActionData;