"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PERMISSIONS_POLICY_VIOLATION_EVENT_TYPE = exports.CSP_VIOLATION_EVENT_TYPE = exports.AnalyticsService = exports.AUTHENTICATION_TYPE_EVENT_TYPE = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AUTHENTICATION_TYPE_EVENT_TYPE = exports.AUTHENTICATION_TYPE_EVENT_TYPE = 'security_authentication_type';
const CSP_VIOLATION_EVENT_TYPE = exports.CSP_VIOLATION_EVENT_TYPE = 'security_csp_violation';
const PERMISSIONS_POLICY_VIOLATION_EVENT_TYPE = exports.PERMISSIONS_POLICY_VIOLATION_EVENT_TYPE = 'security_permissions_policy_violation';

/**
 * Interface that represents how CSP violations are stored as EBT events.
 */

/**
 * Interface that represents how permissions policy violations are stored as EBT events.
 */

/**
 * Describes the shape of the authentication type event.
 */

/**
 * Properties that all Reporting API schemas share.
 */

/**
 * Helper type that transforms any Reporting API schema into its corresponding EBT schema:
 *
 * - Removes `type` property since events are identified by their `eventType` in EBT.
 * - Replaces `age` property with `created` timestamp so that we capture a fully qualified date.
 * - Spreads `body` property to keep the resulting EBT schema flat.
 */

/**
 * Describes the shape of the CSP violation event type.
 */
const cspViolation = {
  eventType: CSP_VIOLATION_EVENT_TYPE,
  schema: {
    created: {
      type: 'keyword',
      _meta: {
        description: 'Timestamp when the violation was captured by the user agent.',
        optional: false
      }
    },
    url: {
      type: 'keyword',
      _meta: {
        description: '"url" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    user_agent: {
      type: 'text',
      _meta: {
        description: '"user_agent" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    documentURL: {
      type: 'text',
      _meta: {
        description: '"documentURL" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    referrer: {
      type: 'text',
      _meta: {
        description: '"referrer" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    blockedURL: {
      type: 'text',
      _meta: {
        description: '"blockedURL" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    effectiveDirective: {
      type: 'text',
      _meta: {
        description: '"effectiveDirective" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    originalPolicy: {
      type: 'text',
      _meta: {
        description: '"originalPolicy" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    sourceFile: {
      type: 'text',
      _meta: {
        description: '"sourceFile" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    sample: {
      type: 'text',
      _meta: {
        description: '"sample" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    disposition: {
      type: 'text',
      _meta: {
        description: '"disposition" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    statusCode: {
      type: 'integer',
      _meta: {
        description: '"statusCode" field of W3 Reporting API CSP violation report.',
        optional: false
      }
    },
    lineNumber: {
      type: 'long',
      _meta: {
        description: '"lineNumber" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    },
    columnNumber: {
      type: 'long',
      _meta: {
        description: '"columnNumber" field of W3 Reporting API CSP violation report.',
        optional: true
      }
    }
  }
};

/**
 * Describes the shape of the CSP violation event type.
 */
const permissionsPolicyViolation = {
  eventType: PERMISSIONS_POLICY_VIOLATION_EVENT_TYPE,
  schema: {
    created: {
      type: 'keyword',
      _meta: {
        description: 'Timestamp when the violation was captured by the user agent.',
        optional: false
      }
    },
    url: {
      type: 'keyword',
      _meta: {
        description: '"url" field of Reporting API permissions policy violation report.',
        optional: false
      }
    },
    user_agent: {
      type: 'text',
      _meta: {
        description: '"user_agent" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    featureId: {
      type: 'text',
      _meta: {
        description: '"featureId" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    policyId: {
      type: 'text',
      _meta: {
        description: '"policyId" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    sourceFile: {
      type: 'text',
      _meta: {
        description: '"sourceFile" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    lineNumber: {
      type: 'long',
      _meta: {
        description: '"lineNumber" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    columnNumber: {
      type: 'long',
      _meta: {
        description: '"columnNumber" field of Reporting API permissions policy violation report.',
        optional: true
      }
    },
    disposition: {
      type: 'text',
      _meta: {
        description: '"disposition" field of Reporting API permissions policy violation report.',
        optional: false
      }
    }
  }
};

/**
 * Service that interacts with the Core's analytics module to collect usage of
 * the various Security plugin features (e.g. type of the authentication used).
 */
class AnalyticsService {
  constructor(logger) {
    this.logger = logger;
  }
  setup({
    analytics
  }) {
    this.logger.debug(`Registering ${AUTHENTICATION_TYPE_EVENT_TYPE} event type.`);
    analytics.registerEventType({
      eventType: AUTHENTICATION_TYPE_EVENT_TYPE,
      schema: {
        authentication_provider_type: {
          type: 'keyword',
          _meta: {
            description: 'Type of the Kibana authentication provider.',
            optional: false
          }
        },
        authentication_realm_type: {
          type: 'keyword',
          _meta: {
            description: 'Type of the Elasticsearch security realm.',
            optional: false
          }
        },
        http_authentication_scheme: {
          type: 'keyword',
          _meta: {
            description: 'Authentication scheme from the `Authorization` HTTP header, if present in the client request.',
            // The field is populated only if authentication_provider_type === `http`.
            optional: true
          }
        }
      }
    });
    this.logger.debug(`Registering ${cspViolation.eventType} event type.`);
    analytics.registerEventType(cspViolation);
    this.logger.debug(`Registering ${permissionsPolicyViolation.eventType} event type.`);
    analytics.registerEventType(permissionsPolicyViolation);
    return {
      reportAuthenticationTypeEvent(event) {
        var _event$httpAuthentica;
        analytics.reportEvent(AUTHENTICATION_TYPE_EVENT_TYPE, {
          authentication_provider_type: event.authenticationProviderType.toLowerCase(),
          authentication_realm_type: event.authenticationRealmType.toLowerCase(),
          http_authentication_scheme: (_event$httpAuthentica = event.httpAuthenticationScheme) === null || _event$httpAuthentica === void 0 ? void 0 : _event$httpAuthentica.toLowerCase()
        });
      },
      reportCSPViolation(event) {
        analytics.reportEvent(CSP_VIOLATION_EVENT_TYPE, event);
      },
      reportPermissionsPolicyViolation(event) {
        analytics.reportEvent(PERMISSIONS_POLICY_VIOLATION_EVENT_TYPE, event);
      }
    };
  }
}
exports.AnalyticsService = AnalyticsService;