"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parsePlaygroundSO = parsePlaygroundSO;
exports.parsePlaygroundSOList = parsePlaygroundSOList;
exports.validatePlayground = validatePlayground;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validatePlayground(playground) {
  const errors = [];
  if (playground.name.trim().length === 0) {
    errors.push(_i18n.i18n.translate('xpack.searchPlayground.playgroundNameError', {
      defaultMessage: 'Playground name cannot be empty'
    }));
  }
  try {
    JSON.parse(playground.elasticsearchQueryJSON);
  } catch (e) {
    errors.push(_i18n.i18n.translate('xpack.searchPlayground.esQueryJSONError', {
      defaultMessage: 'Elasticsearch query JSON is invalid\n{jsonParseError}',
      values: {
        jsonParseError: e.message
      }
    }));
  }
  if (playground.userElasticsearchQueryJSON) {
    try {
      JSON.parse(playground.userElasticsearchQueryJSON);
    } catch (e) {
      errors.push(_i18n.i18n.translate('xpack.searchPlayground.userESQueryJSONError', {
        defaultMessage: 'User Elasticsearch query JSON is invalid\n{jsonParseError}',
        values: {
          jsonParseError: e.message
        }
      }));
    }
  }
  // validate query fields greater than 0 and match selected indices
  let totalFieldsCount = 0;
  Object.entries(playground.queryFields).forEach(([index, fields]) => {
    if (!playground.indices.includes(index)) {
      errors.push(_i18n.i18n.translate('xpack.searchPlayground.queryFieldsIndexError', {
        defaultMessage: 'Query fields index {index} does not match selected indices',
        values: {
          index
        }
      }));
    }
    fields === null || fields === void 0 ? void 0 : fields.forEach((field, i) => {
      if (field.trim().length === 0) {
        errors.push(_i18n.i18n.translate('xpack.searchPlayground.queryFieldsError', {
          defaultMessage: 'Query field cannot be empty, {index} item {i} is empty',
          values: {
            index,
            i
          }
        }));
      } else {
        totalFieldsCount++;
      }
    });
  });
  if (totalFieldsCount === 0) {
    errors.push(_i18n.i18n.translate('xpack.searchPlayground.queryFieldsEmptyError', {
      defaultMessage: 'Query fields cannot be empty'
    }));
  }
  if (playground.context) {
    // validate source fields greater than 0 and match a selected index
    let totalSourceFieldsCount = 0;
    Object.entries(playground.context.sourceFields).forEach(([index, fields]) => {
      if (!playground.indices.includes(index)) {
        errors.push(_i18n.i18n.translate('xpack.searchPlayground.sourceFieldsIndexError', {
          defaultMessage: 'Source fields index {index} does not match selected indices',
          values: {
            index
          }
        }));
      }
      fields === null || fields === void 0 ? void 0 : fields.forEach((field, i) => {
        if (field.trim().length === 0) {
          errors.push(_i18n.i18n.translate('xpack.searchPlayground.sourceFieldsError', {
            defaultMessage: 'Source field cannot be empty, {index} item {i} is empty',
            values: {
              index,
              i
            }
          }));
        } else {
          totalSourceFieldsCount++;
        }
      });
    });
    if (totalSourceFieldsCount === 0) {
      errors.push(_i18n.i18n.translate('xpack.searchPlayground.sourceFieldsEmptyError', {
        defaultMessage: 'Source fields cannot be empty'
      }));
    }
  }
  return errors;
}
function parsePlaygroundSO(soPlayground) {
  const {
    id,
    created_at: createdAt,
    created_by: createdBy,
    updated_at: updatedAt,
    updated_by: updatedBy,
    attributes
  } = soPlayground;
  return {
    _meta: {
      id,
      createdAt,
      createdBy,
      updatedAt,
      updatedBy
    },
    data: attributes
  };
}
function parsePlaygroundSOList(playgroundsResponse) {
  const items = playgroundsResponse.saved_objects.map(soPlayground => {
    const {
      id,
      created_at: createdAt,
      created_by: createdBy,
      updated_at: updatedAt,
      updated_by: updatedBy,
      attributes: {
        name
      }
    } = soPlayground;
    return {
      id,
      name,
      createdAt,
      createdBy,
      updatedAt,
      updatedBy
    };
  });
  return {
    _meta: {
      total: playgroundsResponse.total,
      page: playgroundsResponse.page,
      size: playgroundsResponse.per_page
    },
    items
  };
}