"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveToolParameters = exports.createHandler = void 0;
var _utils = require("@kbn/onechat-genai-utils/tools/utils");
var _onechatCommon = require("@kbn/onechat-common");
var _tools = require("@kbn/onechat-server/tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Resolves parameter values by applying defaults for missing parameters.
 * @param paramDefinitions - The parameter definitions from the tool configuration
 * @param providedParams - The parameters provided by the LLM
 * @returns Resolved parameters with defaults applied
 */
const resolveToolParameters = (paramDefinitions, providedParams) => {
  return Object.keys(paramDefinitions).reduce((acc, paramName) => {
    const param = paramDefinitions[paramName];
    const providedValue = providedParams[paramName];
    if (providedValue !== undefined) {
      // LLM provided a value, use it
      acc[paramName] = providedValue;
    } else if (param.optional && param.defaultValue !== undefined) {
      // LLM didn't provide a value, but we have a default
      acc[paramName] = param.defaultValue;
    } else {
      // No value provided and no default, use null
      acc[paramName] = null;
    }
    return acc;
  }, {});
};
exports.resolveToolParameters = resolveToolParameters;
const createHandler = configuration => {
  return async (params, {
    esClient
  }) => {
    const client = esClient.asCurrentUser;

    // Apply default values for parameters that weren't provided by the LLM
    const resolvedParams = resolveToolParameters(configuration.params, params);
    const paramArray = Object.keys(configuration.params).map(param => {
      var _resolvedParams$param;
      return {
        [param]: (_resolvedParams$param = resolvedParams[param]) !== null && _resolvedParams$param !== void 0 ? _resolvedParams$param : null
      };
    });
    const result = await client.esql.query({
      query: configuration.query,
      // TODO: wait until client is fixed: https://github.com/elastic/elasticsearch-specification/issues/5083
      params: paramArray
    });

    // need the interpolated query to return in the results / to display in the UI
    const interpolatedQuery = (0, _utils.interpolateEsqlQuery)(configuration.query, resolvedParams);
    return {
      results: [{
        type: _onechatCommon.ToolResultType.query,
        data: {
          esql: interpolatedQuery
        }
      }, {
        tool_result_id: (0, _tools.getToolResultId)(),
        type: _onechatCommon.ToolResultType.tabularData,
        data: {
          source: 'esql',
          query: interpolatedQuery,
          columns: result.columns,
          values: result.values
        }
      }]
    };
  };
};
exports.createHandler = createHandler;