"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertGraphEvents = void 0;
var _uuid = require("uuid");
var _messages = require("@langchain/core/messages");
var _rxjs = require("rxjs");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const convertGraphEvents = ({
  graphName,
  toolIdMapping,
  logger
}) => {
  return streamEvents$ => {
    const toolCallIdToIdMap = new Map();
    const messageId = (0, _uuid.v4)();
    return streamEvents$.pipe((0, _rxjs.mergeMap)(event => {
      if (!(0, _langchain.matchGraphName)(event, graphName)) {
        return _rxjs.EMPTY;
      }

      // stream answering text chunks for the UI
      if ((0, _langchain.matchEvent)(event, 'on_chat_model_stream') && (0, _langchain.hasTag)(event, _constants.tags.answerAgent)) {
        const chunk = event.data.chunk;
        const textContent = (0, _langchain.extractTextContent)(chunk);
        if (textContent) {
          return (0, _rxjs.of)((0, _langchain.createTextChunkEvent)(textContent, {
            messageId
          }));
        }
      }

      // emit tool calls for research agent steps
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.researchAgent)) {
        const events = [];
        const lastMessage = event.data.output.nextMessage;
        const toolCalls = (0, _langchain.extractToolCalls)(lastMessage);
        if (toolCalls.length > 0) {
          const messageText = (0, _langchain.extractTextContent)(lastMessage);
          let hasReasoningEvent = false;
          for (const toolCall of toolCalls) {
            const toolId = (0, _langchain.toolIdentifierFromToolCall)(toolCall, toolIdMapping);
            const {
              toolCallId,
              args
            } = toolCall;
            const {
              _reasoning,
              ...toolCallArgs
            } = args;
            if (_reasoning) {
              events.push((0, _langchain.createReasoningEvent)(_reasoning));
              hasReasoningEvent = true;
            }
            toolCallIdToIdMap.set(toolCall.toolCallId, toolId);
            events.push((0, _langchain.createToolCallEvent)({
              toolId,
              toolCallId,
              params: toolCallArgs
            }));
          }
          if (messageText && !hasReasoningEvent) {
            events.push((0, _langchain.createReasoningEvent)(messageText));
          }
        }
        return (0, _rxjs.of)(...events);
      }

      // emit messages for answering step
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.answerAgent)) {
        var _event$data$output$ad;
        const events = [];

        // process last emitted message
        const addedMessages = (_event$data$output$ad = event.data.output.addedMessages) !== null && _event$data$output$ad !== void 0 ? _event$data$output$ad : [];
        const lastMessage = addedMessages[addedMessages.length - 1];
        const messageEvent = (0, _langchain.createMessageEvent)((0, _langchain.extractTextContent)(lastMessage), {
          messageId
        });
        events.push(messageEvent);
        return (0, _rxjs.of)(...events);
      }

      // emit tool result events
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.executeTool)) {
        var _addedMessages;
        const toolMessages = ((_addedMessages = event.data.output.addedMessages) !== null && _addedMessages !== void 0 ? _addedMessages : []).filter(_messages.isToolMessage);
        const toolResultEvents = [];
        for (const toolMessage of toolMessages) {
          const toolId = toolCallIdToIdMap.get(toolMessage.tool_call_id);
          const toolReturn = (0, _langchain.extractToolReturn)(toolMessage);
          toolResultEvents.push((0, _langchain.createToolResultEvent)({
            toolCallId: toolMessage.tool_call_id,
            toolId: toolId !== null && toolId !== void 0 ? toolId : 'unknown',
            results: toolReturn.results
          }));
        }
        return (0, _rxjs.of)(...toolResultEvents);
      }
      return _rxjs.EMPTY;
    }));
  };
};
exports.convertGraphEvents = convertGraphEvents;