"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@kbn/config-schema");
var _nodePath = _interopRequireDefault(require("node:path"));
var _rxjs = require("rxjs");
var _sseUtilsServer = require("@kbn/sse-utils-server");
var _onechatCommon = require("@kbn/onechat-common");
var _constants = require("../../common/constants");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerChatRoutes({
  router,
  getInternalServices,
  coreSetup,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });
  const conversePayloadSchema = _configSchema.schema.object({
    agent_id: _configSchema.schema.string({
      defaultValue: _onechatCommon.oneChatDefaultAgentId,
      meta: {
        description: 'The ID of the agent to chat with. Defaults to the default Elastic AI agent.'
      }
    }),
    connector_id: _configSchema.schema.maybe(_configSchema.schema.string({
      meta: {
        description: 'Optional connector ID for the agent to use for external integrations.'
      }
    })),
    conversation_id: _configSchema.schema.maybe(_configSchema.schema.string({
      meta: {
        description: 'Optional existing conversation ID to continue a previous conversation.'
      }
    })),
    input: _configSchema.schema.string({
      meta: {
        description: 'The user input message to send to the agent.'
      }
    }),
    capabilities: _configSchema.schema.maybe(_configSchema.schema.object({
      visualizations: _configSchema.schema.maybe(_configSchema.schema.boolean({
        meta: {
          description: 'When true, allows the agent to render tabular data from tool results as interactive visualizations using custom XML elements in responses.'
        }
      }))
    }, {
      meta: {
        description: 'Controls agent capabilities during conversation. Currently supports visualization rendering for tabular tool results.'
      }
    }))
  });
  const callConverse = ({
    payload,
    request,
    chatService,
    abortSignal
  }) => {
    const {
      agent_id: agentId,
      connector_id: connectorId,
      conversation_id: conversationId,
      input,
      capabilities
    } = payload;
    return chatService.converse({
      agentId,
      connectorId,
      conversationId,
      capabilities,
      abortSignal,
      nextInput: {
        message: input
      },
      request
    });
  };
  router.versioned.post({
    path: `${_constants.publicApiPath}/converse`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Send chat message',
    description: 'Send a message to an agent and receive a complete response. This synchronous endpoint waits for the agent to fully process your request before returning the final result. Use this for simple chat interactions where you need the complete response.',
    options: {
      tags: ['oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: conversePayloadSchema
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/chat_converse.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      chat: chatService
    } = getInternalServices();
    const payload = request.body;
    const abortController = new AbortController();
    request.events.aborted$.subscribe(() => {
      abortController.abort();
    });
    const chatEvents$ = callConverse({
      chatService,
      payload,
      request,
      abortSignal: abortController.signal
    });
    const events = await (0, _rxjs.firstValueFrom)(chatEvents$.pipe((0, _rxjs.toArray)()));
    const {
      data: {
        round
      }
    } = events.find(_onechatCommon.isRoundCompleteEvent);
    const {
      data: {
        conversation_id: convId
      }
    } = events.find(e => (0, _onechatCommon.isConversationUpdatedEvent)(e) || (0, _onechatCommon.isConversationCreatedEvent)(e));
    return response.ok({
      body: {
        conversation_id: convId,
        trace_id: round.trace_id,
        steps: round.steps,
        response: round.response
      }
    });
  }));
  router.versioned.post({
    path: `${_constants.publicApiPath}/converse/async`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Send chat message (streaming)',
    description: "Send a message to an agent and receive real-time streaming events. This asynchronous endpoint provides live updates as the agent processes your request, allowing you to see intermediate steps and progress. Use this for interactive experiences where you want to monitor the agent's thinking process.",
    options: {
      tags: ['oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: conversePayloadSchema
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/chat_converse_async.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const [, {
      cloud
    }] = await coreSetup.getStartServices();
    const {
      chat: chatService
    } = getInternalServices();
    const payload = request.body;
    const abortController = new AbortController();
    request.events.aborted$.subscribe(() => {
      abortController.abort();
    });
    const chatEvents$ = callConverse({
      chatService,
      payload,
      request,
      abortSignal: abortController.signal
    });
    return response.ok({
      headers: {
        // cloud compress text/* types, loosing chunking capabilities which we need for SSE
        'Content-Type': cloud !== null && cloud !== void 0 && cloud.isCloudEnabled ? 'application/octet-stream' : 'text/event-stream',
        'Cache-Control': 'no-cache',
        Connection: 'keep-alive',
        'Transfer-Encoding': 'chunked',
        'X-Content-Type-Options': 'nosniff',
        // This disables response buffering on proxy servers
        'X-Accel-Buffering': 'no'
      },
      body: (0, _sseUtilsServer.observableIntoEventSourceStream)(chatEvents$, {
        signal: abortController.signal,
        flushThrottleMs: 100,
        flushMinBytes: cloud !== null && cloud !== void 0 && cloud.isCloudEnabled ? _sseUtilsServer.cloudProxyBufferSize : undefined,
        logger
      })
    });
  }));
}