"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerExecuteQueryFunction = void 0;
exports.registerQueryFunction = registerQueryFunction;
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _inferenceCommon = require("@kbn/inference-common");
var _common = require("@kbn/inference-plugin/common");
var _server = require("@kbn/inference-plugin/server");
var _common2 = require("@kbn/observability-ai-assistant-plugin/common");
var _create_function_response_message = require("@kbn/observability-ai-assistant-plugin/common/utils/create_function_response_message");
var _convert_messages_for_inference = require("@kbn/observability-ai-assistant-plugin/common/convert_messages_for_inference");
var _validate_esql_query = require("./validate_esql_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerExecuteQueryFunction = ({
  functions,
  resources,
  signal
}) => {
  functions.registerFunction({
    name: _common2.EXECUTE_QUERY_FUNCTION_NAME,
    isInternal: true,
    description: `Execute a generated ES|QL query on behalf of the user. The results
        will be returned to you.

        You must use this tool if the user is asking for the result of a query,
        such as a metric or list of things, but does not want to visualize it in
        a table or chart. You do NOT need to ask permission to execute the query
        after generating it, use the "${_common2.EXECUTE_QUERY_FUNCTION_NAME}" tool directly instead.

        **EXCEPTION**: Do NOT use when the user just asks for an **EXAMPLE**.`,
    parameters: {
      type: 'object',
      properties: {
        query: {
          type: 'string'
        }
      },
      required: ['query']
    }
  }, async ({
    arguments: {
      query
    }
  }) => {
    const correctedQuery = (0, _common.correctCommonEsqlMistakes)(query).output;
    const client = (await resources.context.core).elasticsearch.client.asCurrentUser;
    const {
      error,
      errorMessages,
      rows,
      columns
    } = await (0, _validate_esql_query.runAndValidateEsqlQuery)({
      query: correctedQuery,
      client,
      signal
    });
    if (!!error) {
      return {
        content: {
          message: 'The query failed to execute',
          error,
          errorMessages
        }
      };
    }
    return {
      content: {
        columns,
        rows
      }
    };
  });
};
exports.registerExecuteQueryFunction = registerExecuteQueryFunction;
function registerQueryFunction(params) {
  const {
    functions,
    resources,
    pluginsStart
  } = params;
  registerExecuteQueryFunction(params);
  functions.registerFunction({
    name: _common2.QUERY_FUNCTION_NAME,
    description: `This tool generates, executes and/or visualizes a query
      based on the user's request. It also explains how ES|QL works and how to
      convert queries from one language to another. Make sure you call one of
      the get_dataset functions first if you need index or field names. This
      tool takes no input.`
  }, async ({
    messages,
    connectorId,
    simulateFunctionCalling
  }) => {
    const esqlFunctions = functions.getFunctions().filter(fn => fn.definition.name === _common2.EXECUTE_QUERY_FUNCTION_NAME || fn.definition.name === _common2.VISUALIZE_QUERY_FUNCTION_NAME).map(fn => fn.definition);
    const actions = functions.getActions();
    const inferenceMessages = (0, _convert_messages_for_inference.convertMessagesForInference)(
    // remove system message and query function request
    messages.filter(message => message.message.role !== _common2.MessageRole.System).slice(0, -1), resources.logger);
    const availableToolDefinitions = Object.fromEntries([...actions, ...esqlFunctions].map(fn => [fn.name, {
      description: fn.description,
      schema: fn.parameters
    }]));
    const events$ = (0, _server.naturalLanguageToEsql)({
      client: pluginsStart.inference.getClient({
        request: resources.request
      }),
      connectorId,
      messages: inferenceMessages,
      logger: resources.logger,
      tools: availableToolDefinitions,
      functionCalling: simulateFunctionCalling ? 'simulated' : 'auto',
      maxRetries: 0,
      metadata: {
        connectorTelemetry: {
          pluginId: 'observability_ai_assistant'
        }
      },
      system: `
<CriticalInstructions>
 1. **CHECK YOUR TOOLS FIRST.** Your capabilities are strictly limited to the tools listed in the AvailableTools section below.
 2. **DISREGARD PAST TOOLS.** 
  * Under NO circumstances should you use any tool that is not explicitly defined in the AvailableTools section for THIS turn. 
  * Tools used or mentioned in previous parts of the conversation are NOT available unless they are listed below. 
  * Calling unavailable tools will result in a **critical error and task failure**.
 3. **Critical ES|QL syntax rules:**
      * When using \`DATE_FORMAT\`, any literal text in the format string **MUST** be in single quotes. Example: \`DATE_FORMAT("d 'of' MMMM yyyy", @timestamp)\`.
      * When grouping with \`STATS\`, use the field name directly. Example: \`STATS count = COUNT(*) BY destination.domain\`
</CriticalInstructions>
 <AvailableTools>
 * These are the only known and available tools for use: 
      \`\`\`json
      ${JSON.stringify(availableToolDefinitions, null, 4)}
      \'\'\'
 * ALL OTHER tools not listed here are **NOT AVAILABLE** and calls to them will **FAIL**.
 </AvailableTools> `
    });
    const chatMessageId = (0, _uuid.v4)();
    return events$.pipe((0, _rxjs.map)(event => {
      if ((0, _inferenceCommon.isOutputEvent)(event)) {
        return (0, _create_function_response_message.createFunctionResponseMessage)({
          content: {},
          name: _common2.QUERY_FUNCTION_NAME,
          data: event.output
        });
      }
      if ((0, _inferenceCommon.isChatCompletionChunkEvent)(event)) {
        return {
          id: chatMessageId,
          type: _common2.StreamingChatResponseEventType.ChatCompletionChunk,
          message: {
            content: event.content
          }
        };
      }
      const fnCall = event.toolCalls[0] ? {
        name: event.toolCalls[0].function.name,
        arguments: JSON.stringify(event.toolCalls[0].function.arguments),
        trigger: _common2.MessageRole.Assistant
      } : undefined;
      const messageAddEvent = {
        type: _common2.StreamingChatResponseEventType.MessageAdd,
        id: chatMessageId,
        message: {
          '@timestamp': new Date().toISOString(),
          message: {
            content: event.content,
            role: _common2.MessageRole.Assistant,
            function_call: fnCall
          }
        }
      };
      return messageAddEvent;
    }));
  });
}