"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildConfigFromDetector = buildConfigFromDetector;
var _lodash = require("lodash");
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _job_utils = require("../../../common/util/job_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Builds the configuration object used to plot a chart showing anomalies
 * in the source metric data.
 */

/**
 * Builds the basic configuration to plot a chart of the source data
 * analyzed by the the detector at the given index from the specified ML job.
 * @param job Job config info
 * @param detectorIndex The index of the detector in the job config
 * @param metricFunctionDescription The underlying function (min, max, avg) for "metric" detector type
 * @returns
 */
function buildConfigFromDetector(job, detectorIndex, metricFunctionDescription) {
  const analysisConfig = job.analysis_config;
  const detector = analysisConfig.detectors[detectorIndex];
  const config = {
    jobId: job.job_id,
    detectorIndex,
    metricFunction: detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.LAT_LONG ? _mlAnomalyUtils.ML_JOB_AGGREGATION.LAT_LONG : (0, _job_utils.mlFunctionToESAggregation)(detector.function),
    timeField: job.data_description.time_field,
    // @ts-expect-error bucket_span is of type estypes.Duration
    interval: job.analysis_config.bucket_span,
    datafeedConfig: job.datafeed_config,
    summaryCountFieldName: job.analysis_config.summary_count_field_name
  };
  if (detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.METRIC && metricFunctionDescription !== undefined) {
    config.metricFunction = metricFunctionDescription;
  }
  if (detector.field_name !== undefined) {
    config.metricFieldName = detector.field_name;
  }

  // Extra checks if the job config uses a summary count field.
  const summaryCountFieldName = analysisConfig.summary_count_field_name;
  if (config.metricFunction === _mlAnomalyUtils.ES_AGGREGATION.COUNT && summaryCountFieldName !== undefined && summaryCountFieldName !== _mlAnomalyUtils.DOC_COUNT && summaryCountFieldName !== _mlAnomalyUtils._DOC_COUNT) {
    // Check for a detector looking at cardinality (distinct count) using an aggregation.
    // The cardinality field will be in:
    // aggregations/<agg_name>/aggregations/<summaryCountFieldName>/cardinality/field
    // or aggs/<agg_name>/aggs/<summaryCountFieldName>/cardinality/field
    let cardinalityField;
    const topAgg = (0, _lodash.get)(job.datafeed_config, 'aggregations') || (0, _lodash.get)(job.datafeed_config, 'aggs');
    if (topAgg !== undefined && Object.values(topAgg).length > 0) {
      cardinalityField = (0, _lodash.get)(Object.values(topAgg)[0], ['aggregations', summaryCountFieldName, _mlAnomalyUtils.ES_AGGREGATION.CARDINALITY, 'field']) || (0, _lodash.get)(Object.values(topAgg)[0], ['aggs', summaryCountFieldName, _mlAnomalyUtils.ES_AGGREGATION.CARDINALITY, 'field']);
    }
    if ((detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.NON_ZERO_COUNT || detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_NON_ZERO_COUNT || detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_NON_ZERO_COUNT || detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.COUNT || detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_COUNT || detector.function === _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_COUNT) && cardinalityField !== undefined) {
      config.metricFunction = _mlAnomalyUtils.ES_AGGREGATION.CARDINALITY;
      config.metricFieldName = undefined;
    } else {
      // For count detectors using summary_count_field, plot sum(summary_count_field_name)
      config.metricFunction = _mlAnomalyUtils.ES_AGGREGATION.SUM;
      config.metricFieldName = summaryCountFieldName;
    }
  }
  return config;
}