"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getThemeResolvedSeverityColor = getThemeResolvedSeverityColor;
exports.getThemeResolvedSeverityStrokeColor = getThemeResolvedSeverityStrokeColor;
exports.useSeverityColor = void 0;
var _react = require("react");
var _eui = require("@elastic/eui");
var _anomaly_threshold = require("./anomaly_threshold");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Resolves severity color based on normalized score and EUI theme colors.
 * This function can be used in contexts where the EuiTheme object is available,
 * for example, within class components that receive the theme via props or context.
 *
 * @param normalizedScore The anomaly score, normalized (typically 0-100).
 * @param euiTheme The EuiTheme object.
 * @returns The hex color string for the severity.
 */
function getThemeResolvedSeverityColor(normalizedScore, euiTheme) {
  const skyBluePalette = (0, _eui.euiPaletteSkyBlue)(6);
  if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.CRITICAL) {
    return euiTheme.colors.severity.danger;
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.MAJOR) {
    return euiTheme.colors.severity.risk;
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.MINOR) {
    return euiTheme.colors.severity.warning;
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.WARNING) {
    return skyBluePalette[2];
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.LOW) {
    return skyBluePalette[0];
  } else {
    return euiTheme.colors.severity.unknown;
  }
}
function getThemeResolvedSeverityStrokeColor(normalizedScore, euiTheme) {
  const orangePalette = (0, _eui.euiPaletteOrange)(6);
  const yellowPalette = (0, _eui.euiPaletteYellow)(6);
  const skyBluePalette = (0, _eui.euiPaletteSkyBlue)(6);
  if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.CRITICAL) {
    return euiTheme.colors.danger;
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.MAJOR) {
    return orangePalette[5];
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.MINOR) {
    return yellowPalette[5];
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.WARNING) {
    return skyBluePalette[3];
  } else if (normalizedScore >= _anomaly_threshold.ML_ANOMALY_THRESHOLD.LOW) {
    return skyBluePalette[0];
  } else {
    return euiTheme.colors.severity.unknown;
  }
}

/**
 * A React hook to get a theme-aware severity color string directly.
 * This hook computes the color based on the normalized score and the current EUI theme.
 *
 * @param normalizedScore The anomaly score, normalized (typically 0-100).
 * @returns The hex color string for the severity, corresponding to the current theme.
 */
const useSeverityColor = normalizedScore => {
  const {
    euiTheme
  } = (0, _eui.useEuiTheme)();
  return (0, _react.useMemo)(() => {
    return getThemeResolvedSeverityColor(normalizedScore, euiTheme);
  }, [normalizedScore, euiTheme]);
};
exports.useSeverityColor = useSeverityColor;