"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLogsHasDataFetcher = getLogsHasDataFetcher;
exports.getLogsOverviewDataFetcher = getLogsOverviewDataFetcher;
var _common = require("@kbn/logs-shared-plugin/common");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getLogsHasDataFetcher(getStartServices) {
  return async () => {
    const [, {
      logsShared
    }] = await getStartServices();
    const resolvedLogView = await logsShared.logViews.client.getResolvedLogView(_common.DEFAULT_LOG_VIEW);
    const logViewStatus = await logsShared.logViews.client.getResolvedLogViewStatus(resolvedLogView);
    const hasData = logViewStatus.index === 'available';
    const indices = resolvedLogView.indices;
    return {
      hasData,
      indices
    };
  };
}
function getLogsOverviewDataFetcher(getStartServices) {
  return async params => {
    const [, {
      data,
      logsShared,
      share
    }] = await getStartServices();
    const resolvedLogView = await logsShared.logViews.client.getResolvedLogView(_common.DEFAULT_LOG_VIEW);
    const {
      stats,
      series
    } = await fetchLogsOverview({
      index: resolvedLogView.indices
    }, params, data);
    const logsLocator = (0, _common.getLogsLocatorFromUrlService)(share.url);
    const timeSpanInMinutes = (params.absoluteTime.end - params.absoluteTime.start) / (1000 * 60);
    const appLink = logsLocator.getRedirectUrl({
      timeRange: {
        from: (0, _moment.default)(params.absoluteTime.start).toISOString(),
        to: (0, _moment.default)(params.absoluteTime.end).toISOString()
      }
    });
    return {
      appLink,
      stats: normalizeStats(stats, timeSpanInMinutes),
      series: normalizeSeries(series)
    };
  };
}
async function fetchLogsOverview(logParams, params, dataPlugin) {
  return new Promise((resolve, reject) => {
    let esResponse;
    dataPlugin.search.search({
      params: {
        index: logParams.index,
        body: {
          size: 0,
          query: buildLogOverviewQuery(logParams, params),
          aggs: buildLogOverviewAggregations(logParams, params)
        }
      }
    }).subscribe(response => esResponse = response.rawResponse, error => reject(error), () => {
      var _esResponse;
      if ((_esResponse = esResponse) !== null && _esResponse !== void 0 && _esResponse.aggregations) {
        resolve(processLogsOverviewAggregations(esResponse.aggregations));
      } else {
        resolve({
          stats: {},
          series: {}
        });
      }
    });
  });
}
function buildLogOverviewQuery(logParams, params) {
  return {
    range: {
      [_constants.TIMESTAMP_FIELD]: {
        gt: new Date(params.absoluteTime.start).toISOString(),
        lte: new Date(params.absoluteTime.end).toISOString(),
        format: 'strict_date_optional_time'
      }
    }
  };
}
function buildLogOverviewAggregations(logParams, params) {
  return {
    stats: {
      terms: {
        field: 'event.dataset',
        size: 4,
        missing: 'unknown'
      },
      aggs: {
        series: {
          date_histogram: {
            field: _constants.TIMESTAMP_FIELD,
            fixed_interval: params.intervalString
          }
        }
      }
    }
  };
}
function processLogsOverviewAggregations(aggregations) {
  const processedStats = {};
  const processedSeries = {};
  aggregations.stats.buckets.forEach(stat => {
    const label = stat.key;
    processedStats[stat.key] = {
      type: 'number',
      label,
      value: stat.doc_count
    };
    stat.series.buckets.forEach(series => {
      processedSeries[label] = processedSeries[label] || {
        label,
        coordinates: []
      };
      processedSeries[label].coordinates.push({
        x: series.key,
        y: series.doc_count
      });
    });
  });
  return {
    stats: processedStats,
    series: processedSeries
  };
}
function normalizeStats(stats, timeSpanInMinutes) {
  return Object.keys(stats).reduce((normalized, key) => {
    normalized[key] = {
      ...stats[key],
      value: stats[key].value / timeSpanInMinutes
    };
    return normalized;
  }, {});
}
function normalizeSeries(series) {
  const seriesKeys = Object.keys(series);
  const timestamps = seriesKeys.flatMap(key => series[key].coordinates.map(c => c.x));
  const [first, second] = [...new Set(timestamps)].sort();
  const timeSpanInMinutes = (second - first) / (1000 * 60);
  return seriesKeys.reduce((normalized, key) => {
    normalized[key] = {
      ...series[key],
      coordinates: series[key].coordinates.map(c => {
        if (c.y) {
          return {
            ...c,
            y: c.y / timeSpanInMinutes
          };
        }
        return c;
      })
    };
    return normalized;
  }, {});
}