"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fleetServerHostService = void 0;
exports.getFleetServerHostsForAgentPolicy = getFleetServerHostsForAgentPolicy;
exports.migrateSettingsToFleetServerHost = migrateSettingsToFleetServerHost;
var _lodash = require("lodash");
var _services = require("../../common/services");
var _constants = require("../constants");
var _errors = require("../errors");
var _app_context = require("./app_context");
var _agent_policy = require("./agent_policy");
var _saved_object = require("./saved_object");
var _secrets = require("./secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function savedObjectToFleetServerHost(so) {
  const {
    ssl,
    proxy_id: proxyId,
    ...attributes
  } = so.attributes;
  return {
    id: so.id,
    ...attributes,
    ...(ssl ? {
      ssl: JSON.parse(ssl)
    } : {}),
    ...(proxyId ? {
      proxy_id: proxyId
    } : {})
  };
}
const fakeRequest = {
  headers: {},
  getBasePath: () => '',
  path: '/',
  route: {
    settings: {}
  },
  url: {
    href: '/'
  },
  raw: {
    req: {
      url: '/'
    }
  }
};
class FleetServerHostService {
  get soClient() {
    return _app_context.appContextService.getInternalUserSOClient(fakeRequest);
  }
  get encryptedSoClient() {
    return _app_context.appContextService.getEncryptedSavedObjects();
  }
  async create(soClient, esClient, fleetServerHost, options) {
    var _appContextService$ge;
    const logger = _app_context.appContextService.getLogger();
    const data = {
      ...(0, _lodash.omit)(fleetServerHost, ['ssl', 'secrets'])
    };
    if (!((_appContextService$ge = _app_context.appContextService.getEncryptedSavedObjectsSetup()) !== null && _appContextService$ge !== void 0 && _appContextService$ge.canEncrypt)) {
      throw new _errors.FleetEncryptedSavedObjectEncryptionKeyRequired(`Fleet server host needs encrypted saved object api key to be set`);
    }
    if (fleetServerHost.is_default) {
      const defaultItem = await this.getDefaultFleetServerHost();
      if (defaultItem && defaultItem.id !== (options === null || options === void 0 ? void 0 : options.id)) {
        await this.update(soClient, esClient, defaultItem.id, {
          is_default: false
        }, {
          fromPreconfiguration: options === null || options === void 0 ? void 0 : options.fromPreconfiguration
        });
      }
    }
    if (fleetServerHost.host_urls) {
      data.host_urls = fleetServerHost.host_urls.map(_services.normalizeHostsForAgents);
    }
    if (fleetServerHost.ssl) {
      data.ssl = JSON.stringify(fleetServerHost.ssl);
    }

    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
      const {
        fleetServerHost: fleetServerHostWithSecrets
      } = await (0, _secrets.extractAndWriteFleetServerHostsSecrets)({
        fleetServerHost,
        esClient,
        secretHashes: fleetServerHost.is_preconfigured ? options === null || options === void 0 ? void 0 : options.secretHashes : undefined
      });
      if (fleetServerHostWithSecrets.secrets) data.secrets = fleetServerHostWithSecrets.secrets;
    } else {
      var _fleetServerHost$ssl, _fleetServerHost$secr, _fleetServerHost$secr2, _fleetServerHost$ssl2, _fleetServerHost$secr3, _fleetServerHost$secr4, _fleetServerHost$ssl3, _fleetServerHost$secr5, _fleetServerHost$secr6;
      if (!((_fleetServerHost$ssl = fleetServerHost.ssl) !== null && _fleetServerHost$ssl !== void 0 && _fleetServerHost$ssl.key) && (_fleetServerHost$secr = fleetServerHost.secrets) !== null && _fleetServerHost$secr !== void 0 && (_fleetServerHost$secr2 = _fleetServerHost$secr.ssl) !== null && _fleetServerHost$secr2 !== void 0 && _fleetServerHost$secr2.key || !((_fleetServerHost$ssl2 = fleetServerHost.ssl) !== null && _fleetServerHost$ssl2 !== void 0 && _fleetServerHost$ssl2.es_key) && (_fleetServerHost$secr3 = fleetServerHost.secrets) !== null && _fleetServerHost$secr3 !== void 0 && (_fleetServerHost$secr4 = _fleetServerHost$secr3.ssl) !== null && _fleetServerHost$secr4 !== void 0 && _fleetServerHost$secr4.es_key || !((_fleetServerHost$ssl3 = fleetServerHost.ssl) !== null && _fleetServerHost$ssl3 !== void 0 && _fleetServerHost$ssl3.agent_key) && (_fleetServerHost$secr5 = fleetServerHost.secrets) !== null && _fleetServerHost$secr5 !== void 0 && (_fleetServerHost$secr6 = _fleetServerHost$secr5.ssl) !== null && _fleetServerHost$secr6 !== void 0 && _fleetServerHost$secr6.agent_key) {
        data.ssl = JSON.stringify({
          ...fleetServerHost.ssl,
          ...fleetServerHost.secrets.ssl
        });
      }
    }
    const res = await this.soClient.create(_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE, data, {
      id: options === null || options === void 0 ? void 0 : options.id,
      overwrite: options === null || options === void 0 ? void 0 : options.overwrite
    });
    logger.debug(`Created fleet server host ${options === null || options === void 0 ? void 0 : options.id}`);
    // soClient.create doesn't return the decrypted attributes, so we need to fetch it again.
    const retrievedSo = await this.encryptedSoClient.getDecryptedAsInternalUser(_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE, res.id);
    return savedObjectToFleetServerHost(retrievedSo);
  }
  async get(id) {
    const res = await this.encryptedSoClient.getDecryptedAsInternalUser(_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE, id);
    return savedObjectToFleetServerHost(res);
  }
  async list() {
    const fleetServerHostsFinder = await this.encryptedSoClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE,
      perPage: _constants.SO_SEARCH_LIMIT
    });
    let fleetServerHosts = [];
    let total = 0;
    let page = 0;
    let perPage = 0;
    for await (const result of fleetServerHostsFinder.find()) {
      fleetServerHosts = result.saved_objects;
      page = result.page;
      total = result.total;
      perPage = result.per_page;
      break; // Return first page;
    }
    await fleetServerHostsFinder.close();
    return {
      items: fleetServerHosts.map(savedObjectToFleetServerHost),
      total,
      page,
      perPage
    };
  }
  async listAllForProxyId(proxyId) {
    const res = await this.soClient.find({
      type: _constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE,
      perPage: _constants.SO_SEARCH_LIMIT,
      searchFields: ['proxy_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(proxyId)
    });
    return {
      items: res.saved_objects.map(savedObjectToFleetServerHost),
      total: res.total,
      page: res.page,
      perPage: res.per_page
    };
  }
  async delete(esClient, id, options) {
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Deleting fleet server host ${id}`);
    const fleetServerHost = await this.get(id);
    if (fleetServerHost.is_preconfigured && !(options !== null && options !== void 0 && options.fromPreconfiguration)) {
      throw new _errors.FleetServerHostUnauthorizedError(`Cannot delete ${id} preconfigured fleet server host`);
    }
    if (fleetServerHost.is_default) {
      throw new _errors.FleetServerHostUnauthorizedError(`Default Fleet Server hosts ${id} cannot be deleted.`);
    }
    await _agent_policy.agentPolicyService.removeFleetServerHostFromAll(esClient, id, {
      force: options === null || options === void 0 ? void 0 : options.fromPreconfiguration
    });
    const soDeleteResult = await this.soClient.delete(_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE, id);
    await (0, _secrets.deleteFleetServerHostsSecrets)({
      fleetServerHost,
      esClient: _app_context.appContextService.getInternalUserESClient()
    });
    return soDeleteResult;
  }
  async update(soClient, esClient, id, data, options) {
    let secretsToDelete = [];
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Updating fleet server host ${id}`);
    const originalItem = await this.get(id);
    const updateData = {
      ...(0, _lodash.omit)(data, ['ssl', 'secrets'])
    };
    if (data.is_preconfigured && !(options !== null && options !== void 0 && options.fromPreconfiguration)) {
      throw new _errors.FleetServerHostUnauthorizedError(`Cannot update ${id} preconfigured fleet server host`);
    }
    if (data.is_default) {
      const defaultItem = await this.getDefaultFleetServerHost();
      if (defaultItem && defaultItem.id !== id) {
        await this.update(soClient, esClient, defaultItem.id, {
          is_default: false
        }, {
          fromPreconfiguration: options === null || options === void 0 ? void 0 : options.fromPreconfiguration
        });
      }
    }
    if (data.host_urls) {
      updateData.host_urls = data.host_urls.map(_services.normalizeHostsForAgents);
    }
    if (data.ssl) {
      updateData.ssl = JSON.stringify(data.ssl);
    } else if (data.ssl === null) {
      // Explicitly set to null to allow to delete the field
      updateData.ssl = null;
    }

    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
      const secretsRes = await (0, _secrets.extractAndUpdateFleetServerHostsSecrets)({
        oldFleetServerHost: originalItem,
        fleetServerHostUpdate: data,
        esClient,
        secretHashes: data.is_preconfigured ? options === null || options === void 0 ? void 0 : options.secretHashes : undefined
      });
      updateData.secrets = secretsRes.fleetServerHostUpdate.secrets;
      secretsToDelete = secretsRes.secretsToDelete;
    } else {
      var _data$ssl, _data$secrets, _data$secrets$ssl, _data$ssl2, _data$secrets2, _data$secrets2$ssl, _data$ssl3, _data$secrets3, _data$secrets3$ssl;
      if (!((_data$ssl = data.ssl) !== null && _data$ssl !== void 0 && _data$ssl.key) && (_data$secrets = data.secrets) !== null && _data$secrets !== void 0 && (_data$secrets$ssl = _data$secrets.ssl) !== null && _data$secrets$ssl !== void 0 && _data$secrets$ssl.key || !((_data$ssl2 = data.ssl) !== null && _data$ssl2 !== void 0 && _data$ssl2.es_key) && (_data$secrets2 = data.secrets) !== null && _data$secrets2 !== void 0 && (_data$secrets2$ssl = _data$secrets2.ssl) !== null && _data$secrets2$ssl !== void 0 && _data$secrets2$ssl.es_key || !((_data$ssl3 = data.ssl) !== null && _data$ssl3 !== void 0 && _data$ssl3.agent_key) && (_data$secrets3 = data.secrets) !== null && _data$secrets3 !== void 0 && (_data$secrets3$ssl = _data$secrets3.ssl) !== null && _data$secrets3$ssl !== void 0 && _data$secrets3$ssl.agent_key) {
        updateData.ssl = JSON.stringify({
          ...data.ssl,
          ...data.secrets.ssl
        });
      }
    }
    await this.soClient.update(_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE, id, updateData);
    if (secretsToDelete.length) {
      try {
        await (0, _secrets.deleteSecrets)({
          esClient,
          ids: secretsToDelete.map(s => s.id)
        });
      } catch (err) {
        logger.warn(`Error cleaning up secrets for output ${id}: ${err.message}`);
      }
    }
    logger.debug(`Updated fleet server host ${id}`);
    return {
      ...originalItem,
      ...updateData
    };
  }
  async bulkGet(ids, {
    ignoreNotFound = false
  } = {
    ignoreNotFound: true
  }) {
    if (ids.length === 0) {
      return [];
    }
    const res = await this.soClient.bulkGet(ids.map(id => ({
      id,
      type: _constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE
    })));
    return res.saved_objects.map(so => {
      if (so.error) {
        if (!ignoreNotFound || so.error.statusCode !== 404) {
          throw so.error;
        }
        return undefined;
      }
      return savedObjectToFleetServerHost(so);
    }).filter(fleetServerHostOrUndefined => typeof fleetServerHostOrUndefined !== 'undefined');
  }

  /**
   * Get the default Fleet server policy hosts or throw if it does not exists
   */
  async getDefaultFleetServerHost() {
    const res = await this.soClient.find({
      type: _constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE,
      filter: `${_constants.FLEET_SERVER_HOST_SAVED_OBJECT_TYPE}.attributes.is_default:true`
    });
    if (res.saved_objects.length === 0) {
      return null;
    }
    return savedObjectToFleetServerHost(res.saved_objects[0]);
  }
}
const fleetServerHostService = exports.fleetServerHostService = new FleetServerHostService();
async function getFleetServerHostsForAgentPolicy(soClient, agentPolicy) {
  if (agentPolicy.fleet_server_host_id) {
    return fleetServerHostService.get(agentPolicy.fleet_server_host_id);
  }
  const defaultFleetServerHost = await fleetServerHostService.getDefaultFleetServerHost();
  if (!defaultFleetServerHost) {
    throw new _errors.FleetServerHostNotFoundError('Default Fleet Server host is not setup');
  }
  return defaultFleetServerHost;
}

/**
 * Migrate Global setting fleet server hosts to their own saved object
 */
async function migrateSettingsToFleetServerHost(soClient, esClient) {
  const defaultFleetServerHost = await fleetServerHostService.getDefaultFleetServerHost();
  if (defaultFleetServerHost) {
    return;
  }
  const res = await soClient.find({
    type: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
  });
  const oldSettings = res.saved_objects[0];
  if (!oldSettings || !oldSettings.attributes.fleet_server_hosts || oldSettings.attributes.fleet_server_hosts.length === 0) {
    return;
  }

  // Migrate
  await fleetServerHostService.create(soClient, esClient, {
    name: 'Default',
    host_urls: oldSettings.attributes.fleet_server_hosts,
    is_default: true,
    is_preconfigured: false
  }, {
    id: _constants.DEFAULT_FLEET_SERVER_HOST_ID,
    overwrite: true
  });
}