"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = exports.registerGetFleetStatusRoute = exports.registerFleetSetupRoute = exports.registerCreateFleetSetupRoute = exports.GetAgentsSetupResponseSchema = exports.FleetSetupResponseSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _api_privileges = require("../../constants/api_privileges");
var _constants = require("../../constants");
var _constants2 = require("../../../common/constants");
var _errors = require("../schema/errors");
var _handlers = require("./handlers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FleetSetupResponseSchema = exports.FleetSetupResponseSchema = _configSchema.schema.object({
  isInitialized: _configSchema.schema.boolean(),
  nonFatalErrors: _configSchema.schema.arrayOf(_configSchema.schema.object({
    name: _configSchema.schema.string(),
    message: _configSchema.schema.string()
  }))
}, {
  meta: {
    description: "A summary of the result of Fleet's `setup` lifecycle. If `isInitialized` is true, Fleet is ready to accept agent enrollment. `nonFatalErrors` may include useful insight into non-blocking issues with Fleet setup."
  }
});
const registerFleetSetupRoute = router => {
  router.versioned.post({
    path: _constants.SETUP_API_ROUTE,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Initiate Fleet setup`,
    options: {
      tags: ['oas-tag:Fleet internals']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => FleetSetupResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        },
        500: {
          body: _errors.internalErrorResponse
        }
      }
    }
  }, _handlers.fleetSetupHandler);
};
exports.registerFleetSetupRoute = registerFleetSetupRoute;
const GetAgentsSetupResponseSchema = exports.GetAgentsSetupResponseSchema = _configSchema.schema.object({
  isReady: _configSchema.schema.boolean(),
  missing_requirements: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal('security_required'), _configSchema.schema.literal('tls_required'), _configSchema.schema.literal('api_keys'), _configSchema.schema.literal('fleet_admin_user'), _configSchema.schema.literal('fleet_server')])),
  missing_optional_features: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal('encrypted_saved_object_encryption_key_required')])),
  package_verification_key_id: _configSchema.schema.maybe(_configSchema.schema.string()),
  is_space_awareness_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_secrets_storage_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_action_secrets_storage_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean())
}, {
  meta: {
    description: 'A summary of the agent setup status. `isReady` indicates whether the setup is ready. If the setup is not ready, `missing_requirements` lists which requirements are missing.'
  }
});

// That route is used by agent to setup Fleet
const registerCreateFleetSetupRoute = router => {
  router.versioned.post({
    path: _constants.AGENTS_SETUP_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Initiate agent setup`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => FleetSetupResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.fleetSetupHandler);
};
exports.registerCreateFleetSetupRoute = registerCreateFleetSetupRoute;
const registerGetFleetStatusRoute = router => {
  router.versioned.get({
    path: _constants.AGENTS_SETUP_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get agent setup info`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => GetAgentsSetupResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getFleetStatusHandler);
};
exports.registerGetFleetStatusRoute = registerGetFleetStatusRoute;
const registerRoutes = (router, config) => {
  // Ingest manager setup
  registerFleetSetupRoute(router);
  if (!config.agents.enabled) {
    return;
  }

  // Get Fleet setup
  registerGetFleetStatusRoute(router);

  // Create Fleet setup
  registerCreateFleetSetupRoute(router);
};
exports.registerRoutes = registerRoutes;