"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQLStatsQueryMeta = exports.constructCascadeQuery = void 0;
exports.mutateQueryStatsGrouping = mutateQueryStatsGrouping;
var _esqlAst = require("@kbn/esql-ast");
var _extract_categorize_tokens = require("./extract_categorize_tokens");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// list of stats functions we support for grouping in the cascade experience
const SUPPORTED_STATS_COMMAND_OPTION_FUNCTIONS = ['categorize'];
const isSupportedStatsFunction = fnName => SUPPORTED_STATS_COMMAND_OPTION_FUNCTIONS.includes(fnName);

// helper for removing backticks from field names of function names
const removeBackticks = str => str.replace(/`/g, '');
function getStatsCommandToOperateOn(esqlQuery) {
  if (esqlQuery.errors.length) {
    return null;
  }
  const statsCommands = Array.from(_esqlAst.mutate.commands.stats.list(esqlQuery.ast));
  if (statsCommands.length === 0) {
    return null;
  }
  let summarizedStatsCommand = null;

  // accounting for the possibility of multiple stats commands in the query,
  // we always want to operate on the last stats command that has valid grouping options
  for (let i = statsCommands.length - 1; i >= 0; i--) {
    summarizedStatsCommand = _esqlAst.mutate.commands.stats.summarizeCommand(esqlQuery, statsCommands[i]);
    if (summarizedStatsCommand.grouping && Object.keys(summarizedStatsCommand.grouping).length) {
      break;
    }
  }
  return summarizedStatsCommand;
}
function getESQLQueryDataSourceCommand(esqlQuery) {
  return _esqlAst.mutate.generic.commands.find(esqlQuery.ast, cmd => cmd.name === 'from' || cmd.name === 'ts');
}

/**
 * Returns the summary of the stats command at the given command index in the esql query
 */
function getStatsCommandAtIndexSummary(esqlQuery, commandIndex) {
  const declarationCommand = _esqlAst.mutate.commands.stats.byIndex(esqlQuery.ast, commandIndex);
  if (!declarationCommand) {
    return null;
  }
  return _esqlAst.mutate.commands.stats.summarizeCommand(esqlQuery, declarationCommand);
}
const getESQLStatsQueryMeta = queryString => {
  const groupByFields = [];
  const appliedFunctions = [];
  const esqlQuery = _esqlAst.EsqlQuery.fromSrc(queryString);
  const summarizedStatsCommand = getStatsCommandToOperateOn(esqlQuery);
  if (!summarizedStatsCommand) {
    return {
      groupByFields,
      appliedFunctions
    };
  }

  // get all the new fields created by the stats commands in the query,
  // so we might tell if the command we are operating on is referencing a field that was defined by a preceding command
  const queryRuntimeFields = Array.from(_esqlAst.mutate.commands.stats.summarize(esqlQuery)).map(command => command.newFields);
  const grouping = Object.values(summarizedStatsCommand.grouping);
  for (let j = 0; j < grouping.length; j++) {
    const group = grouping[j];
    if (!group.definition) {
      // query received is malformed without complete grouping definition, there's no need to proceed further
      return {
        groupByFields: [],
        appliedFunctions: []
      };
    }
    const groupFieldName = removeBackticks(group.field);
    let groupFieldNode = group;
    const groupDeclarationCommandIndex = queryRuntimeFields.findIndex(field => field.has(groupFieldName));
    let groupDeclarationCommandSummary = null;
    if (groupDeclarationCommandIndex !== -1 && (groupDeclarationCommandSummary = getStatsCommandAtIndexSummary(esqlQuery, groupDeclarationCommandIndex))) {
      // update the group field node to it's actual definition
      groupFieldNode = groupDeclarationCommandSummary.grouping[groupFieldName];
    }

    // check if there is a where command targeting the group field in the stats command
    const whereCommandGroupFieldSearch = _esqlAst.mutate.commands.where.byField(esqlQuery.ast, _esqlAst.Builder.expression.column({
      args: [_esqlAst.Builder.identifier({
        name: groupFieldName
      })]
    }));
    if (whereCommandGroupFieldSearch !== null && whereCommandGroupFieldSearch !== void 0 && whereCommandGroupFieldSearch.length) {
      if (groupByFields.length > 0) {
        // if there's a where command targeting the group in this current iteration,
        // then this specific query can only be grouped by the current group, pivoting on any other columns though they exist
        // in the query would be invalid, hence we clear out any previously added group by fields since they are no longer valid
        groupByFields.splice(0, groupByFields.length);
      }

      // add the current group and break out of the loop
      // since there's no need to continue processing other groups
      // as they are not valid in this context
      groupByFields.push({
        field: groupFieldName,
        type: groupFieldNode.definition.type === 'function' ? groupFieldNode.definition.name : groupFieldNode.definition.type
      });
      break;
    }
    if ((0, _esqlAst.isFunctionExpression)(groupFieldNode.definition)) {
      const functionName = groupFieldNode.definition.name;
      if (!isSupportedStatsFunction(functionName)) {
        continue;
      }
    }
    groupByFields.push({
      field: groupFieldName,
      type: groupFieldNode.definition.type === 'function' ? groupFieldNode.definition.name : groupFieldNode.definition.type
    });
  }
  Object.values(summarizedStatsCommand.aggregates).forEach(aggregate => {
    var _operator$name, _operator;
    appliedFunctions.push({
      identifier: removeBackticks(aggregate.field),
      // we remove backticks to have a clean identifier that gets displayed in the UI
      aggregation: (_operator$name = (_operator = aggregate.definition.operator) === null || _operator === void 0 ? void 0 : _operator.name) !== null && _operator$name !== void 0 ? _operator$name : aggregate.definition.text
    });
  });
  return {
    groupByFields,
    appliedFunctions
  };
};
exports.getESQLStatsQueryMeta = getESQLStatsQueryMeta;
/**
 * Constructs a cascade query from the provided query, based on the node type, node path and node path map.
 */
const constructCascadeQuery = ({
  query,
  nodeType,
  nodePath,
  nodePathMap
}) => {
  const EditorESQLQuery = _esqlAst.EsqlQuery.fromSrc(query.esql);
  if (EditorESQLQuery.errors.length) {
    throw new Error('Query is malformed');
  }
  const dataSourceCommand = getESQLQueryDataSourceCommand(EditorESQLQuery);
  if (!dataSourceCommand) {
    throw new Error('Query does not have a data source');
  }
  const summarizedStatsCommand = getStatsCommandToOperateOn(EditorESQLQuery);
  if (!summarizedStatsCommand) {
    throw new Error('Query does not have a valid stats command with grouping options');
  }
  const queryRuntimeFields = Array.from(_esqlAst.mutate.commands.stats.summarize(EditorESQLQuery)).map(command => command.newFields);
  if (nodeType === 'leaf') {
    var _grouping$pathSegment, _groupDeclarationComm, _groupDeclarationComm2;
    const pathSegment = nodePath[nodePath.length - 1];
    const groupDeclarationCommandIndex = queryRuntimeFields.findIndex(field => field.has(pathSegment));
    let groupDeclarationCommandSummary = null;
    if (groupDeclarationCommandIndex !== -1) {
      groupDeclarationCommandSummary = getStatsCommandAtIndexSummary(EditorESQLQuery, groupDeclarationCommandIndex);
    }
    const groupValue = (_grouping$pathSegment = ((_groupDeclarationComm = groupDeclarationCommandSummary) !== null && _groupDeclarationComm !== void 0 ? _groupDeclarationComm : summarizedStatsCommand).grouping[pathSegment]) !== null && _grouping$pathSegment !== void 0 ? _grouping$pathSegment :
    // when a column name is not assigned, one is created automatically that includes backticks
    ((_groupDeclarationComm2 = groupDeclarationCommandSummary) !== null && _groupDeclarationComm2 !== void 0 ? _groupDeclarationComm2 : summarizedStatsCommand).grouping[`\`${pathSegment}\``];
    const isOperable = groupValue && nodePathMap[pathSegment];
    if (isOperable && (0, _esqlAst.isColumn)(groupValue.definition)) {
      return handleStatsByColumnLeafOperation(dataSourceCommand, {
        [pathSegment]: nodePathMap[pathSegment]
      });
    } else if (isOperable && (0, _esqlAst.isFunctionExpression)(groupValue.definition)) {
      switch (groupValue.definition.name) {
        case 'categorize':
          {
            return handleStatsByCategorizeLeafOperation(dataSourceCommand, groupValue, nodePathMap);
          }
        default:
          {
            throw new Error(`The "${groupValue.definition.name}" function is not supported for leaf node operations`);
          }
      }
    }
  } else if (nodeType === 'group') {
    throw new Error('Group node operations are not yet supported');
  }
};

/**
 * @description adds a where command with current value for a matched column option as a side-effect on the passed query,
 * helps us with fetching leaf node data for stats operation in the data cascade experience.
 */
exports.constructCascadeQuery = constructCascadeQuery;
function handleStatsByColumnLeafOperation(dataSourceCommand, columnInterpolationRecord) {
  // create new query which we will modify to contain the valid query for the cascade experience
  const cascadeOperationQuery = _esqlAst.EsqlQuery.fromSrc('');

  // set data source for the new query
  _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, dataSourceCommand);
  const newCommands = Object.entries(columnInterpolationRecord).map(([key, value]) => {
    return _esqlAst.Builder.command({
      name: 'where',
      args: [_esqlAst.Builder.expression.func.binary('==', [_esqlAst.Builder.expression.column({
        args: [_esqlAst.Builder.identifier({
          name: key
        })]
      }), _esqlAst.Builder.expression.literal.string(value)])]
    });
  });
  newCommands.forEach(command => {
    _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, command);
  });
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(cascadeOperationQuery.ast)
  };
}

/**
 * Handles the stats command for a leaf operation that contains a categorize function by modifying the query and adding necessary commands.
 */
function handleStatsByCategorizeLeafOperation(dataSourceCommand, categorizeCommand, nodePathMap) {
  // create new query which we will modify to contain the valid query for the cascade experience
  const cascadeOperationQuery = _esqlAst.EsqlQuery.fromSrc('');

  // set data source for the new query
  _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, dataSourceCommand);

  // build a where command with match expressions for the selected categorize function
  const categorizeWhereCommand = _esqlAst.Builder.command({
    name: 'where',
    args: categorizeCommand.definition.args.map(arg => {
      const namedColumn = categorizeCommand.column.name;
      const matchValue = nodePathMap[removeBackticks(namedColumn)];
      if (!matchValue) {
        return null;
      }
      return _esqlAst.Builder.expression.func.call('match', [_esqlAst.Builder.identifier({
        name: arg.text
      }), _esqlAst.Builder.expression.literal.string((0, _extract_categorize_tokens.extractCategorizeTokens)(matchValue).join(' ')), _esqlAst.Builder.expression.map({
        entries: [_esqlAst.Builder.expression.entry('auto_generate_synonyms_phrase_query', _esqlAst.Builder.expression.literal.boolean(false)), _esqlAst.Builder.expression.entry('fuzziness', _esqlAst.Builder.expression.literal.integer(0)), _esqlAst.Builder.expression.entry('operator', _esqlAst.Builder.expression.literal.string('AND'))]
      })]);
    }).filter(Boolean)
  });
  _esqlAst.mutate.generic.commands.append(cascadeOperationQuery.ast, categorizeWhereCommand);
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(cascadeOperationQuery.ast)
  };
}

/**
 * Modifies the provided ESQL query to only include the specified columns in the stats by option.
 */
function mutateQueryStatsGrouping(query, pick) {
  const EditorESQLQuery = _esqlAst.EsqlQuery.fromSrc(query.esql);
  const dataSourceCommand = getESQLQueryDataSourceCommand(EditorESQLQuery);
  if (!dataSourceCommand) {
    throw new Error('Query does not have a data source');
  }
  const statsCommands = Array.from(_esqlAst.mutate.commands.stats.list(EditorESQLQuery.ast));
  if (statsCommands.length === 0) {
    throw new Error(`Query does not include a "stats" command`);
  }
  let statsCommandToOperateOn = null;
  let statsCommandToOperateOnGrouping = null;

  // accounting for the possibility of multiple stats commands in the query,
  // we always want to operate on the last stats command that has valid grouping options
  for (let i = statsCommands.length - 1; i >= 0; i--) {
    ({
      grouping: statsCommandToOperateOnGrouping
    } = _esqlAst.mutate.commands.stats.summarizeCommand(EditorESQLQuery, statsCommands[i]));
    if (statsCommandToOperateOnGrouping && Object.keys(statsCommandToOperateOnGrouping).length) {
      statsCommandToOperateOn = statsCommands[i];
      break;
    }
  }
  if (!statsCommandToOperateOn) {
    throw new Error(`No valid "stats" command was found in the query`);
  }
  const isValidPick = pick.every(col => Object.keys(statsCommandToOperateOnGrouping).includes(col) || Object.keys(statsCommandToOperateOnGrouping).includes(`\`${col}\``));
  if (!isValidPick) {
    // nothing to do, return query as is
    return {
      esql: _esqlAst.BasicPrettyPrinter.print(EditorESQLQuery.ast)
    };
  }

  // Create a modified stats command with only the specified column as args for the "by" option
  const modifiedStatsCommand = _esqlAst.Builder.command({
    name: 'stats',
    args: statsCommandToOperateOn.args.map(statsCommandArg => {
      if ((0, _esqlAst.isOptionNode)(statsCommandArg) && statsCommandArg.name === 'by') {
        return _esqlAst.Builder.option({
          name: statsCommandArg.name,
          args: statsCommandArg.args.reduce((acc, cur) => {
            var _find$name, _find, _cur$args$find$name, _cur$args$find;
            if ((0, _esqlAst.isColumn)(cur) && pick.includes(removeBackticks(cur.name))) {
              acc.push(_esqlAst.Builder.expression.column({
                args: [_esqlAst.Builder.identifier({
                  name: cur.name
                })]
              }));
            } else if ((0, _esqlAst.isFunctionExpression)(cur) && isSupportedStatsFunction(cur.subtype === 'variadic-call' ? cur.name : (_find$name = (_find = cur.args[1].find(_esqlAst.isFunctionExpression)) === null || _find === void 0 ? void 0 : _find.name) !== null && _find$name !== void 0 ? _find$name : '') && pick.includes(cur.subtype === 'variadic-call' ? cur.text : removeBackticks((_cur$args$find$name = (_cur$args$find = cur.args.find(_esqlAst.isColumn)) === null || _cur$args$find === void 0 ? void 0 : _cur$args$find.name) !== null && _cur$args$find$name !== void 0 ? _cur$args$find$name : ''))) {
              acc.push(_esqlAst.synth.exp(cur.text, {
                withFormatting: false
              }));
            }
            return acc;
          }, [])
        });
      }

      // leverage synth to clone the rest of the args since we'd want to use those parts as is
      return _esqlAst.synth.exp(statsCommandArg.text, {
        withFormatting: false
      });
    })
  });

  // Get the position of the original stats command
  const statsCommandIndex = EditorESQLQuery.ast.commands.findIndex(cmd => cmd === statsCommandToOperateOn);

  // remove stats command
  _esqlAst.mutate.generic.commands.remove(EditorESQLQuery.ast, statsCommandToOperateOn);

  // insert modified stats command at same position previous one was at
  _esqlAst.mutate.generic.commands.insert(EditorESQLQuery.ast, modifiedStatsCommand, statsCommandIndex);
  return {
    esql: _esqlAst.BasicPrettyPrinter.print(EditorESQLQuery.ast)
  };
}