"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CstToAstConverter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var antlr = _interopRequireWildcard(require("antlr4"));
var cst = _interopRequireWildcard(require("../antlr/esql_parser"));
var _is = require("../ast/is");
var _pretty_print = require("../pretty_print");
var _tokens = require("./tokens");
var _helpers = require("./helpers");
var _utils = require("../visitor/utils");
var _builder = require("../builder");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const textExistsAndIsValid = text => !!(text && !/<missing /.test(text));
/**
 * Transforms an ANTLR ES|QL Concrete Syntax Tree (CST) into a
 * Kibana Abstract Syntax Tree (AST).
 *
 * Most of the methods in this class are 1-to-1 mapping from CST-to-AST,
 * they are designed to convert specific CST nodes into their
 * corresponding AST nodes.
 */
class CstToAstConverter {
  constructor(parser) {
    // ------------------------------------------------------------- CHANGE_POINT
    (0, _defineProperty2.default)(this, "fromChangePointCommand", ctx => {
      const value = this.toColumn(ctx._value);
      const command = this.createCommand('change_point', ctx, {
        value
      });
      command.args.push(value);
      if (ctx._key && ctx._key.getText()) {
        const key = this.toColumn(ctx._key);
        const option = _builder.Builder.option({
          name: 'on',
          args: [key]
        }, {
          location: (0, _tokens.getPosition)(ctx.ON().symbol, ctx._key.stop)
        });
        command.key = key;
        command.args.push(option);
      }
      if (ctx._targetType && ctx._targetPvalue) {
        const type = this.toColumn(ctx._targetType);
        const pvalue = this.toColumn(ctx._targetPvalue);
        const option = _builder.Builder.option({
          name: 'as',
          args: [type, pvalue]
        }, {
          location: (0, _tokens.getPosition)(ctx.AS().symbol, ctx._targetPvalue.stop)
        });
        command.target = {
          type,
          pvalue
        };
        command.args.push(option);
      }
      return command;
    });
    this.parser = parser;
  }

  // -------------------------------------------------------------------- utils

  getParserFields(ctx) {
    return {
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception)
    };
  }
  createParserFieldsFromToken(token, text = token.text) {
    const fields = {
      text,
      location: (0, _tokens.getPosition)(token, token),
      incomplete: false
    };
    return fields;
  }
  toIdentifierFromTerminalNode(node) {
    return this.toIdentifierFromToken(node.symbol);
  }
  toIdentifierFromToken(token) {
    const name = token.text;
    return _builder.Builder.identifier({
      name
    }, this.createParserFieldsFromToken(token));
  }
  fromParserRuleToUnknown(ctx) {
    return {
      type: 'unknown',
      name: 'unknown',
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception)
    };
  }

  // TODO: Rename this.
  computeLocationExtends(fn) {
    const location = fn.location;
    if (fn.args) {
      // get min location navigating in depth keeping the left/first arg
      location.min = this.walkFunctionStructure(fn.args, location, 'min', () => 0);
      // get max location navigating in depth keeping the right/last arg
      location.max = this.walkFunctionStructure(fn.args, location, 'max', args => args.length - 1);
      // in case of empty array as last arg, bump the max location by 3 chars (empty brackets)
      if (Array.isArray(fn.args[fn.args.length - 1]) && !fn.args[fn.args.length - 1].length) {
        location.max += 3;
      }
    }
    return location;
  }

  /**
   * @todo Replace this by `Walker`, if necessary, or remove completely.
   */
  walkFunctionStructure(args, initialLocation, prop, getNextItemIndex) {
    let nextArg = args[getNextItemIndex(args)];
    const location = {
      ...initialLocation
    };
    while (Array.isArray(nextArg) || nextArg) {
      if (Array.isArray(nextArg)) {
        nextArg = nextArg[getNextItemIndex(nextArg)];
      } else {
        location[prop] = Math[prop](location[prop], nextArg.location[prop]);
        if (nextArg.type === 'function') {
          nextArg = nextArg.args[getNextItemIndex(nextArg.args)];
        } else {
          nextArg = undefined;
        }
      }
    }
    return location[prop];
  }

  /**
   * Follow a similar logic to the ES one:
   * * remove backticks at the beginning and at the end
   * * remove double backticks
   */
  safeBackticksRemoval(text) {
    return (text === null || text === void 0 ? void 0 : text.replace(/^`{1}|`{1}$/g, '').replace(/``/g, '`')) || '';
  }
  sanitizeIdentifierString(ctx) {
    const result = this.safeBackticksRemoval(ctx.getText());
    // TODO - understand why <missing null> is now returned as the match text for the FROM command
    return result === '<missing null>' ? '' : result;
  }

  // -------------------------------------------------------------------- query

  fromStatements(ctx) {
    const setCommandCtxs = ctx.setCommand_list();
    const singleStatement = ctx.singleStatement();
    let header;
    // Process SET instructions and create header if they exist
    if (setCommandCtxs && setCommandCtxs.length > 0) {
      header = this.fromSetCommands(setCommandCtxs);
    }

    // Get the main query from singleStatement
    const query = this.fromSingleStatement(singleStatement);
    if (!query) {
      const emptyQuery = _builder.Builder.expression.query([], this.getParserFields(ctx), header);
      emptyQuery.incomplete = true;
      return emptyQuery;
    }
    query.header = header;
    return query;
  }
  fromSingleStatement(ctx) {
    if (!ctx) return undefined;
    return this.fromAnyQuery(ctx.query());
  }
  fromAnyQuery(ctx) {
    if (ctx instanceof cst.CompositeQueryContext) {
      return this.fromCompositeQuery(ctx);
    } else {
      if (ctx instanceof cst.QueryContext) {
        return this.fromQuery(ctx);
      } else {
        return undefined;
      }
    }
  }
  fromCompositeQuery(ctx) {
    const query = this.fromAnyQuery(ctx.query());
    if (!query) {
      return;
    }
    const processingCommandCtx = ctx.processingCommand();
    const processingCommand = this.fromProcessingCommand(processingCommandCtx);
    if (processingCommand) {
      query.commands.push(processingCommand);
    }
    return query;
  }
  fromSingleCommandQuery(ctx) {
    return this.fromSourceCommand(ctx.sourceCommand());
  }
  fromQuery(ctx) {
    const children = ctx.children;
    if (!children) {
      return;
    }
    const length = children.length;
    if (!length) {
      return;
    }
    const commands = [];
    for (let i = 0; i < length; i++) {
      const childCtx = children[i];
      const child = this.fromAny(childCtx);
      if ((0, _is.isCommand)(child)) {
        commands.push(child);
      }
    }
    return _builder.Builder.expression.query(commands, this.getParserFields(ctx));
  }
  fromAny(ctx) {
    if (ctx instanceof cst.SingleCommandQueryContext) {
      return this.fromSingleCommandQuery(ctx);
    } else if (ctx instanceof cst.SourceCommandContext) {
      return this.fromSourceCommand(ctx);
    } else if (ctx instanceof cst.ProcessingCommandContext) {
      return this.fromProcessingCommand(ctx);
    }
    return undefined;
  }
  fromSubqueryExpression(ctx) {
    const queryCtx = ctx.query();
    if (queryCtx instanceof cst.QueryContext) {
      return this.fromQuery(queryCtx);
    } else {
      return undefined;
    }
  }

  // ------------------------------------------------------------- query header

  fromSetCommands(setCommandCtxs) {
    const setCommands = setCommandCtxs.map(setCommandCtx => this.fromSetCommand(setCommandCtx)).filter(_helpers.nonNullable);
    return setCommands;
  }

  // ---------------------------------------------------------------------- SET

  fromSetCommand(ctx) {
    const setFieldCtx = ctx.setField();
    const binaryExpression = this.fromSetFieldContext(setFieldCtx);
    const args = binaryExpression ? [binaryExpression] : [];
    const command = _builder.Builder.header.command.set(args, {}, this.getParserFields(ctx));
    return command;
  }
  fromSetFieldContext(ctx) {
    const leftCtx = ctx.identifier();
    const rightCtx = ctx.constant();
    if (!leftCtx || !rightCtx) {
      return null;
    }
    const left = this.toIdentifierFromContext(leftCtx);
    const right = this.fromConstant(rightCtx);
    const expression = this.toBinaryExpression('=', ctx, [left, right]);
    if (left.incomplete || right.incomplete) {
      expression.incomplete = true;
    }
    return expression;
  }
  toIdentifierFromContext(ctx) {
    const identifierToken = ctx.UNQUOTED_IDENTIFIER() || ctx.QUOTED_IDENTIFIER();
    if (identifierToken) {
      return this.toIdentifierFromTerminalNode(identifierToken);
    }

    // Fallback: create identifier from the full context text
    return _builder.Builder.identifier({
      name: ctx.getText()
    }, this.getParserFields(ctx));
  }

  // ----------------------------------------------------------------- commands

  fromSourceCommand(ctx) {
    const fromCommandCtx = ctx.fromCommand();
    if (fromCommandCtx) {
      return this.fromFromCommand(fromCommandCtx);
    }
    const rowCommandCtx = ctx.rowCommand();
    if (rowCommandCtx) {
      return this.fromRowCommand(rowCommandCtx);
    }
    const tsCommandCtx = ctx.timeSeriesCommand();
    if (tsCommandCtx) {
      return this.fromTimeseriesCommand(tsCommandCtx);
    }
    const explainCommandCtx = ctx.explainCommand();
    if (explainCommandCtx) {
      return this.fromExplainCommand(explainCommandCtx);
    }
    const showCommandCtx = ctx.showCommand();
    if (showCommandCtx) {
      return this.fromShowCommand(showCommandCtx);
    }

    // throw new Error(`Unknown source command: ${this.getSrc(ctx)}`);
  }
  fromProcessingCommand(ctx) {
    const limitCommandCtx = ctx.limitCommand();
    if (limitCommandCtx) {
      return this.fromLimitCommand(limitCommandCtx);
    }
    const evalCommandCtx = ctx.evalCommand();
    if (evalCommandCtx) {
      return this.fromEvalCommand(evalCommandCtx);
    }
    const whereCommandCtx = ctx.whereCommand();
    if (whereCommandCtx) {
      return this.fromWhereCommand(whereCommandCtx);
    }
    const keepCommandCtx = ctx.keepCommand();
    if (keepCommandCtx) {
      return this.fromKeepCommand(keepCommandCtx);
    }
    const statsCommandCtx = ctx.statsCommand();
    if (statsCommandCtx) {
      return this.fromStatsCommand(statsCommandCtx);
    }
    const sortCommandCtx = ctx.sortCommand();
    if (sortCommandCtx) {
      return this.fromSortCommand(sortCommandCtx);
    }
    const dropCommandCtx = ctx.dropCommand();
    if (dropCommandCtx) {
      return this.fromDropCommand(dropCommandCtx);
    }
    const renameCommandCtx = ctx.renameCommand();
    if (renameCommandCtx) {
      return this.fromRenameCommand(renameCommandCtx);
    }
    const dissectCommandCtx = ctx.dissectCommand();
    if (dissectCommandCtx) {
      return this.fromDissectCommand(dissectCommandCtx);
    }
    const grokCommandCtx = ctx.grokCommand();
    if (grokCommandCtx) {
      return this.fromGrokCommand(grokCommandCtx);
    }
    const enrichCommandCtx = ctx.enrichCommand();
    if (enrichCommandCtx) {
      return this.fromEnrichCommand(enrichCommandCtx);
    }
    const mvExpandCommandCtx = ctx.mvExpandCommand();
    if (mvExpandCommandCtx) {
      return this.fromMvExpandCommand(mvExpandCommandCtx);
    }
    const joinCommandCtx = ctx.joinCommand();
    if (joinCommandCtx) {
      return this.fromJoinCommand(joinCommandCtx);
    }
    const changePointCommandCtx = ctx.changePointCommand();
    if (changePointCommandCtx) {
      return this.fromChangePointCommand(changePointCommandCtx);
    }
    const completionCommandCtx = ctx.completionCommand();
    if (completionCommandCtx) {
      return this.fromCompletionCommand(completionCommandCtx);
    }
    const sampleCommandCtx = ctx.sampleCommand();
    if (sampleCommandCtx) {
      return this.fromSampleCommand(sampleCommandCtx);
    }
    const inlinestatsCommandCtx = ctx.inlineStatsCommand();
    if (inlinestatsCommandCtx) {
      return this.fromInlinestatsCommand(inlinestatsCommandCtx);
    }
    const rerankCommandCtx = ctx.rerankCommand();
    if (rerankCommandCtx) {
      return this.fromRerankCommand(rerankCommandCtx);
    }
    const fuseCommandCtx = ctx.fuseCommand();
    if (fuseCommandCtx) {
      return this.fromFuseCommand(fuseCommandCtx);
    }
    const forkCommandCtx = ctx.forkCommand();
    if (forkCommandCtx) {
      return this.fromForkCommand(forkCommandCtx);
    }

    // throw new Error(`Unknown processing command: ${this.getSrc(ctx)}`);
  }
  createCommand(name, ctx, partial) {
    const parserFields = this.getParserFields(ctx);
    const command = _builder.Builder.command({
      name,
      args: []
    }, parserFields);
    if (partial) {
      Object.assign(command, partial);
    }
    return command;
  }
  toOption(name, ctx, args = [], incomplete) {
    var _ctx$children;
    return {
      type: 'option',
      name,
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      args,
      incomplete: incomplete !== null && incomplete !== void 0 ? incomplete : Boolean(ctx.exception || ((_ctx$children = ctx.children) === null || _ctx$children === void 0 ? void 0 : _ctx$children.some(c => {
        // TODO: 1. Remove this expect error comment
        // TODO: 2. .isErrorNode is function: .isErrorNode()
        // @ts-expect-error not exposed in type but exists see https://github.com/antlr/antlr4/blob/v4.11.1/runtime/JavaScript/src/antlr4/tree/ErrorNodeImpl.js#L19
        return Boolean(c.isErrorNode);
      })))
    };
  }

  // ------------------------------------------------------------------ EXPLAIN

  fromExplainCommand(ctx) {
    const command = this.createCommand('explain', ctx);
    const arg = this.fromSubqueryExpression(ctx.subqueryExpression());
    if (arg) {
      command.args.push(arg);
    }
    return command;
  }

  // --------------------------------------------------------------------- FROM

  fromFromCommand(ctx) {
    return this.fromFromCompatibleCommand('from', ctx);
  }
  fromFromCompatibleCommand(commandName, ctx) {
    const command = this.createCommand(commandName, ctx);
    const indexPatternCtx = ctx.indexPatternAndMetadataFields();
    const metadataCtx = indexPatternCtx.metadata();
    const sources = indexPatternCtx.getTypedRuleContexts(cst.IndexPatternContext).map(sourceCtx => this.toSource(sourceCtx));
    command.args.push(...sources);
    if (metadataCtx && metadataCtx.METADATA()) {
      const name = metadataCtx.METADATA().getText().toLowerCase();
      const option = this.toOption(name, metadataCtx);
      const optionArgs = this.toColumnsFromCommand(metadataCtx);
      option.args.push(...optionArgs);
      command.args.push(option);
    }
    return command;
  }

  // ---------------------------------------------------------------------- ROW

  fromRowCommand(ctx) {
    const command = this.createCommand('row', ctx);
    const fields = this.fromFields(ctx.fields());
    command.args.push(...fields);
    return command;
  }

  // ----------------------------------------------------------------------- TS

  fromTimeseriesCommand(ctx) {
    return this.fromFromCompatibleCommand('ts', ctx);
  }

  // --------------------------------------------------------------------- SHOW

  fromShowCommand(ctx) {
    const command = this.createCommand('show', ctx);
    if (ctx instanceof cst.ShowInfoContext) {
      const infoCtx = ctx;
      const arg = this.toIdentifierFromTerminalNode(infoCtx.INFO());
      arg.name = arg.name.toUpperCase();
      command.args.push(arg);
    }
    return command;
  }

  // -------------------------------------------------------------------- LIMIT

  fromLimitCommand(ctx) {
    const command = this.createCommand('limit', ctx);
    if (ctx.constant()) {
      const limitValue = this.fromConstant(ctx.constant());
      if (limitValue != null) {
        command.args.push(limitValue);
      }
    }
    return command;
  }

  // --------------------------------------------------------------------- EVAL

  fromEvalCommand(ctx) {
    const command = this.createCommand('eval', ctx);
    const fields = this.fromFields(ctx.fields());
    command.args.push(...fields);
    return command;
  }

  // -------------------------------------------------------------------- WHERE

  fromWhereCommand(ctx) {
    const command = this.createCommand('where', ctx);
    const expressions = this.collectBooleanExpression(ctx.booleanExpression());
    command.args.push(expressions[0]);
    return command;
  }

  // --------------------------------------------------------------------- KEEP

  fromKeepCommand(ctx) {
    const args = this.fromQualifiedNamePatterns(ctx.qualifiedNamePatterns());
    const command = this.createCommand('keep', ctx, {
      args
    });
    return command;
  }
  fromQualifiedNamePatterns(ctx) {
    const itemCtxs = ctx.qualifiedNamePattern_list();
    const result = [];
    for (const itemCtx of itemCtxs) {
      const node = this.fromQualifiedNamePattern(itemCtx);
      result.push(node);
    }
    return result;
  }

  // -------------------------------------------------------------------- STATS

  fromStatsCommand(ctx) {
    return this.fromStatsLikeCommand('stats', ctx);
  }
  fromStatsLikeCommand(name, ctx) {
    const command = this.createCommand(name, ctx);
    if (ctx._stats) {
      command.args.push(...this.fromAggFields(ctx.aggFields()));
    }
    if (ctx._grouping) {
      const option = this.toByOption(ctx, ctx.fields());
      if (option) {
        command.args.push(option);
      }
    }
    return command;
  }
  fromAggField(ctx) {
    var _firstItem$location$m, _firstItem, _firstItem$location, _firstItem$location$m2, _firstItem2, _firstItem2$location;
    const fieldCtx = ctx.field();
    const field = this.fromField(fieldCtx);
    if (!field) {
      return;
    }
    const booleanExpression = ctx.booleanExpression();
    if (!booleanExpression) {
      return field;
    }
    const condition = this.collectBooleanExpression(booleanExpression)[0];
    const aggField = _builder.Builder.expression.where([field, condition], {}, {
      location: {
        min: (_firstItem$location$m = (_firstItem = (0, _utils.firstItem)([(0, _utils.resolveItem)(field)])) === null || _firstItem === void 0 ? void 0 : (_firstItem$location = _firstItem.location) === null || _firstItem$location === void 0 ? void 0 : _firstItem$location.min) !== null && _firstItem$location$m !== void 0 ? _firstItem$location$m : 0,
        max: (_firstItem$location$m2 = (_firstItem2 = (0, _utils.firstItem)([(0, _utils.resolveItem)(condition)])) === null || _firstItem2 === void 0 ? void 0 : (_firstItem2$location = _firstItem2.location) === null || _firstItem2$location === void 0 ? void 0 : _firstItem2$location.max) !== null && _firstItem$location$m2 !== void 0 ? _firstItem$location$m2 : 0
      }
    });
    return aggField;
  }
  toByOption(ctx, expr) {
    const byCtx = ctx.BY();
    if (!byCtx || !expr) {
      return;
    }
    const option = this.toOption(byCtx.getText().toLowerCase(), ctx);
    option.args.push(...this.fromFields(expr));
    option.location.min = byCtx.symbol.start;
    const lastArg = (0, _utils.lastItem)(option.args);
    if (lastArg) option.location.max = lastArg.location.max;
    return option;
  }

  // --------------------------------------------------------------------- SORT

  fromSortCommand(ctx) {
    const command = this.createCommand('sort', ctx);
    command.args.push(...this.fromOrderExpressions(ctx.orderExpression_list()));
    return command;
  }
  fromOrderExpressions(ctx) {
    const expressions = [];
    for (const orderCtx of ctx) {
      expressions.push(this.fromOrderExpression(orderCtx));
    }
    return expressions;
  }
  fromOrderExpression(ctx) {
    var _ctx$_ordering, _ctx$_ordering$text, _ctx$_nullOrdering, _ctx$_nullOrdering$te;
    const arg = this.collectBooleanExpression(ctx.booleanExpression())[0];
    let order = '';
    let nulls = '';
    const ordering = (_ctx$_ordering = ctx._ordering) === null || _ctx$_ordering === void 0 ? void 0 : (_ctx$_ordering$text = _ctx$_ordering.text) === null || _ctx$_ordering$text === void 0 ? void 0 : _ctx$_ordering$text.toUpperCase();
    if (ordering) order = ordering;
    const nullOrdering = (_ctx$_nullOrdering = ctx._nullOrdering) === null || _ctx$_nullOrdering === void 0 ? void 0 : (_ctx$_nullOrdering$te = _ctx$_nullOrdering.text) === null || _ctx$_nullOrdering$te === void 0 ? void 0 : _ctx$_nullOrdering$te.toUpperCase();
    switch (nullOrdering) {
      case 'LAST':
        nulls = 'NULLS LAST';
        break;
      case 'FIRST':
        nulls = 'NULLS FIRST';
        break;
    }
    if (!order && !nulls) {
      return arg;
    }
    return _builder.Builder.expression.order(arg, {
      order,
      nulls
    }, this.getParserFields(ctx));
  }

  // --------------------------------------------------------------------- DROP

  fromDropCommand(ctx) {
    const args = this.fromQualifiedNamePatterns(ctx.qualifiedNamePatterns());
    const command = this.createCommand('drop', ctx, {
      args
    });
    return command;
  }

  // ------------------------------------------------------------------- RENAME

  fromRenameCommand(ctx) {
    const command = this.createCommand('rename', ctx);
    const renameArgs = this.fromRenameClauses(ctx.renameClause_list());
    command.args.push(...renameArgs);
    return command;
  }
  fromRenameClauses(clausesCtx) {
    return clausesCtx.map(clause => {
      var _clause$_oldName;
      const asToken = clause.getToken(cst.default.AS, 0);
      const assignToken = clause.getToken(cst.default.ASSIGN, 0);
      const renameToken = asToken || assignToken;
      if (renameToken && textExistsAndIsValid(renameToken.getText())) {
        const renameFunction = this.toFunction(renameToken.getText().toLowerCase(), clause, undefined, 'binary-expression');
        const renameArgsInOrder = asToken ? [clause._oldName, clause._newName] : [clause._newName, clause._oldName];
        for (const arg of renameArgsInOrder) {
          if (textExistsAndIsValid(arg.getText())) {
            renameFunction.args.push(this.toColumn(arg));
          }
        }
        const firstArg = (0, _utils.firstItem)(renameFunction.args);
        const lastArg = (0, _utils.lastItem)(renameFunction.args);
        const location = renameFunction.location;
        if (firstArg) location.min = firstArg.location.min;
        if (lastArg) location.max = lastArg.location.max;
        return renameFunction;
      }
      if (textExistsAndIsValid((_clause$_oldName = clause._oldName) === null || _clause$_oldName === void 0 ? void 0 : _clause$_oldName.getText())) {
        return this.toColumn(clause._oldName);
      }
    }).filter(_helpers.nonNullable);
  }

  // ------------------------------------------------------------------ DISSECT

  fromDissectCommand(ctx) {
    const command = this.createCommand('dissect', ctx);
    const primaryExpression = this.visitPrimaryExpression(ctx.primaryExpression());
    const stringContext = ctx.string_();
    const pattern = stringContext.getToken(cst.default.QUOTED_STRING, 0);
    const doParseStringAndOptions = pattern && textExistsAndIsValid(pattern.getText());
    command.args.push(primaryExpression);
    if (doParseStringAndOptions) {
      const stringNode = this.toStringLiteral(stringContext);
      command.args.push(stringNode);
      command.args.push(...this.fromDissectCommandOptions(ctx.dissectCommandOptions()));
    }
    return command;
  }
  fromDissectCommandOptions(ctx) {
    if (!ctx) {
      return [];
    }
    const options = [];
    for (const optionCtx of ctx.dissectCommandOption_list()) {
      const option = this.toOption(this.sanitizeIdentifierString(optionCtx.identifier()).toLowerCase(), optionCtx);
      options.push(option);
      // it can throw while accessing constant for incomplete commands, so try catch it
      try {
        const optionValue = this.fromConstant(optionCtx.constant());
        if (optionValue != null) {
          option.args.push(optionValue);
        }
      } catch (e) {
        // do nothing here
      }
    }
    return options;
  }

  // --------------------------------------------------------------------- GROK

  fromGrokCommand(ctx) {
    const command = this.createCommand('grok', ctx);
    const primaryExpression = this.visitPrimaryExpression(ctx.primaryExpression());
    command.args.push(primaryExpression);
    const stringContexts = ctx.string__list();
    for (let i = 0; i < stringContexts.length; i++) {
      const stringContext = stringContexts[i];
      const pattern = stringContext.getToken(cst.default.QUOTED_STRING, 0);
      const doParseStringAndOptions = pattern && textExistsAndIsValid(pattern.getText());
      if (doParseStringAndOptions) {
        const stringNode = this.toStringLiteral(stringContext);
        command.args.push(stringNode);
      }
    }
    return command;
  }
  // ------------------------------------------------------------------- ENRICH

  fromEnrichCommand(ctx) {
    const command = this.createCommand('enrich', ctx);
    const policy = this.toPolicyNameFromEnrichCommand(ctx);
    command.args.push(policy);
    if (policy.incomplete) {
      command.incomplete = true;
    }
    const onOption = this.toOnOptionFromEnrichCommand(ctx);
    if (onOption) {
      command.args.push(onOption);
    }
    const withOption = this.toWithOptionFromEnrichCommand(ctx);
    if (withOption) {
      command.args.push(withOption);
    }
    return command;
  }
  toPolicyNameFromEnrichCommand(ctx) {
    const policyNameCtx = ctx._policyName;
    const policyName = (policyNameCtx === null || policyNameCtx === void 0 ? void 0 : policyNameCtx.ENRICH_POLICY_NAME()) || (policyNameCtx === null || policyNameCtx === void 0 ? void 0 : policyNameCtx.QUOTED_STRING());
    if (!policyName || !textExistsAndIsValid(policyName.getText())) {
      const source = _builder.Builder.expression.source.node({
        sourceType: 'policy',
        name: '',
        index: '',
        prefix: ''
      }, {
        incomplete: true,
        text: '',
        location: {
          min: policyNameCtx.start.start,
          max: policyNameCtx.start.stop
        }
      });
      return source;
    }
    const name = policyName.getText();
    const colonIndex = name.indexOf(':');
    const withPrefix = colonIndex !== -1;
    const incomplete = false;
    let index;
    let prefix;
    if (withPrefix) {
      const prefixName = name.substring(0, colonIndex);
      const indexName = name.substring(colonIndex + 1);
      prefix = _builder.Builder.expression.literal.string(prefixName, {
        unquoted: true
      }, {
        text: prefixName,
        incomplete: false,
        location: {
          min: policyNameCtx.start.start,
          max: policyNameCtx.start.start + prefixName.length - 1
        }
      });
      index = _builder.Builder.expression.literal.string(indexName, {
        unquoted: true
      }, {
        text: indexName,
        incomplete: false,
        location: {
          min: policyNameCtx.start.start + prefixName.length + 1,
          max: policyNameCtx.start.stop
        }
      });
    } else {
      index = _builder.Builder.expression.literal.string(name, {
        unquoted: true
      }, {
        text: name,
        incomplete: false,
        location: {
          min: policyNameCtx.start.start,
          max: policyNameCtx.start.stop
        }
      });
    }
    const source = _builder.Builder.expression.source.node({
      sourceType: 'policy',
      name,
      index,
      prefix
    }, {
      incomplete,
      text: name,
      location: {
        min: policyNameCtx.start.start,
        max: policyNameCtx.start.stop
      }
    });
    return source;
  }
  toOnOptionFromEnrichCommand(ctx) {
    if (!ctx._matchField) {
      return undefined;
    }
    const identifier = ctx.qualifiedNamePattern();
    if (identifier) {
      const fn = this.toOption(ctx.ON().getText().toLowerCase(), ctx);
      let max = ctx.ON().symbol.stop;
      if (textExistsAndIsValid(identifier.getText())) {
        const column = this.toColumn(identifier);
        fn.args.push(column);
        max = column.location.max;
      }
      fn.location.min = ctx.ON().symbol.start;
      fn.location.max = max;
      return fn;
    }
    return undefined;
  }
  toWithOptionFromEnrichCommand(ctx) {
    const withCtx = ctx.WITH();
    if (withCtx) {
      const option = this.toOption(withCtx.getText().toLowerCase(), ctx);
      const clauses = ctx.enrichWithClause_list();
      for (const clause of clauses) {
        var _lastArg$location$max, _lastArg$location;
        if (clause._enrichField) {
          const args = [];
          if (clause.ASSIGN()) {
            var _clause$_enrichField;
            args.push(this.toColumn(clause._newName));
            if (textExistsAndIsValid((_clause$_enrichField = clause._enrichField) === null || _clause$_enrichField === void 0 ? void 0 : _clause$_enrichField.getText())) {
              args.push(this.toColumn(clause._enrichField));
            }
          } else {
            var _clause$_enrichField2;
            // if an explicit assign is not set, create a fake assign with
            // both left and right value with the same column
            if (textExistsAndIsValid((_clause$_enrichField2 = clause._enrichField) === null || _clause$_enrichField2 === void 0 ? void 0 : _clause$_enrichField2.getText())) {
              args.push(this.toColumn(clause._enrichField), this.toColumn(clause._enrichField));
            }
          }
          if (args.length) {
            const fn = this.toFunction('=', clause, undefined, 'binary-expression');
            fn.args.push(args[0], args[1] ? [args[1]] : []);
            option.args.push(fn);
          }
        }
        const location = option.location;
        const lastArg = (0, _utils.lastItem)(option.args);
        location.min = withCtx.symbol.start;
        location.max = (_lastArg$location$max = lastArg === null || lastArg === void 0 ? void 0 : (_lastArg$location = lastArg.location) === null || _lastArg$location === void 0 ? void 0 : _lastArg$location.max) !== null && _lastArg$location$max !== void 0 ? _lastArg$location$max : withCtx.symbol.stop;
      }
      return option;
    }
    return undefined;
  }

  // ---------------------------------------------------------------- MV_EXPAND

  fromMvExpandCommand(ctx) {
    const command = this.createCommand('mv_expand', ctx);
    const identifiers = this.toColumnsFromCommand(ctx);
    command.args.push(...identifiers);
    return command;
  }

  // --------------------------------------------------------------------- JOIN

  fromJoinCommand(ctx) {
    var _ctx$_type_$text, _ctx$_type_;
    const command = this.createCommand('join', ctx);

    // Pick-up the <TYPE> of the command.
    command.commandType = ((_ctx$_type_$text = (_ctx$_type_ = ctx._type_) === null || _ctx$_type_ === void 0 ? void 0 : _ctx$_type_.text) !== null && _ctx$_type_$text !== void 0 ? _ctx$_type_$text : 'lookup').toLocaleLowerCase();
    const joinTarget = this.fromJoinTarget(ctx.joinTarget());
    const joinCondition = ctx.joinCondition();
    const onOption = this.toOption('on', joinCondition);
    const joinPredicates = onOption.args;
    for (const joinPredicateCtx of joinCondition.booleanExpression_list()) {
      const expression = this.fromBooleanExpression(joinPredicateCtx);
      if (expression) {
        joinPredicates.push(expression);
      }
    }
    command.args.push(joinTarget);
    if (onOption.args.length) {
      command.args.push(onOption);
    }
    return command;
  }
  fromJoinTarget(ctx) {
    if (ctx._index) {
      return this.toSource(ctx._index);
    } else {
      return _builder.Builder.expression.source.node({
        sourceType: 'index',
        name: ''
      }, {
        location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
        incomplete: true,
        text: ctx === null || ctx === void 0 ? void 0 : ctx.getText()
      });
    }
  }
  // --------------------------------------------------------------- COMPLETION

  fromCompletionCommand(ctx) {
    const command = this.createCommand('completion', ctx);
    if (ctx._targetField && ctx.ASSIGN()) {
      const targetField = this.toColumn(ctx._targetField);
      const prompt = this.visitPrimaryExpression(ctx._prompt);
      command.prompt = prompt;
      const assignment = this.toFunction(ctx.ASSIGN().getText(), ctx, undefined, 'binary-expression');
      assignment.args.push(targetField, prompt);
      // update the location of the assign based on arguments
      assignment.location = this.computeLocationExtends(assignment);
      command.targetField = targetField;
      command.args.push(assignment);
    } else if (ctx._prompt) {
      const prompt = this.visitPrimaryExpression(ctx._prompt);
      command.prompt = prompt;
      command.args.push(prompt);
    } else {
      // When the user is typing a column as prompt i.e: | COMPLETION message^,
      // ANTLR does not know if it is trying to type a prompt
      // or a target field, so it does not return neither _prompt nor _targetField. We fill the AST
      // with an unknown item until the user inserts the next keyword and breaks the tie.
      const unknownItem = this.fromParserRuleToUnknown(ctx);
      unknownItem.text = ctx.getText().replace(/^completion/i, '');
      command.prompt = unknownItem;
      command.args.push(unknownItem);
    }
    let inferenceId = _builder.Builder.expression.literal.string('', {
      name: 'inferenceId'
    }, {
      incomplete: true
    });
    const commandNamedParametersContext = ctx.commandNamedParameters();
    if (commandNamedParametersContext) {
      var _namedParameters$entr;
      const namedParametersOption = this.fromCommandNamedParameters(commandNamedParametersContext);
      const namedParameters = namedParametersOption.args[0];
      const inferenceIdParam = namedParameters === null || namedParameters === void 0 ? void 0 : (_namedParameters$entr = namedParameters.entries.find(param => param.key.valueUnquoted === 'inference_id')) === null || _namedParameters$entr === void 0 ? void 0 : _namedParameters$entr.value;
      if (inferenceIdParam) {
        var _inferenceIdParam$val;
        inferenceId = inferenceIdParam;
        inferenceId.incomplete = ((_inferenceIdParam$val = inferenceIdParam.valueUnquoted) === null || _inferenceIdParam$val === void 0 ? void 0 : _inferenceIdParam$val.length) === 0;
      }
      command.args.push(namedParametersOption);
    }
    command.inferenceId = inferenceId;
    return command;
  }

  // ------------------------------------------------------------------- SAMPLE

  fromSampleCommand(ctx) {
    const command = this.createCommand('sample', ctx);
    if (ctx.constant()) {
      const probability = this.fromConstant(ctx.constant());
      if (probability != null) {
        command.args.push(probability);
      }
    }
    return command;
  }

  // ------------------------------------------------------------- INLINE STATS

  fromInlinestatsCommand(ctx) {
    return this.fromStatsLikeCommand('inline stats', ctx);
  }

  // ------------------------------------------------------------------- RERANK

  /**
   * Supports two syntax forms:
   * - RERANK "query" ON fields WITH {options}
   * - RERANK target = "query" ON fields WITH {options}
   */
  fromRerankCommand(ctx) {
    const command = this.createCommand('rerank', ctx);
    this.parseRerankQuery(ctx, command);
    this.parseRerankOnOption(ctx, command);
    this.parseRerankWithOption(ctx, command);
    return command;
  }

  /**
   * Parses the query text and optional target field assignment.
   * Handles: RERANK [target =] "query" ...
   */
  parseRerankQuery(ctx, command) {
    if (!ctx._queryText) {
      return;
    }
    const queryText = this.fromConstant(ctx._queryText);
    if (!queryText) {
      return;
    }
    command.query = queryText;

    // Handle target field assignment: RERANK target = "query"
    if (ctx._targetField && ctx.ASSIGN()) {
      const targetField = this.toColumn(ctx._targetField);
      const assignment = this.toFunction(ctx.ASSIGN().getText(), ctx, undefined, 'binary-expression');
      assignment.args.push(targetField, queryText);
      assignment.location = this.computeLocationExtends(assignment);
      command.targetField = targetField;
      command.args.push(assignment);
    } else {
      command.args.push(queryText);
    }
  }

  /**
   * Parses the ON fields list.
   * Handles: ... ON field1, field2 = X(field2, 2)
   */
  parseRerankOnOption(ctx, command) {
    const onToken = ctx.ON();
    const rerankFieldsCtx = ctx.rerankFields();
    if (!onToken || !rerankFieldsCtx) {
      return;
    }
    const onOption = this.toOption(onToken.getText().toLowerCase(), rerankFieldsCtx);
    const fields = this.fromRerankFields(rerankFieldsCtx);
    onOption.args.push(...fields);
    onOption.location.min = onToken.symbol.start;
    const lastArg = (0, _utils.lastItem)(onOption.args);
    if (lastArg) {
      onOption.location.max = lastArg.location.max;
    }
    command.args.push(onOption);
    command.fields = fields;
  }

  /**
   * Parses WITH parameters as a generic map.
   * Handles: ... WITH {inference_id: "model", scoreColumn: "score", ...}
   */
  parseRerankWithOption(ctx, command) {
    const namedParamsCtx = ctx.commandNamedParameters();
    if (!namedParamsCtx) {
      return;
    }
    const withOption = this.fromCommandNamedParameters(namedParamsCtx);
    if (!withOption) {
      return;
    }

    // check the existence of inference_id
    const namedParameters = withOption.args[0];
    command.inferenceId = undefined;
    if (namedParameters) {
      var _namedParameters$entr2;
      command.args.push(withOption);
      command.inferenceId = _builder.Builder.expression.literal.string('', {
        name: 'inferenceId'
      }, {
        incomplete: true
      });
      const inferenceIdParam = namedParameters === null || namedParameters === void 0 ? void 0 : (_namedParameters$entr2 = namedParameters.entries.find(param => param.key.valueUnquoted === 'inference_id')) === null || _namedParameters$entr2 === void 0 ? void 0 : _namedParameters$entr2.value;
      if (inferenceIdParam) {
        var _inferenceIdParam$val2;
        command.inferenceId = inferenceIdParam;
        command.inferenceId.incomplete = ((_inferenceIdParam$val2 = inferenceIdParam.valueUnquoted) === null || _inferenceIdParam$val2 === void 0 ? void 0 : _inferenceIdParam$val2.length) === 0;
      }
    }
  }

  /**
   * Collects all ON fields for RERANK.
   *
   * - Accepts simple columns (e.g. `title`).
   * - Accepts assignments (e.g. `title = X(title, 2)`).
   * - Accepts expressions after a qualified name (e.g. `field < 10`).
   */
  fromRerankFields(ctx) {
    const fields = [];
    if (!ctx) {
      return fields;
    }
    try {
      for (const fieldCtx of ctx.rerankField_list()) {
        const field = this.fromRerankField(fieldCtx);
        if (field) {
          fields.push(field);
        }
      }
    } catch (e) {
      // do nothing
    }
    return fields;
  }

  /**
   * Parses a single RERANK field entry.
   *
   * Supports three forms:
   * 1) Assignment: qualifiedName '=' booleanExpression
   * 2) Column only: qualifiedName
   */
  fromRerankField(ctx) {
    try {
      const qualifiedNameCtx = ctx.qualifiedName();
      if (!qualifiedNameCtx) {
        return undefined;
      }

      // 1) field assignment: <col> = <booleanExpression>
      if (ctx.ASSIGN()) {
        const left = this.toColumn(qualifiedNameCtx);
        const assignment = this.toFunction(ctx.ASSIGN().getText(), ctx, undefined, 'binary-expression');
        if (ctx.booleanExpression()) {
          var _ctx$booleanExpressio;
          const right = this.collectBooleanExpression(ctx.booleanExpression());
          // Mark as incomplete if RHS is empty, contains incomplete items, or the parser raised an exception
          const hasItems = right.length > 0;
          const hasIncompleteItem = right.some(item => Array.isArray(item) ? false : !!(item !== null && item !== void 0 && item.incomplete));
          const hasException = !!((_ctx$booleanExpressio = ctx.booleanExpression()) !== null && _ctx$booleanExpressio !== void 0 && _ctx$booleanExpressio.exception);
          if (!hasItems || hasIncompleteItem || hasException) {
            assignment.incomplete = true;
          }
          assignment.args.push(left, right);
          assignment.location = this.computeLocationExtends(assignment);
        } else {
          // User typed something like `ON col0 =` and stopped.
          // Build an assignment with only the left operand, mark it as incomplete,
          assignment.args.push(left, []);
          assignment.incomplete = true;
          assignment.location = {
            min: left.location.min,
            max: ctx.ASSIGN().symbol.stop
          };
        }
        return assignment;
      }

      // 2) simple column reference
      return this.toColumn(qualifiedNameCtx);
    } catch (e) {
      // do nothing
    }
    return undefined;
  }

  // --------------------------------------------------------------------- FUSE

  fromFuseCommand(ctx) {
    const fuseTypeCtx = ctx.identifier();
    const args = [];
    let incomplete = false;
    const fuseType = fuseTypeCtx ? this.fromIdentifier(fuseTypeCtx) : undefined;

    // FUSE <fuse_method>
    if (fuseType) {
      args.push(fuseType);
    }

    // FUSE SCORE BY <score_column> GROUP BY <group_column> KEY BY <key_columns> WITH <options>
    for (const config of ctx.fuseConfiguration_list()) {
      var _configurationItemCom;
      const configurationItemCommandOption = this.fromFuseConfigurationItem(config);
      if (configurationItemCommandOption) {
        args.push(configurationItemCommandOption);
      }
      incomplete ||= (_configurationItemCom = configurationItemCommandOption === null || configurationItemCommandOption === void 0 ? void 0 : configurationItemCommandOption.incomplete) !== null && _configurationItemCom !== void 0 ? _configurationItemCom : true;
    }
    const fuseCommand = this.createCommand('fuse', ctx, {
      args,
      incomplete
    });
    if (fuseType) {
      fuseCommand.fuseType = fuseType;
    }
    return fuseCommand;
  }
  fromFuseConfigurationItem(configCtx) {
    const byContext = configCtx.BY();

    // SCORE BY <score_column>
    const scoreCtx = configCtx.SCORE();
    if (scoreCtx && byContext) {
      const args = [];
      const scoreColumnCtx = configCtx.qualifiedName();
      if (textExistsAndIsValid(scoreColumnCtx.getText())) {
        args.push(this.toColumn(scoreColumnCtx));
      }
      const incomplete = args.length === 0;
      return this.toOption('score by', configCtx, args, incomplete);
    }

    // KEY BY <key_columns>
    const keyCtx = configCtx.KEY();
    if (keyCtx && byContext) {
      const args = this.fromFields(configCtx.fields());
      const incomplete = args.length === 0 || args.some(arg => arg.incomplete);
      return this.toOption('key by', configCtx, args, incomplete);
    }

    // GROUP BY <group_column>
    const groupCtx = configCtx.GROUP();
    if (groupCtx && byContext) {
      const args = [];
      const groupColumnCtx = configCtx.qualifiedName();
      if (textExistsAndIsValid(groupColumnCtx.getText())) {
        args.push(this.toColumn(groupColumnCtx));
      }
      const incomplete = args.length === 0;
      return this.toOption('group by', configCtx, args, incomplete);
    }

    // WITH <map_expression>
    const withCtx = configCtx.WITH();
    if (withCtx) {
      const args = [];
      const mapExpressionCtx = configCtx.mapExpression();
      const map = this.fromMapExpression(mapExpressionCtx);
      if (map && textExistsAndIsValid(map.text)) {
        args.push(map);
      }
      const incomplete = args.length === 0 || map.incomplete || !textExistsAndIsValid(configCtx.getText());
      return this.toOption('with', configCtx, args, incomplete);
    }
    return null;
  }

  // --------------------------------------------------------------------- FORK

  fromForkCommand(ctx) {
    const subQueriesCtx = ctx.forkSubQueries();
    const subQueryCtxs = subQueriesCtx.forkSubQuery_list();
    const args = subQueryCtxs.map(subQueryCtx => this.fromForkSubQuery(subQueryCtx));
    const command = this.createCommand('fork', ctx, {
      args
    });
    return command;
  }
  fromForkSubQuery(ctx) {
    const commands = [];
    const collectCommand = cmdCtx => {
      const processingCommandCtx = cmdCtx.processingCommand();
      const command = this.fromProcessingCommand(processingCommandCtx);
      if (command) {
        commands.push(command);
      }
    };
    let commandCtx = ctx.forkSubQueryCommand();
    while (commandCtx) {
      if (commandCtx instanceof cst.ForkSubQueryProcessingCommandContext) {
        collectCommand(commandCtx);
        break;
      } else if (commandCtx instanceof cst.SingleForkSubQueryCommandContext) {
        collectCommand(commandCtx.forkSubQueryProcessingCommand());
        break;
      } else if (commandCtx instanceof cst.CompositeForkSubQueryContext) {
        collectCommand(commandCtx.forkSubQueryProcessingCommand());
        commandCtx = commandCtx.forkSubQueryCommand();
      }
    }
    commands.reverse();
    const parserFields = this.getParserFields(ctx);
    const query = _builder.Builder.expression.query(commands, parserFields);
    return query;
  }

  // -------------------------------------------------------------- expressions

  toColumnsFromCommand(ctx) {
    const identifiers = this.extractIdentifiers(ctx);
    return this.toColumns(identifiers);
  }
  extractIdentifiers(ctx) {
    if (ctx instanceof cst.MetadataContext) {
      return ctx.UNQUOTED_SOURCE_list().map(node => {
        // TODO: Parse this without wrapping into ParserRuleContext
        return this.terminalNodeToParserRuleContext(node);
      }).flat();
    }
    if (ctx instanceof cst.MvExpandCommandContext) {
      return this.wrapIdentifierAsArray(ctx.qualifiedName());
    }
    return this.wrapIdentifierAsArray(ctx.qualifiedNamePatterns().qualifiedNamePattern_list());
  }
  wrapIdentifierAsArray(identifierCtx) {
    return Array.isArray(identifierCtx) ? identifierCtx : [identifierCtx];
  }

  /**
   * @deprecated
   * @todo Parse without constructing this ANTLR internal class instance.
   */
  terminalNodeToParserRuleContext(node) {
    const context = new antlr.ParserRuleContext();
    context.start = node.symbol;
    context.stop = node.symbol;
    context.children = [node];
    return context;
  }
  toColumns(identifiers) {
    var _identifiers$filter$m;
    const args = (_identifiers$filter$m = identifiers.filter(child => textExistsAndIsValid(child.getText())).map(sourceContext => {
      return this.toColumn(sourceContext);
    })) !== null && _identifiers$filter$m !== void 0 ? _identifiers$filter$m : [];
    return args;
  }
  visitLogicalNot(ctx) {
    const fn = this.toFunction('not', ctx, undefined, 'unary-expression');
    fn.args.push(...this.collectBooleanExpression(ctx.booleanExpression()));
    // update the location of the assign based on arguments
    const argsLocationExtends = this.computeLocationExtends(fn);
    fn.location = argsLocationExtends;
    return fn;
  }
  visitLogicalAndsOrs(ctx) {
    const fn = this.toFunction(ctx.AND() ? 'and' : 'or', ctx, undefined, 'binary-expression');
    fn.args.push(...this.collectBooleanExpression(ctx._left), ...this.collectBooleanExpression(ctx._right));
    // update the location of the assign based on arguments
    const argsLocationExtends = this.computeLocationExtends(fn);
    fn.location = argsLocationExtends;
    return fn;
  }

  /**
   * Constructs a tuple list (round parens):
   *
   * ```
   * (1, 2, 3)
   * ```
   *
   * Can be used in IN-expression:
   *
   * ```
   * WHERE x IN (1, 2, 3)
   * ```
   *
   * @todo Rename this method.
   * @todo Move to "list"
   */
  visitTuple(ctxs, leftParen, rightParen) {
    var _leftParen$symbol, _ctxs$, _rightParen$symbol, _ctxs;
    const values = [];
    let incomplete = false;
    for (const elementCtx of ctxs) {
      const element = this.visitValueExpression(elementCtx);
      if (!element) {
        continue;
      }
      const resolved = (0, _utils.resolveItem)(element);
      if (!resolved) {
        continue;
      }
      values.push(resolved);
      if (resolved.incomplete) {
        incomplete = true;
      }
    }
    if (!values.length) {
      incomplete = true;
    }
    const node = _builder.Builder.expression.list.tuple({
      values
    }, {
      incomplete,
      location: (0, _tokens.getPosition)((_leftParen$symbol = leftParen === null || leftParen === void 0 ? void 0 : leftParen.symbol) !== null && _leftParen$symbol !== void 0 ? _leftParen$symbol : (_ctxs$ = ctxs[0]) === null || _ctxs$ === void 0 ? void 0 : _ctxs$.start, (_rightParen$symbol = rightParen === null || rightParen === void 0 ? void 0 : rightParen.symbol) !== null && _rightParen$symbol !== void 0 ? _rightParen$symbol : (_ctxs = ctxs[ctxs.length - 1]) === null || _ctxs === void 0 ? void 0 : _ctxs.stop)
    });
    return node;
  }
  visitLogicalIns(ctx) {
    var _this$visitValueExpre, _ctx$stop$stop, _ctx$stop;
    const [leftCtx, ...rightCtxs] = ctx.valueExpression_list();
    const left = (0, _utils.resolveItem)((_this$visitValueExpre = this.visitValueExpression(leftCtx)) !== null && _this$visitValueExpre !== void 0 ? _this$visitValueExpre : this.fromParserRuleToUnknown(leftCtx));
    const right = this.visitTuple(rightCtxs, ctx.LP(), ctx.RP());
    const expression = this.toFunction(ctx.NOT() ? 'not in' : 'in', ctx, {
      min: ctx.start.start,
      max: (_ctx$stop$stop = (_ctx$stop = ctx.stop) === null || _ctx$stop === void 0 ? void 0 : _ctx$stop.stop) !== null && _ctx$stop$stop !== void 0 ? _ctx$stop$stop : ctx.RP().symbol.stop
    }, 'binary-expression', [left, right], left.incomplete || right.incomplete);
    return expression;
  }
  getMathOperation(ctx) {
    return (ctx.PLUS() || ctx.MINUS() || ctx.ASTERISK() || ctx.SLASH() || ctx.PERCENT()).getText() || '';
  }
  visitValueExpression(ctx) {
    if (!textExistsAndIsValid(ctx.getText())) {
      return [];
    }
    if (ctx instanceof cst.ValueExpressionDefaultContext) {
      return this.visitOperatorExpression(ctx.operatorExpression());
    }
    if (ctx instanceof cst.ComparisonContext) {
      const operatorCtx = ctx.comparisonOperator();
      const comparisonOperatorText = (operatorCtx.EQ() || operatorCtx.NEQ() || operatorCtx.LT() || operatorCtx.LTE() || operatorCtx.GT() || operatorCtx.GTE()).getText() || '';
      const comparisonFn = this.toFunction(comparisonOperatorText, operatorCtx, undefined, 'binary-expression');
      comparisonFn.args.push(this.visitOperatorExpression(ctx._left), this.visitOperatorExpression(ctx._right));
      // update the location of the comparisonFn based on arguments
      const argsLocationExtends = this.computeLocationExtends(comparisonFn);
      comparisonFn.location = argsLocationExtends;
      return comparisonFn;
    }
  }
  visitOperatorExpression(ctx) {
    if (ctx instanceof cst.ArithmeticUnaryContext) {
      const arg = this.visitOperatorExpression(ctx.operatorExpression());
      // this is a number sign thing
      const fn = this.toFunction('*', ctx, undefined, 'binary-expression');
      fn.args.push(this.createFakeMultiplyLiteral(ctx, 'integer'));
      if (arg) {
        fn.args.push(arg);
      }
      return fn;
    } else if (ctx instanceof cst.ArithmeticBinaryContext) {
      const fn = this.toFunction(this.getMathOperation(ctx), ctx, undefined, 'binary-expression');
      const args = [this.visitOperatorExpression(ctx._left), this.visitOperatorExpression(ctx._right)];
      for (const arg of args) {
        if (arg) {
          fn.args.push(arg);
        }
      }
      // update the location of the assign based on arguments
      const argsLocationExtends = this.computeLocationExtends(fn);
      fn.location = argsLocationExtends;
      return fn;
    } else if (ctx instanceof cst.OperatorExpressionDefaultContext) {
      return this.visitPrimaryExpression(ctx.primaryExpression());
    }
  }
  createFakeMultiplyLiteral(ctx, literalType) {
    return {
      type: 'literal',
      literalType,
      text: ctx.getText(),
      name: ctx.getText(),
      value: ctx.PLUS() ? 1 : -1,
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception)
    };
  }
  getBooleanValue(ctx) {
    const parentNode = ctx instanceof cst.BooleanLiteralContext ? ctx.booleanValue() : ctx;
    const booleanTerminalNode = parentNode.TRUE() || parentNode.FALSE();
    return this.toLiteral('boolean', booleanTerminalNode);
  }
  visitPrimaryExpression(ctx) {
    if (ctx instanceof cst.ConstantDefaultContext) {
      return this.fromConstant(ctx.constant());
    } else if (ctx instanceof cst.DereferenceContext) {
      return this.toColumn(ctx.qualifiedName());
    } else if (ctx instanceof cst.ParenthesizedExpressionContext) {
      return this.collectBooleanExpression(ctx.booleanExpression());
    } else if (ctx instanceof cst.FunctionContext) {
      return this.fromFunction(ctx);
    } else if (ctx instanceof cst.InlineCastContext) {
      return this.collectInlineCast(ctx);
    } else {
      return this.fromParserRuleToUnknown(ctx);
    }
  }
  fromCommandNamedParameters(ctx) {
    const withOption = this.toOption('with', ctx);
    const withCtx = ctx.WITH();
    if (!withCtx) {
      withOption.incomplete = true;
      return withOption;
    }
    const mapExpressionCtx = ctx.mapExpression();
    if (!mapExpressionCtx) {
      withOption.location.min = withCtx.symbol.start;
      withOption.location.max = withCtx.symbol.stop;
      withOption.incomplete = true;
    }
    const map = this.fromMapExpression(mapExpressionCtx);
    withOption.args.push(map);
    withOption.location.min = withCtx.symbol.start;
    withOption.location.max = map.location.max;
    withOption.incomplete = map.incomplete;
    return withOption;
  }
  collectInlineCast(ctx) {
    const value = this.visitPrimaryExpression(ctx.primaryExpression());
    return _builder.Builder.expression.inlineCast({
      castType: ctx.dataType().getText().toLowerCase(),
      value
    }, this.getParserFields(ctx));
  }
  collectLogicalExpression(ctx) {
    if (ctx instanceof cst.LogicalNotContext) {
      return [this.visitLogicalNot(ctx)];
    }
    if (ctx instanceof cst.LogicalBinaryContext) {
      return [this.visitLogicalAndsOrs(ctx)];
    }
    if (ctx instanceof cst.LogicalInContext) {
      return [this.visitLogicalIns(ctx)];
    }
    return [];
  }
  collectRegexExpression(ctx) {
    const regexes = ctx.getTypedRuleContexts(cst.RegexBooleanExpressionContext);
    const ret = [];
    return ret.concat(regexes.map(regex => {
      if (regex instanceof cst.RlikeExpressionContext || regex instanceof cst.LikeExpressionContext) {
        const negate = regex.NOT();
        const likeType = regex instanceof cst.RlikeExpressionContext ? 'rlike' : 'like';
        const fnName = `${negate ? 'not ' : ''}${likeType}`;
        const fn = this.toFunction(fnName, regex, undefined, 'binary-expression');
        const arg = this.visitValueExpression(regex.valueExpression());
        if (arg) {
          fn.args.push(arg);
          const literal = this.toStringLiteral(regex.string_());
          fn.args.push(literal);
        }
        return fn;
      }
      return undefined;
    }).filter(_helpers.nonNullable));
  }
  collectIsNullExpression(ctx) {
    if (!(ctx instanceof cst.IsNullContext)) {
      return [];
    }
    const negate = ctx.NOT();
    const fnName = `is${negate ? ' not ' : ' '}null`;
    const fn = this.toFunction(fnName, ctx, undefined, 'postfix-unary-expression');
    const arg = this.visitValueExpression(ctx.valueExpression());
    if (arg) {
      fn.args.push(arg);
    }
    return [fn];
  }
  collectDefaultExpression(ctx) {
    if (!(ctx instanceof cst.BooleanDefaultContext)) {
      return [];
    }
    const arg = this.visitValueExpression(ctx.valueExpression());
    return arg ? [arg] : [];
  }
  collectBooleanExpression(ctx) {
    const list = [];
    if (!ctx) {
      return list;
    }
    if (ctx instanceof cst.MatchExpressionContext) {
      return [this.visitMatchExpression(ctx)];
    }

    // TODO: Remove these list traversals and concatenations.
    return list.concat(this.collectLogicalExpression(ctx), this.collectRegexExpression(ctx), this.collectIsNullExpression(ctx), this.collectDefaultExpression(ctx)).flat();
  }
  fromBooleanExpressions(ctx) {
    const list = [];
    if (!ctx) {
      return list;
    }
    for (const expr of ctx) {
      list.push(...this.collectBooleanExpression(expr));
    }
    return list;
  }
  fromBooleanExpression(ctx) {
    return this.collectBooleanExpression(ctx)[0] || this.fromParserRuleToUnknown(ctx);
  }
  visitMatchExpression(ctx) {
    return this.visitMatchBooleanExpression(ctx.matchBooleanExpression());
  }
  visitMatchBooleanExpression(ctx) {
    let expression = this.toColumn(ctx.qualifiedName());
    const dataTypeCtx = ctx.dataType();
    const constantCtx = ctx.constant();
    if (dataTypeCtx) {
      expression = _builder.Builder.expression.inlineCast({
        castType: dataTypeCtx.getText().toLowerCase(),
        value: expression
      }, {
        location: (0, _tokens.getPosition)(ctx.start, dataTypeCtx.stop),
        incomplete: Boolean(ctx.exception)
      });
    }
    if (constantCtx) {
      const constantExpression = this.fromConstant(constantCtx);
      expression = this.toBinaryExpression(':', ctx, [expression, constantExpression]);
    }
    return expression;
  }

  // ----------------------------------------------------- expression: "source"

  toSource(ctx, type = 'index') {
    const text = this.sanitizeSourceString(ctx);
    let prefix;
    let index;
    let selector;
    if (ctx instanceof cst.IndexPatternContext) {
      const clusterStringCtx = ctx.clusterString();
      const unquotedIndexString = ctx.unquotedIndexString();
      const indexStringCtx = ctx.indexString();
      const selectorStringCtx = ctx.selectorString();
      if (clusterStringCtx) {
        prefix = this.visitQuotedString(clusterStringCtx);
      }
      if (unquotedIndexString) {
        index = this.visitQuotedString(unquotedIndexString);
      }
      if (indexStringCtx) {
        index = this.visitUnquotedOrQuotedString(indexStringCtx);
      }
      if (selectorStringCtx) {
        selector = this.visitQuotedString(selectorStringCtx);
      }
    }
    return _builder.Builder.expression.source.node({
      sourceType: type,
      prefix,
      index,
      selector,
      name: text
    }, {
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception || text === ''),
      text: ctx === null || ctx === void 0 ? void 0 : ctx.getText()
    });
  }

  // TODO: clean up "source" node helpers.

  createParserFieldsFromTerminalNode(node) {
    const text = node.getText();
    const symbol = node.symbol;
    const fields = {
      text,
      location: (0, _tokens.getPosition)(symbol, symbol),
      incomplete: false
    };
    return fields;
  }
  sanitizeSourceString(ctx) {
    const contextText = ctx.getText();
    // If wrapped by triple quote, remove
    if (contextText.startsWith(`"""`) && contextText.endsWith(`"""`)) {
      return contextText.replace(/\"\"\"/g, '');
    }
    // If wrapped by single quote, remove
    if (contextText.startsWith(`"`) && contextText.endsWith(`"`)) {
      return contextText.slice(1, -1);
    }
    return contextText;
  }
  visitQuotedString(ctx) {
    const unquotedCtx = ctx.UNQUOTED_SOURCE();
    const valueUnquoted = unquotedCtx.getText();
    const quotedString = _pretty_print.LeafPrinter.string({
      valueUnquoted
    });
    return _builder.Builder.expression.literal.string(valueUnquoted, {
      name: quotedString,
      unquoted: true
    }, this.createParserFieldsFromTerminalNode(unquotedCtx));
  }
  visitUnquotedOrQuotedString(ctx) {
    const unquotedCtx = ctx.UNQUOTED_SOURCE();
    if (unquotedCtx) {
      const valueUnquoted = unquotedCtx.getText();
      const quotedString = _pretty_print.LeafPrinter.string({
        valueUnquoted
      });
      return _builder.Builder.expression.literal.string(valueUnquoted, {
        name: quotedString,
        unquoted: true
      }, this.createParserFieldsFromTerminalNode(unquotedCtx));
    }
    return this.toStringLiteral(ctx);
  }

  // ----------------------------------------------------- expression: "column"

  toColumn(ctx) {
    const args = [];
    if (ctx instanceof cst.QualifiedNamePatternContext) {
      const node = this.fromQualifiedNamePattern(ctx);
      if (node.type === 'column') {
        return node;
      } else if (node) {
        args.push(node);
      } else {
        throw new Error(`Unexpected node type: ${node.type} in toColumn`);
      }
    } else if (ctx instanceof cst.QualifiedNameContext) {
      var _ctx$_name;
      // TODO: new grammar also introduced here bracketed syntax.
      // See: https://github.com/elastic/kibana/pull/233585/files#diff-cecb7eac6ebaa167a4c232db56b2912984308749e8b79092c7802230bca7dff5R156-R158
      const fieldNameCtx = (_ctx$_name = ctx._name) !== null && _ctx$_name !== void 0 ? _ctx$_name : ctx.fieldName();
      const list = fieldNameCtx ? fieldNameCtx.identifierOrParameter_list() : [];
      for (const item of list) {
        if (item instanceof cst.IdentifierOrParameterContext) {
          const node = this.fromIdentifierOrParam(item);
          if (node) {
            args.push(node);
          }
        }
      }
    } else {
      // This happens when ANTLR grammar does not specify a rule, for which
      // a context is created. For example, as of this writing, the FROM ... METADATA
      // uses `UNQUOTED_SOURCE` lexer tokens directly for column names, without
      // wrapping them into a context.
      const name = this.sanitizeIdentifierString(ctx);
      const node = _builder.Builder.identifier({
        name
      }, this.getParserFields(ctx));
      args.push(node);
    }
    const text = this.sanitizeIdentifierString(ctx);
    const hasQuotes = Boolean(this.isQuoted(ctx.getText()));
    const column = _builder.Builder.expression.column({
      args
    }, {
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception || text === '')
    });
    column.name = text;
    column.quoted = hasQuotes;
    return column;
  }
  fromQualifiedNamePattern(ctx) {
    var _ctx$_name2;
    const args = [];
    // TODO: new grammar also introduced here bracketed syntax.
    // See: https://github.com/elastic/kibana/pull/233585/files#diff-cecb7eac6ebaa167a4c232db56b2912984308749e8b79092c7802230bca7dff5R165-R167
    const fieldNamePatternCtx = (_ctx$_name2 = ctx._name) !== null && _ctx$_name2 !== void 0 ? _ctx$_name2 : ctx.fieldNamePattern();
    const patterns = fieldNamePatternCtx ? fieldNamePatternCtx.identifierPattern_list() : [];

    // Special case: a single parameter is returned as a param literal
    if (patterns.length === 1) {
      const only = patterns[0];
      if (!only.ID_PATTERN()) {
        var _only$parameter, _only$doubleParameter;
        const paramCtx = ((_only$parameter = only.parameter) === null || _only$parameter === void 0 ? void 0 : _only$parameter.call(only)) || ((_only$doubleParameter = only.doubleParameter) === null || _only$doubleParameter === void 0 ? void 0 : _only$doubleParameter.call(only));
        if (paramCtx) {
          const param = this.toParam(paramCtx);
          if (param) return param;
        }
      }
    }
    for (const identifierPattern of patterns) {
      const ID_PATTERN = identifierPattern.ID_PATTERN();
      if (ID_PATTERN) {
        const node = this.fromNodeToIdentifier(ID_PATTERN);
        args.push(node);
      } else {
        var _identifierPattern$pa, _identifierPattern$do;
        // Support single and double parameters inside identifierPattern
        const paramCtx = ((_identifierPattern$pa = identifierPattern.parameter) === null || _identifierPattern$pa === void 0 ? void 0 : _identifierPattern$pa.call(identifierPattern)) || ((_identifierPattern$do = identifierPattern.doubleParameter) === null || _identifierPattern$do === void 0 ? void 0 : _identifierPattern$do.call(identifierPattern));
        const parameter = paramCtx ? this.toParam(paramCtx) : undefined;
        if (parameter) {
          args.push(parameter);
        }
      }
    }
    const text = this.sanitizeIdentifierString(ctx);
    const hasQuotes = Boolean(this.isQuoted(ctx.getText()));
    const column = _builder.Builder.expression.column({
      args
    }, {
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception || text === '')
    });
    column.name = text;
    column.quoted = hasQuotes;
    return column;
  }
  toColumnStar(ctx) {
    const text = ctx.getText();
    const parserFields = {
      text,
      location: (0, _tokens.getPosition)(ctx.symbol),
      incomplete: ctx.getText() === '',
      quoted: false
    };
    const node = _builder.Builder.expression.column({
      args: [_builder.Builder.identifier({
        name: '*'
      }, parserFields)]
    }, parserFields);
    node.name = text;
    return node;
  }
  isQuoted(text) {
    return text && /^(`)/.test(text);
  }
  fromFields(ctx) {
    const fields = [];
    if (!ctx) {
      return fields;
    }
    try {
      for (const fieldCtx of ctx.field_list()) {
        const field = this.fromField(fieldCtx);
        if (field) {
          fields.push(field);
        }
      }
    } catch (e) {
      // do nothing
    }
    return fields;
  }
  fromAggFields(ctx) {
    const fields = [];
    if (!ctx) {
      return fields;
    }
    try {
      for (const aggField of ctx.aggField_list()) {
        if (aggField.getText() === '') continue;
        const field = this.fromAggField(aggField);
        if (field) {
          fields.push(field);
        }
      }
    } catch (e) {
      // do nothing
    }
    return fields;
  }
  fromField(ctx) {
    // TODO: Just checking on `ctx.qualifiedName()` should be enough and we
    //     should dereference `ctx.qualifiedName()` only once.
    if (ctx.qualifiedName() && ctx.ASSIGN()) {
      // TODO: use binary expression construction method.
      const assignment = this.toFunction(ctx.ASSIGN().getText(), ctx, undefined, 'binary-expression');
      assignment.args.push(this.toColumn(ctx.qualifiedName()),
      // TODO: The second argument here is an array, should be a single item.
      this.collectBooleanExpression(ctx.booleanExpression()));

      // TODO: Avoid using `computeLocationExtends` here. Location should be computed
      //       without that function, and we should eventually remove it.
      assignment.location = this.computeLocationExtends(assignment);
      return assignment;
    }

    // The boolean expression parsing might result into no fields, this
    // happens when ANTLR continues to try to parse an invalid query.
    return this.collectBooleanExpression(ctx.booleanExpression())[0];
  }

  // --------------------------------------------------- expression: "function"

  fromFunction(ctx) {
    const functionExpressionCtx = ctx.functionExpression();
    const functionNameCtx = functionExpressionCtx.functionName();
    const mapExpressionCtx = functionExpressionCtx.mapExpression();
    const args = this.fromBooleanExpressions(functionExpressionCtx.booleanExpression_list());
    const fn = {
      type: 'function',
      subtype: 'variadic-call',
      name: functionNameCtx.getText().toLowerCase(),
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      args,
      incomplete: Boolean(ctx.exception)
    };
    const identifierOrParameterCtx = functionNameCtx.identifierOrParameter();
    const asteriskNode = functionExpressionCtx.ASTERISK() ? this.toColumnStar(functionExpressionCtx.ASTERISK()) : undefined;
    if (identifierOrParameterCtx instanceof cst.IdentifierOrParameterContext) {
      const operator = this.fromIdentifierOrParam(identifierOrParameterCtx);
      if (operator) {
        fn.operator = operator;
      }
    } else {
      const lastCtx = functionNameCtx.LAST();
      if (lastCtx) {
        fn.operator = this.fromNodeToIdentifier(lastCtx);
        fn.operator.name = fn.operator.name.toLowerCase();
      } else {
        const firstCtx = functionNameCtx.FIRST();
        if (firstCtx) {
          fn.operator = this.fromNodeToIdentifier(firstCtx);
          fn.operator.name = fn.operator.name.toLowerCase();
        }
      }
    }
    if (asteriskNode) {
      fn.args.push(asteriskNode);
    }
    if (mapExpressionCtx) {
      const trailingMap = this.fromMapExpression(mapExpressionCtx);
      fn.args.push(trailingMap);
    }
    return fn;
  }
  toFunction(name, ctx, customPosition, subtype, args = [], incomplete) {
    const node = {
      type: 'function',
      name,
      text: ctx.getText(),
      location: customPosition !== null && customPosition !== void 0 ? customPosition : (0, _tokens.getPosition)(ctx.start, ctx.stop),
      args,
      incomplete: Boolean(ctx.exception) || !!incomplete
    };
    if (subtype) {
      node.subtype = subtype;
    }
    return node;
  }
  toBinaryExpression(operator, ctx, args) {
    return _builder.Builder.expression.func.binary(operator, args, {}, {
      text: ctx.getText(),
      location: (0, _tokens.getPosition)(ctx.start, ctx.stop),
      incomplete: Boolean(ctx.exception)
    });
  }

  // -------------------------------------------------------- expression: "map"

  fromMapExpression(ctx) {
    const location = (0, _tokens.getPosition)(ctx.start, ctx.stop);
    const map = _builder.Builder.expression.map({}, {
      text: this.parser.src.slice(location.min, location.max + 1),
      location,
      incomplete: Boolean(ctx.exception)
    });
    const entryCtxs = ctx.entryExpression_list();
    for (const entryCtx of entryCtxs) {
      const entry = this.fromMapEntryExpression(entryCtx);
      if (entry) {
        map.entries.push(entry);
        if (entry.incomplete) {
          map.incomplete = true;
        }
      } else {
        map.incomplete = true;
      }
    }
    return map;
  }
  fromMapEntryExpression(ctx) {
    const keyCtx = ctx._key;
    const valueCtx = ctx._value;
    if (keyCtx && valueCtx) {
      let value;
      const key = this.toStringLiteral(keyCtx);
      if (!key) {
        return undefined;
      }
      const constantCtx = valueCtx.constant();
      if (constantCtx) {
        value = this.fromConstant(constantCtx);
      }
      const mapExpressionCtx = valueCtx.mapExpression();
      if (mapExpressionCtx) {
        value = this.fromMapExpression(mapExpressionCtx);
      }
      if (!value) {
        value = this.fromParserRuleToUnknown(valueCtx);
        value.incomplete = true;
      }
      if (value) {
        const location = (0, _tokens.getPosition)(ctx.start, ctx.stop);
        const entry = _builder.Builder.expression.entry(key, value, {
          text: this.parser.src.slice(location.min, location.max + 1),
          location,
          incomplete: Boolean(ctx.exception) || key.incomplete || value.incomplete
        });
        return entry;
      }
    }
    return undefined;
  }

  // ----------------------------------------------------- constant expressions

  /**
   * @todo Make return type more specific.
   */
  fromConstant(ctx) {
    if (ctx instanceof cst.NullLiteralContext) {
      return this.toLiteral('null', ctx.NULL());
    } else if (ctx instanceof cst.QualifiedIntegerLiteralContext) {
      return this.fromQualifiedIntegerLiteral(ctx);
    } else if (ctx instanceof cst.DecimalLiteralContext) {
      return this.toNumericLiteral(ctx.decimalValue(), 'double');
    } else if (ctx instanceof cst.IntegerLiteralContext) {
      return this.toNumericLiteral(ctx.integerValue(), 'integer');
    } else if (ctx instanceof cst.BooleanLiteralContext) {
      return this.getBooleanValue(ctx);
    } else if (ctx instanceof cst.StringLiteralContext) {
      return this.toStringLiteral(ctx.string_());
    } else if (ctx instanceof cst.NumericArrayLiteralContext) {
      return this.fromNumericArrayLiteral(ctx);
    } else if (ctx instanceof cst.BooleanArrayLiteralContext) {
      return this.fromBooleanArrayLiteral(ctx);
    } else if (ctx instanceof cst.StringArrayLiteralContext) {
      return this.fromStringArrayLiteral(ctx);
    } else if (ctx instanceof cst.InputParameterContext && ctx.children) {
      return this.fromInputParameter(ctx);
    } else {
      return this.fromParserRuleToUnknown(ctx);
    }
  }

  // ------------------------------------------- constant expression: "literal"

  /**
   * @deprecated
   * @todo This method should not exist, the two call sites should be replaced
   *     by a more specific implementation.
   */
  toLiteral(type, node) {
    if (!node) {
      // TODO: This should not return a *broken) literal.
      return {
        type: 'literal',
        name: 'unknown',
        text: 'unknown',
        value: 'unknown',
        literalType: type,
        location: {
          min: 0,
          max: 0
        },
        incomplete: false // TODO: Should be true?
      };
    }
    const text = node.getText();
    const partialLiteral = {
      type: 'literal',
      text,
      name: text,
      location: (0, _tokens.getPosition)(node.symbol),
      incomplete: /<missing /.test(text)
    };
    if (type === 'double' || type === 'integer') {
      return {
        ...partialLiteral,
        literalType: type,
        value: Number(text),
        paramType: 'number'
      };
    } else if (type === 'param') {
      throw new Error('Should never happen');
    }
    return {
      ...partialLiteral,
      literalType: type,
      value: text
    };
  }
  toNumericLiteral(ctx, literalType) {
    return _builder.Builder.expression.literal.numeric({
      value: Number(ctx.getText()),
      literalType
    }, this.getParserFields(ctx));
  }
  fromNumericValue(ctx) {
    const integerCtx = ctx.integerValue();
    if (integerCtx) {
      return this.toNumericLiteral(integerCtx, 'integer');
    }
    return this.toNumericLiteral(ctx.decimalValue(), 'double');
  }
  toStringLiteral(ctx) {
    var _ctx$QUOTED_STRING$ge, _ctx$QUOTED_STRING;
    const quotedString = (_ctx$QUOTED_STRING$ge = (_ctx$QUOTED_STRING = ctx.QUOTED_STRING()) === null || _ctx$QUOTED_STRING === void 0 ? void 0 : _ctx$QUOTED_STRING.getText()) !== null && _ctx$QUOTED_STRING$ge !== void 0 ? _ctx$QUOTED_STRING$ge : '""';
    const isTripleQuoted = quotedString.startsWith('"""') && quotedString.endsWith('"""');
    let valueUnquoted = isTripleQuoted ? quotedString.slice(3, -3) : quotedString.slice(1, -1);
    if (!isTripleQuoted) {
      valueUnquoted = valueUnquoted.replace(/\\"/g, '"').replace(/\\r/g, '\r').replace(/\\n/g, '\n').replace(/\\t/g, '\t').replace(/\\\\/g, '\\');
    }
    return _builder.Builder.expression.literal.string(valueUnquoted, {
      name: quotedString
    }, this.getParserFields(ctx));
  }
  fromInputParameter(ctx) {
    const values = [];
    const children = ctx.children;
    if (children) {
      for (const child of children) {
        const param = this.toParam(child);
        if (param) values.push(param);
      }
    }
    return values;
  }
  toParam(ctx) {
    if (ctx instanceof cst.InputParamContext || ctx instanceof cst.InputDoubleParamsContext) {
      const isDoubleParam = ctx instanceof cst.InputDoubleParamsContext;
      const paramKind = isDoubleParam ? '??' : '?';
      return _builder.Builder.param.unnamed(this.getParserFields(ctx), {
        paramKind
      });
    } else if (ctx instanceof cst.InputNamedOrPositionalParamContext || ctx instanceof cst.InputNamedOrPositionalDoubleParamsContext) {
      const isDoubleParam = ctx instanceof cst.InputNamedOrPositionalDoubleParamsContext;
      const paramKind = isDoubleParam ? '??' : '?';
      const text = ctx.getText();
      const value = text.slice(isDoubleParam ? 2 : 1);
      const valueAsNumber = Number(value);
      const isPositional = String(valueAsNumber) === value;
      const parserFields = this.getParserFields(ctx);
      if (isPositional) {
        return _builder.Builder.param.positional({
          paramKind,
          value: valueAsNumber
        }, parserFields);
      } else {
        return _builder.Builder.param.named({
          paramKind,
          value
        }, parserFields);
      }
    }
  }

  // ---------------------------------------------- constant expression: "list"

  fromNumericArrayLiteral(ctx) {
    const values = ctx.numericValue_list().map(childCtx => this.fromNumericValue(childCtx));
    const parserFields = this.getParserFields(ctx);
    return _builder.Builder.expression.list.literal({
      values
    }, parserFields);
  }
  fromBooleanArrayLiteral(ctx) {
    const values = ctx.booleanValue_list().map(childCtx => this.getBooleanValue(childCtx));
    const parserFields = this.getParserFields(ctx);
    return _builder.Builder.expression.list.literal({
      values
    }, parserFields);
  }
  fromStringArrayLiteral(ctx) {
    const values = ctx.string__list().map(childCtx => this.toStringLiteral(childCtx));
    const parserFields = this.getParserFields(ctx);
    return _builder.Builder.expression.list.literal({
      values
    }, parserFields);
  }

  // -------------------------------------- constant expression: "timeInterval"

  fromQualifiedIntegerLiteral(ctx) {
    const value = ctx.integerValue().INTEGER_LITERAL().getText();
    const unit = ctx.UNQUOTED_IDENTIFIER().symbol.text;
    const parserFields = this.getParserFields(ctx);
    return _builder.Builder.expression.literal.timespan(Number(value), unit, parserFields);
  }

  // ---------------------------------------- constant expression: "identifier"

  fromIdentifierOrParam(ctx) {
    var _ctx$parameter;
    const identifier = ctx.identifier();
    if (identifier) {
      return this.fromIdentifier(identifier);
    }
    const parameter = (_ctx$parameter = ctx.parameter()) !== null && _ctx$parameter !== void 0 ? _ctx$parameter : ctx.doubleParameter();
    if (parameter) {
      return this.toParam(parameter);
    }
  }
  fromIdentifier(ctx) {
    const node = ctx.QUOTED_IDENTIFIER() || ctx.UNQUOTED_IDENTIFIER() || ctx.start;
    return this.fromNodeToIdentifier(node);
  }
  fromNodeToIdentifier(node) {
    let name = node.getText();
    const firstChar = name[0];
    const lastChar = name[name.length - 1];
    const isQuoted = firstChar === '`' && lastChar === '`';
    if (isQuoted) {
      name = name.slice(1, -1).replace(/``/g, '`');
    }
    const parserFields = this.createParserFieldsFromToken(node.symbol);
    const identifier = _builder.Builder.identifier({
      name
    }, parserFields);
    return identifier;
  }
}
exports.CstToAstConverter = CstToAstConverter;