"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFieldsDefinitions = exports.buildColumnSuggestions = void 0;
exports.buildFunctionLookup = buildFunctionLookup;
exports.checkFunctionInvocationComplete = checkFunctionInvocationComplete;
exports.filterFunctionSignatures = exports.filterFunctionDefinitions = void 0;
exports.getAllFunctions = getAllFunctions;
exports.getFunctionDefinition = getFunctionDefinition;
exports.getFunctionSignatures = getFunctionSignatures;
exports.getFunctionSuggestion = getFunctionSuggestion;
exports.printArguments = printArguments;
var _i18n = require("@kbn/i18n");
var _esqlTypes = require("@kbn/esql-types");
var _types = require("../types");
var _all_operators = require("../all_operators");
var _aggregation_functions = require("../generated/aggregation_functions");
var _time_series_agg_functions = require("../generated/time_series_agg_functions");
var _grouping_functions = require("../generated/grouping_functions");
var _scalar_functions = require("../generated/scalar_functions");
var _helpers = require("./autocomplete/helpers");
var _documentation = require("./documentation");
var _shared = require("./shared");
var _test_functions = require("./test_functions");
var _expressions = require("./expressions");
var _is = require("../../ast/is");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const techPreviewLabel = _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.techPreviewLabel', {
  defaultMessage: `Technical Preview`
});
let fnLookups;
function buildFunctionLookup() {
  // we always refresh if we have test functions
  if (!fnLookups || (0, _test_functions.getTestFunctions)().length) {
    fnLookups = _all_operators.operatorsDefinitions.concat(_scalar_functions.scalarFunctionDefinitions, _aggregation_functions.aggFunctionDefinitions, _grouping_functions.groupingFunctionDefinitions, _time_series_agg_functions.timeSeriesAggFunctionDefinitions, (0, _test_functions.getTestFunctions)()).reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        for (const alias of def.alias) {
          memo.set(alias, def);
        }
      }
      return memo;
    }, new Map());
  }
  return fnLookups;
}
const buildFieldsDefinitions = (fields, openSuggestions = true) => {
  return fields.map(label => {
    const suggestion = {
      label,
      text: (0, _helpers.getSafeInsertText)(label),
      kind: 'Variable',
      detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.fieldDefinition', {
        defaultMessage: `Field specified by the input table`
      }),
      sortText: 'D'
    };
    return openSuggestions ? (0, _helpers.withAutoSuggest)(suggestion) : suggestion;
  });
};
exports.buildFieldsDefinitions = buildFieldsDefinitions;
function getFunctionDefinition(name) {
  return buildFunctionLookup().get(name.toLowerCase());
}
const filterFunctionSignatures = (signatures, hasMinimumLicenseRequired) => {
  if (!hasMinimumLicenseRequired) {
    return signatures;
  }
  return signatures.filter(signature => {
    if (!signature.license) return true;
    return hasMinimumLicenseRequired(signature.license.toLocaleLowerCase());
  });
};
exports.filterFunctionSignatures = filterFunctionSignatures;
const filterFunctionDefinitions = (functions, predicates, hasMinimumLicenseRequired, activeProduct) => {
  if (!predicates) {
    return functions;
  }
  const {
    location,
    returnTypes,
    ignored = [],
    allowed = []
  } = predicates;
  return functions.filter(({
    name,
    locationsAvailable,
    ignoreAsSuggestion,
    signatures,
    license,
    observabilityTier
  }) => {
    if (ignoreAsSuggestion) {
      return false;
    }
    if (!!hasMinimumLicenseRequired && license) {
      if (!hasMinimumLicenseRequired(license.toLocaleLowerCase())) {
        return false;
      }
    }
    if (observabilityTier && activeProduct && activeProduct.type === 'observability' && activeProduct.tier !== observabilityTier.toLowerCase()) {
      return false;
    }
    if (allowed.length > 0 && !allowed.includes(name)) {
      return false;
    }
    if (ignored.includes(name)) {
      return false;
    }
    if (location && !locationsAvailable.includes(location)) {
      return false;
    }
    if (returnTypes && !returnTypes.includes('any')) {
      return signatures.some(signature => returnTypes.includes(signature.returnType));
    }
    return true;
  });
};
exports.filterFunctionDefinitions = filterFunctionDefinitions;
function getAllFunctions(options) {
  const {
    type,
    includeOperators = true
  } = options !== null && options !== void 0 ? options : {};
  const fns = buildFunctionLookup();
  const seen = new Set();
  const uniqueFunctions = [];
  for (const fn of fns.values()) {
    if (!seen.has(fn.name)) {
      seen.add(fn.name);
      uniqueFunctions.push(fn);
    }
  }
  let result = uniqueFunctions;
  if (!includeOperators) {
    result = result.filter(fn => fn.type !== _types.FunctionDefinitionTypes.OPERATOR);
  }
  if (!type) {
    return result;
  }
  const types = new Set(Array.isArray(type) ? type : [type]);
  return result.filter(fn => types.has(fn.type));
}
function printArguments({
  name,
  type,
  optional
}, withTypes) {
  if (!withTypes) {
    return name;
  }
  return `${name}${optional ? ':?' : ':'} ${Array.isArray(type) ? type.join(' | ') : type}`;
}
function handleAdditionalArgs(criteria, additionalArgs, withTypes) {
  return criteria ? `${withTypes ? ' ,[... ' : ', '}${additionalArgs.map(arg => printArguments(arg, withTypes)).join(', ')}${withTypes ? ']' : ''}` : '';
}

/**
 * Given a function definition, this function will return a list of function signatures
 *
 * If withTypes is true, the function will return a formal function definition with all arguments typed.
 * This is used when generating the function signature for the monaco editor. If withTypes is false, you get
 * an "injectable" version of the signature to be used to generate test cases.
 */
function getFunctionSignatures({
  name,
  signatures
}, {
  withTypes,
  capitalize
} = {
  withTypes: true,
  capitalize: false
}) {
  return signatures.map(({
    params,
    returnType,
    minParams
  }) => {
    // for functions with a minimum number of args, repeat the last arg multiple times
    // just make sure to compute the right number of args to add
    const minParamsToAdd = Math.max((minParams || 0) - params.length, 0);
    const extraArg = Array(minParamsToAdd || 1).fill(params[Math.max(params.length - 1, 0)]);
    return {
      declaration: `${capitalize ? name.toUpperCase() : name}(${params.map(arg => printArguments(arg, withTypes)).join(', ')}${handleAdditionalArgs(minParamsToAdd > 0, extraArg, withTypes)})${withTypes ? `: ${returnType}` : ''}`
    };
  });
}
function getFunctionSuggestion(fn) {
  let detail = fn.description;
  const labels = [];
  if (fn.preview) {
    labels.push(techPreviewLabel);
  }
  if (fn.license) {
    labels.push(fn.license);
  }
  if (labels.length > 0) {
    detail = `[${labels.join('] [')}] ${detail}`;
  }
  const fullSignatures = getFunctionSignatures(fn, {
    capitalize: true,
    withTypes: true
  });
  let text = `${fn.name.toUpperCase()}($0)`;
  if (fn.customParametersSnippet) {
    text = `${fn.name.toUpperCase()}(${fn.customParametersSnippet})`;
  }
  let functionsPriority = fn.type === _types.FunctionDefinitionTypes.AGG ? 'A' : 'C';
  if (fn.type === _types.FunctionDefinitionTypes.TIME_SERIES_AGG) {
    functionsPriority = '1A';
  }
  return (0, _helpers.withAutoSuggest)({
    label: fn.name.toUpperCase(),
    text,
    asSnippet: true,
    kind: 'Function',
    detail,
    documentation: {
      value: (0, _documentation.buildFunctionDocumentation)(fullSignatures.map((sig, index) => {
        var _fn$signatures$index, _fn$signatures$index2;
        return {
          declaration: sig.declaration,
          license: !!fn.license || !((_fn$signatures$index = fn.signatures[index]) !== null && _fn$signatures$index !== void 0 && _fn$signatures$index.license) ? '' : `[${[(_fn$signatures$index2 = fn.signatures[index]) === null || _fn$signatures$index2 === void 0 ? void 0 : _fn$signatures$index2.license]}]`
        };
      }), fn.examples)
    },
    // time_series_agg functions have priority over everything else
    sortText: functionsPriority
  });
}
function checkFunctionInvocationComplete(func, getExpressionType) {
  const fnDefinition = getFunctionDefinition(func.name);
  if (!fnDefinition) {
    return {
      complete: false
    };
  }
  const cleanedArgs = (0, _shared.removeFinalUnknownIdentiferArg)(func.args, getExpressionType);
  const argLengthCheck = fnDefinition.signatures.some(def => {
    if (def.minParams && cleanedArgs.length >= def.minParams) {
      return true;
    }
    if (cleanedArgs.length === def.params.length) {
      return true;
    }
    return cleanedArgs.length >= def.params.filter(({
      optional
    }) => !optional).length;
  });
  if (!argLengthCheck) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }
  if (func.incomplete && (fnDefinition.name === 'is null' || fnDefinition.name === 'is not null')) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }
  if ((fnDefinition.name === 'in' || fnDefinition.name === 'not in') && Array.isArray(func.args[1]) && !func.args[1].length) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }

  // If the function is complete, check that the types of the arguments match the function definition
  const givenTypes = func.args.map(arg => getExpressionType(arg));
  const literalMask = func.args.map(arg => (0, _is.isLiteral)(Array.isArray(arg) ? arg[0] : arg));
  const hasCorrectTypes = !!(0, _expressions.getMatchingSignatures)(fnDefinition.signatures, givenTypes, literalMask, true).length;
  if (!hasCorrectTypes) {
    return {
      complete: false,
      reason: 'wrongTypes'
    };
  }
  return {
    complete: true
  };
}

/**
 * Generates a sort key for field suggestions based on their categorization.
 * Recommended fields are prioritized, followed by ECS fields.
 *
 * @param isEcs - True if the field is an Elastic Common Schema (ECS) field.
 * @param isRecommended - True if the field is a recommended field from the registry.
 * @returns A string representing the sort key ('1C' for recommended, '1D' for ECS, 'D' for others).
 */
const getFieldsSortText = (isEcs, isRecommended) => {
  if (isRecommended) {
    return '1C';
  }
  if (isEcs) {
    return '1D';
  }
  return 'D';
};
const getVariablePrefix = variableType => variableType === _esqlTypes.ESQLVariableType.FIELDS || variableType === _esqlTypes.ESQLVariableType.FUNCTIONS ? '??' : '?';
const buildColumnSuggestions = (columns, recommendedFieldsFromExtensions = [], options, variables) => {
  const fieldsSuggestions = columns.map(column => {
    const fieldType = column.type.charAt(0).toUpperCase() + column.type.slice(1);
    const titleCaseType = `${column.name} (${fieldType})`;
    // Check if the field is in the recommended fields from extensions list
    // and if so, mark it as recommended. This also ensures that recommended fields
    // that are registered wrongly, won't be shown as suggestions.
    const fieldIsRecommended = recommendedFieldsFromExtensions.some(recommendedField => recommendedField.name === column.name);
    const sortText = getFieldsSortText(!column.userDefined && Boolean(column.isEcs), Boolean(fieldIsRecommended));
    const suggestion = {
      label: column.name,
      text: (0, _helpers.getSafeInsertText)(column.name) + (options !== null && options !== void 0 && options.addComma ? ',' : '') + (options !== null && options !== void 0 && options.advanceCursor ? ' ' : ''),
      kind: 'Variable',
      detail: titleCaseType,
      sortText
    };
    return options !== null && options !== void 0 && options.openSuggestions ? (0, _helpers.withAutoSuggest)(suggestion) : suggestion;
  });
  const suggestions = [...fieldsSuggestions];
  if (options !== null && options !== void 0 && options.supportsControls) {
    var _options$variableType, _variables$filter;
    const variableType = (_options$variableType = options === null || options === void 0 ? void 0 : options.variableType) !== null && _options$variableType !== void 0 ? _options$variableType : _esqlTypes.ESQLVariableType.FIELDS;
    const userDefinedColumns = (_variables$filter = variables === null || variables === void 0 ? void 0 : variables.filter(variable => variable.type === variableType)) !== null && _variables$filter !== void 0 ? _variables$filter : [];
    const controlSuggestions = columns.length ? (0, _helpers.getControlSuggestion)(variableType, userDefinedColumns === null || userDefinedColumns === void 0 ? void 0 : userDefinedColumns.map(v => `${getVariablePrefix(variableType)}${v.key}`)) : [];
    suggestions.push(...controlSuggestions);
  }
  return [...suggestions];
};
exports.buildColumnSuggestions = buildColumnSuggestions;