"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategorizationField = getCategorizationField;
exports.getRecommendedQueriesTemplatesFromExtensions = exports.getRecommendedQueriesTemplates = exports.getRecommendedQueriesSuggestionsFromStaticTemplates = exports.getRecommendedQueriesSuggestions = void 0;
var _i18n = require("@kbn/i18n");
var _metadata = require("../metadata");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Order starts with the simple ones and goes to more complex ones

const getRecommendedQueriesTemplates = ({
  fromCommand,
  timeField,
  categorizationField
}) => {
  const queries = [{
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.aggregateExample.label', {
      defaultMessage: 'Aggregate with STATS'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.aggregateExample.description', {
      defaultMessage: 'Count aggregation'
    }),
    queryString: `${fromCommand}  | STATS count = COUNT(*) /* you can group by a field using the BY operator */`
  }, {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.searchExample.label', {
      defaultMessage: 'Search all fields'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.searchExample.description', {
      defaultMessage: 'Use WHERE to filter/search data'
    }),
    queryString: `${fromCommand}| WHERE QSTR("""term""") /* Search all fields using QSTR – e.g. WHERE QSTR("""debug""") */`,
    sortText: 'D'
  }, ...(timeField ? [{
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.sortByTime.label', {
      defaultMessage: 'Sort by time'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.sortByTime.description', {
      defaultMessage: 'Sort by time'
    }),
    queryString: `${fromCommand}| SORT ${timeField} /* Data is not sorted by default */`
  }, {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.dateIntervals.label', {
      defaultMessage: 'Create 5 minute time buckets with EVAL'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.dateIntervals.description', {
      defaultMessage: 'Count aggregation over time'
    }),
    queryString: `${fromCommand}| EVAL buckets = DATE_TRUNC(5 minute, ${timeField}) | STATS count = COUNT(*) BY buckets /* try out different intervals */`
  }] : []), {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.caseExample.label', {
      defaultMessage: 'Create a conditional with CASE'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.caseExample.description', {
      defaultMessage: 'Conditional'
    }),
    queryString: `${fromCommand}| STATS count = COUNT(*)| EVAL newField = CASE(count < 100, "groupA", count > 100 and count < 500, "groupB", "Other")| KEEP newField`
  }, ...(timeField ? [{
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.dateHistogram.label', {
      defaultMessage: 'Create a date histogram'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.dateHistogram.description', {
      defaultMessage: 'Count aggregation over time'
    }),
    queryString: `${fromCommand}| WHERE ${timeField} <=?_tend and ${timeField} >?_tstart| STATS count = COUNT(*) BY \`Over time\` = BUCKET(${timeField}, 50, ?_tstart, ?_tend) /* ?_tstart and ?_tend take the values of the time picker */`
  }, {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.eventRate.label', {
      defaultMessage: 'Calculate the event rate'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.eventRate.description', {
      defaultMessage: 'Event rate over time'
    }),
    queryString: `${fromCommand}| STATS count = COUNT(*), min_timestamp = MIN(${timeField}) /* MIN(dateField) finds the earliest timestamp in the dataset. */ | EVAL event_rate = count / DATE_DIFF("seconds", min_timestamp, NOW()) /* Calculates the event rate by dividing the total count of events by the time difference (in seconds) between the earliest event and the current time. */ | KEEP event_rate`
  }, {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.categorize.label', {
      // TODO this item should be hidden if AIOps is disabled or we're not running with a platinum license
      // the capability aiops.enabled can be used to check both of these conditions
      defaultMessage: 'Detect change points'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.categorize.description', {
      defaultMessage: 'Change point on count aggregation'
    }),
    queryString: `${fromCommand} | WHERE ${timeField} <=?_tend and ${timeField} >?_tstart | STATS count = COUNT(*) BY buckets = BUCKET(${timeField}, 50, ?_tstart, ?_tend)  | CHANGE_POINT count ON buckets `
  }, {
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.lastHour.label', {
      defaultMessage: 'Total count vs count last hour'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.lastHour.description', {
      defaultMessage: 'A more complicated example'
    }),
    queryString: `${fromCommand} | SORT ${timeField} | EVAL now = NOW() | EVAL key = CASE(${timeField} < (now - 1 hour) AND ${timeField} > (now - 2 hour), "Last hour", "Other") | STATS count = COUNT(*) BY key | EVAL count_last_hour = CASE(key == "Last hour", count), count_rest = CASE(key == "Other", count) | EVAL total_visits = TO_DOUBLE(COALESCE(count_last_hour, 0::LONG) + COALESCE(count_rest, 0::LONG)) | STATS count_last_hour = SUM(count_last_hour), total_visits  = SUM(total_visits)`
  }] : []), ...(categorizationField ?
  // TODO this item should be hidden if AIOps is disabled or we're not running with a platinum license
  // the capability aiops.enabled can be used to check both of these conditions
  [{
    label: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.patternAnalysis.label', {
      defaultMessage: 'Identify patterns'
    }),
    description: _i18n.i18n.translate('kbn-esql-ast.recommendedQueries.patternAnalysis.description', {
      defaultMessage: 'Use the CATEGORIZE function to identify patterns in your logs'
    }),
    queryString: timeField ? `${fromCommand} | WHERE ${timeField} <=?_tend and ${timeField} >?_tstart | SAMPLE .001 | STATS Count=COUNT(*)/.001 BY Pattern=CATEGORIZE(${categorizationField})| SORT Count DESC` : `${fromCommand} | SAMPLE .001 | STATS Count=COUNT(*)/.001 BY Pattern=CATEGORIZE(${categorizationField})| SORT Count DESC`
  }] : [])];

  // prettify the query string
  queries.forEach(query => {
    // the formatted query needs to start with FROM for prettify to work correctly
    const formattedQuery = fromCommand ? (0, _utils.prettifyQuery)(query.queryString) : (0, _utils.prettifyQueryTemplate)(`FROM index ${query.queryString}`);
    query.queryString = formattedQuery;
  });
  return queries;
};
exports.getRecommendedQueriesTemplates = getRecommendedQueriesTemplates;
const getRecommendedQueriesSuggestionsFromStaticTemplates = async (getFieldsByType, fromCommand = '') => {
  const [fieldSuggestions, textFieldSuggestions] = await Promise.all([getFieldsByType(['date'], [], {
    openSuggestions: true
  }),
  // get text fields separately to avoid mixing them with date fields
  getFieldsByType(['text'], [], {
    openSuggestions: true
  })]);
  let timeField = '';
  let categorizationField = '';
  if (fieldSuggestions.length) {
    var _fieldSuggestions$fin;
    timeField = (fieldSuggestions === null || fieldSuggestions === void 0 ? void 0 : (_fieldSuggestions$fin = fieldSuggestions.find(field => field.label === '@timestamp')) === null || _fieldSuggestions$fin === void 0 ? void 0 : _fieldSuggestions$fin.label) || fieldSuggestions[0].label;
  }
  if (textFieldSuggestions.length) {
    categorizationField = getCategorizationField(textFieldSuggestions.map(field => field.label));
  }
  const recommendedQueries = getRecommendedQueriesTemplates({
    fromCommand,
    timeField,
    categorizationField
  });
  const suggestions = recommendedQueries.map(query => {
    var _query$sortText;
    return {
      label: query.label,
      text: query.queryString,
      kind: 'Issue',
      detail: query.description,
      sortText: (_query$sortText = query === null || query === void 0 ? void 0 : query.sortText) !== null && _query$sortText !== void 0 ? _query$sortText : 'E'
    };
  });
  return suggestions;
};

/**
 * This function extracts the templates from the recommended queries extensions.
 * The templates are the recommended queries without the source command (FROM).
 * This is useful for showing the templates in the autocomplete suggestions when the users have already typed the FROM command.
 * @param recommendedQueriesExtensions, the recommended queries extensions to extract the templates from
 * @returns ISuggestionItem[], the templates extracted from the recommended queries extensions
 */
exports.getRecommendedQueriesSuggestionsFromStaticTemplates = getRecommendedQueriesSuggestionsFromStaticTemplates;
const getRecommendedQueriesTemplatesFromExtensions = recommendedQueriesExtensions => {
  if (!recommendedQueriesExtensions || !recommendedQueriesExtensions.length) {
    return [];
  }

  // the templates are the recommended queries without the source command (FROM)
  const recommendedQueriesTemplates = recommendedQueriesExtensions.map(recommendedQuery => {
    var _recommendedQuery$nam;
    const formattedQuery = (0, _utils.prettifyQueryTemplate)(recommendedQuery.query);
    return {
      label: recommendedQuery.name,
      text: formattedQuery,
      detail: (_recommendedQuery$nam = recommendedQuery.name) !== null && _recommendedQuery$nam !== void 0 ? _recommendedQuery$nam : '',
      ...(recommendedQuery.description ? {
        documentation: {
          value: recommendedQuery.description
        }
      } : {}),
      kind: 'Issue',
      sortText: 'D'
    };
  });
  return recommendedQueriesTemplates;
};

// Function returning suggestions from static templates and editor extensions
exports.getRecommendedQueriesTemplatesFromExtensions = getRecommendedQueriesTemplatesFromExtensions;
const getRecommendedQueriesSuggestions = async (editorExtensions, getColumnsByType, prefix = '') => {
  const recommendedQueriesFromExtensions = getRecommendedQueriesTemplatesFromExtensions(editorExtensions.recommendedQueries);

  // If getColumnsByType is not provided, we cannot get the static templates
  // so we return only the extensions. For example in timeseries command the majority of
  // the static templates are not relevant as the count() aggregation is not supported there.
  if (!getColumnsByType) {
    return recommendedQueriesFromExtensions;
  }
  const recommendedQueriesFromTemplates = await getRecommendedQueriesSuggestionsFromStaticTemplates(getColumnsByType, prefix);
  return [...recommendedQueriesFromExtensions, ...recommendedQueriesFromTemplates];
};

/**
 * This function returns the categorization field from the list of fields.
 * It checks for the presence of 'message', 'error.message', or 'event.original' in that order.
 * If none of these fields are present, it returns the first field from the list,
 * Assumes text fields have been passed in the `fields` array.
 *
 * This function is a duplicate of the one in src/platform/packages/shared/kbn-aiops-utils.
 * It is included here to avoid build errors due to bazel
 *
 * TODO: Remove this function once the bazel issue is resolved.
 *
 * @param fields, the list of fields to check
 * @returns string | undefined, the categorization field if found, otherwise undefined
 */
exports.getRecommendedQueriesSuggestions = getRecommendedQueriesSuggestions;
function getCategorizationField(fields) {
  var _filteredFields$;
  const fieldPriority = ['message', 'error.message', 'event.original'];
  const fieldSet = new Set(fields);
  for (const field of fieldPriority) {
    if (fieldSet.has(field)) {
      return field;
    }
  }

  // Filter out metadata fields
  const filteredFields = fields.filter(field => !_metadata.METADATA_FIELDS.includes(field));
  return (_filteredFields$ = filteredFields[0]) !== null && _filteredFields$ !== void 0 ? _filteredFields$ : undefined;
}