"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEntityDefinitionRoute = void 0;
var _entitiesSchema = require("@kbn/entities-schema");
var _zod = require("@kbn/zod");
var _entity_definition_id_invalid = require("../../lib/entities/errors/entity_definition_id_invalid");
var _entity_id_conflict_error = require("../../lib/entities/errors/entity_id_conflict_error");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
var _auth = require("../../lib/auth");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @openapi
 * /internal/entities/definition:
 *   post:
 *     description: Install an entity definition.
 *     tags:
 *       - definitions
 *     parameters:
 *       - in: query
 *         name: installOnly
 *         description: If true, the definition transforms will not be started
 *         required: false
 *         schema:
 *           type: boolean
 *           default: false
 *     requestBody:
 *       description: The entity definition to install
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/entityDefinitionSchema'
 *     responses:
 *       200:
 *         description: Success
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/entityDefinitionSchema'
 *       409:
 *         description: An entity definition with this ID already exists
 *       400:
 *         description: The entity definition cannot be installed; see the error for more details but commonly due to validation failures of the definition ID or metrics format
 */
const createEntityDefinitionRoute = exports.createEntityDefinitionRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'POST /internal/entities/definition',
  security: {
    authz: {
      enabled: false,
      reason: 'This endpoint mainly manages Elasticsearch resources using the requesting users credentials'
    }
  },
  params: _zod.z.object({
    query: _entitiesSchema.createEntityDefinitionQuerySchema,
    body: _entitiesSchema.entityDefinitionSchema
  }),
  handler: async ({
    context,
    request,
    response,
    params,
    logger,
    getScopedClient
  }) => {
    try {
      const currentUserClient = (await context.core).elasticsearch.client.asCurrentUser;
      const isAuthorized = await (0, _auth.canManageEntityDefinition)(currentUserClient, params.body.indexPatterns);
      if (!isAuthorized) {
        return response.forbidden({
          body: {
            message: 'Current Kibana user does not have the required permissions to create the entity definition'
          }
        });
      }
      const client = await getScopedClient({
        request
      });
      const definition = await client.createEntityDefinition({
        definition: params.body,
        installOnly: params.query.installOnly
      });
      return response.ok({
        body: definition
      });
    } catch (e) {
      logger.error(e);
      if (e instanceof _entity_definition_id_invalid.EntityDefinitionIdInvalid) {
        return response.badRequest({
          body: e
        });
      }
      if (e instanceof _entity_id_conflict_error.EntityIdConflict) {
        return response.conflict({
          body: e
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return response.customError({
          body: e,
          statusCode: 400
        });
      }
      return response.customError({
        body: e,
        statusCode: 500
      });
    }
  }
});