"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useESQLVariables = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _controlsConstants = require("@kbn/controls-constants");
var _rxjs = require("rxjs");
var _redux = require("../../state_management/redux");
var _discover_state_provider = require("../../state_management/discover_state_provider");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Custom hook to manage ESQL variables in the control group for Discover.
 * It synchronizes ESQL control panel state with the application's internal Redux state
 * and handles persistence to storage.
 *
 * @param options - Configuration options for the hook.
 * @param options.isEsqlMode - Indicates if the current application mode is ESQL.
 * @param options.controlGroupApi - The ControlGroupRendererApi instance for interacting with control panels.
 * @param options.currentEsqlVariables - The currently active ESQL variables from the application state.
 * @param options.stateContainer - The DiscoverStateContainer instance for data fetching.
 * @param options.onUpdateESQLQuery - Callback function to update the ESQL query.
 *
 * @returns An object containing handler functions for saving and canceling control changes,
 * and a getter function to retrieve the currently active control panels state for the current tab.
 */

const useESQLVariables = ({
  isEsqlMode,
  controlGroupApi,
  currentEsqlVariables,
  stateContainer,
  onUpdateESQLQuery
}) => {
  const dispatch = (0, _redux.useInternalStateDispatch)();
  const setControlGroupState = (0, _redux.useCurrentTabAction)(_redux.internalStateActions.setControlGroupState);
  const setEsqlVariables = (0, _redux.useCurrentTabAction)(_redux.internalStateActions.setEsqlVariables);
  const currentControlGroupState = (0, _redux.useCurrentTabSelector)(tab => tab.controlGroupState);
  const savedSearchState = (0, _discover_state_provider.useSavedSearch)();
  (0, _react.useEffect)(() => {
    // Only proceed if in ESQL mode and controlGroupApi is available
    if (!controlGroupApi || !isEsqlMode) {
      return;
    }

    // Handling the reset unsaved changes badge
    const savedSearchResetSubsciption = stateContainer.savedSearchState.getInitial$().pipe((0, _rxjs.skip)(1)) // Skip the initial emission since it's a BehaviorSubject
    .subscribe(initialSavedSearch => {
      const savedControlGroupState = (0, _redux.parseControlGroupJson)(initialSavedSearch === null || initialSavedSearch === void 0 ? void 0 : initialSavedSearch.controlGroupJson);
      controlGroupApi.updateInput({
        initialChildControlState: savedControlGroupState
      });
    });
    const inputSubscription = controlGroupApi.getInput$().subscribe(input => {
      if (input && input.initialChildControlState) {
        const currentTabControlState = input.initialChildControlState;
        stateContainer.savedSearchState.updateControlState({
          nextControlState: currentTabControlState
        });
        dispatch(setControlGroupState({
          controlGroupState: currentTabControlState
        }));
        const newVariables = (0, _redux.extractEsqlVariables)(currentTabControlState);
        if (!(0, _lodash.isEqual)(newVariables, currentEsqlVariables)) {
          // Update the ESQL variables in the internal state
          dispatch(setEsqlVariables({
            esqlVariables: newVariables
          }));
          stateContainer.dataState.fetch();
        }
      }
    });
    return () => {
      inputSubscription.unsubscribe();
      savedSearchResetSubsciption.unsubscribe();
    };
  }, [controlGroupApi, currentEsqlVariables, dispatch, isEsqlMode, setControlGroupState, setEsqlVariables, stateContainer.dataState, stateContainer.savedSearchState]);
  const onSaveControl = (0, _react.useCallback)(async (controlState, updatedQuery) => {
    if (!controlGroupApi) {
      // eslint-disable-next-line no-console
      console.error('controlGroupApi is not available when attempting to save control.');
      return;
    }
    // add a new control
    controlGroupApi.addNewPanel({
      panelType: _controlsConstants.ESQL_CONTROL,
      serializedState: {
        rawState: {
          ...controlState
        }
      }
    });

    // update the query
    if (updatedQuery) {
      onUpdateESQLQuery(updatedQuery);
    }
  }, [controlGroupApi, onUpdateESQLQuery]);

  // Getter function to retrieve the currently active control panels state for the current tab
  const getActivePanels = (0, _react.useCallback)(() => {
    if (currentControlGroupState && Object.keys(currentControlGroupState).length > 0) {
      return currentControlGroupState;
    }
    return (0, _redux.parseControlGroupJson)(savedSearchState === null || savedSearchState === void 0 ? void 0 : savedSearchState.controlGroupJson);
  }, [currentControlGroupState, savedSearchState === null || savedSearchState === void 0 ? void 0 : savedSearchState.controlGroupJson]);
  return {
    onSaveControl,
    getActivePanels
  };
};
exports.useESQLVariables = useESQLVariables;