"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initialize = initialize;
var _nodeAssert = _interopRequireDefault(require("node:assert"));
var _elasticsearch = require("@elastic/elasticsearch");
var _lodash = require("lodash");
var _retry_es = require("./retry_es");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function applyDefaults(def) {
  return (0, _lodash.defaultsDeep)(def, (0, _constants.defaultDataStreamDefinition)());
}

/**
 * https://www.elastic.co/docs/manage-data/data-store/data-streams/set-up-data-stream
 *
 * Endeavour to be idempotent and race-condition safe.
 */
async function initialize({
  logger,
  dataStreams,
  elasticsearchClient
}) {
  logger = logger.get('data-streams-setup');
  logger.debug(`Setting up index template for data stream: ${dataStreams.name}`);
  let existingIndexTemplate;
  try {
    ({
      index_templates: [existingIndexTemplate]
    } = await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.getIndexTemplate({
      name: dataStreams.name
    })));
  } catch (error) {
    if (error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 404) {
      // Index template does not exist, we will create it
    } else {
      throw error;
    }
  }
  const version = dataStreams.version;
  const previousVersions = [];
  dataStreams = applyDefaults(dataStreams);
  if (existingIndexTemplate) {
    var _existingIndexTemplat, _existingIndexTemplat2, _existingIndexTemplat3, _existingIndexTemplat4;
    const deployedVersion = (_existingIndexTemplat = existingIndexTemplate.index_template) === null || _existingIndexTemplat === void 0 ? void 0 : (_existingIndexTemplat2 = _existingIndexTemplat._meta) === null || _existingIndexTemplat2 === void 0 ? void 0 : _existingIndexTemplat2.version;
    (0, _nodeAssert.default)(typeof deployedVersion === 'number' && deployedVersion > 0, `Datastream metadata is in an unexpected state, expected version to be a number but got ${deployedVersion}`);
    if (deployedVersion >= version) {
      return; // already applied our mappings etc.
    }
    previousVersions.push(deployedVersion, ...((_existingIndexTemplat3 = existingIndexTemplate.index_template) === null || _existingIndexTemplat3 === void 0 ? void 0 : (_existingIndexTemplat4 = _existingIndexTemplat3._meta) === null || _existingIndexTemplat4 === void 0 ? void 0 : _existingIndexTemplat4.previousVersions));
  }

  // Should be idempotent
  await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.putIndexTemplate({
    name: dataStreams.name,
    priority: dataStreams.template.priority,
    index_patterns: [`${dataStreams.name}*`],
    composed_of: dataStreams.template.composedOf,
    data_stream: {
      hidden: dataStreams.hidden
    },
    template: {
      aliases: dataStreams.template.aliases,
      mappings: dataStreams.template.mappings,
      settings: dataStreams.template.settings
    },
    _meta: {
      ...dataStreams.template._meta,
      version,
      previousVersions
    }
  }));
  let existingDataStream;
  try {
    ({
      data_streams: [existingDataStream]
    } = await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.getDataStream({
      name: dataStreams.name
    })));
  } catch (error) {
    if (error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 404) {
      // Data stream does not exist, we will create it
    } else {
      throw error;
    }
  }
  if (!existingDataStream) {
    logger.debug(`Creating data stream: ${dataStreams.name}`);
    try {
      await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.createDataStream({
        name: dataStreams.name
      }));
    } catch (error) {
      var _error$body;
      if (error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 400 && ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.error.type) === 'resource_already_exists_exception') {
        // Data stream already exists, we can ignore this error, probably racing another create call
        logger.debug(`Data stream already exists: ${dataStreams.name}`);
      } else {
        throw error;
      }
    }
  } else {
    logger.debug(`Data stream already exists: ${dataStreams.name}, applying mappings to write index`);

    // https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-indices-get-data-stream#operation-indices-get-data-stream-200-body-application-json-data_streams-indices
    // The last item in this array contains information about the stream’s current write index.
    const {
      indices
    } = existingDataStream;
    const writeIndex = indices[indices.length - 1];
    if (!writeIndex) {
      logger.debug(`Data stream ${dataStreams.name} has no write index yet, cannot apply mappings or settings.`);
      return;
    } else {
      const {
        template: {
          mappings
        }
      } = await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.simulateIndexTemplate({
        name: dataStreams.name
      }));
      logger.debug(`Applying mappings to write index: ${writeIndex}`);
      await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.putMapping({
        index: writeIndex.index_name,
        ...mappings
      }));
    }
  }
}