"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applyDeprecations = void 0;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _unset_and_clean_empty_parent = require("./unset_and_clean_empty_parent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const noopAddDeprecationFactory = () => () => undefined;

/**
 * Applies deprecations on given configuration and passes addDeprecation hook.
 * This hook is used for logging any deprecation warning using provided logger.
 * This hook is used for exposing deprecated configs that must be handled by the user before upgrading to next major.
 *
 * @internal
 */
const applyDeprecations = (config, deprecations, createAddDeprecation = noopAddDeprecationFactory) => {
  const result = (0, _lodash.cloneDeep)(config);
  const changedPaths = {
    set: [],
    unset: []
  };
  deprecations.forEach(({
    deprecation,
    path,
    context
  }) => {
    const commands = deprecation(result, path, createAddDeprecation(path), context);
    if (commands) {
      if (commands.set) {
        changedPaths.set.push(...commands.set.map(c => c.path));
        commands.set.forEach(function ({
          path: commandPath,
          value
        }) {
          (0, _saferLodashSet.set)(result, commandPath, value);
        });
      }
      if (commands.unset) {
        changedPaths.unset.push(...commands.unset.map(c => c.path));
        commands.unset.forEach(function ({
          path: commandPath
        }) {
          (0, _unset_and_clean_empty_parent.unsetAndCleanEmptyParent)(result, commandPath);
        });
      }
    }
  });
  return {
    config: result,
    changedPaths
  };
};
exports.applyDeprecations = applyDeprecations;